import {
  Controller,
  Get,
  Post,
  Put,
  Delete,
  Body,
  Param,
  Query,
  UseGuards,
  ParseIntPipe,
  DefaultValuePipe,
  ParseBoolPipe,
} from '@nestjs/common';
import {
  ZoneService,
  ZoneType,
  CreateZoneDto,
  UpdateZoneDto,
  Coordinate,
} from './zone.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantId } from '../../common/decorators/merchant.decorator';

// ============================================================================
// DTOs
// ============================================================================

class CheckPointDto {
  lat: number;
  lng: number;
  vehicleTypeId?: number;
}

class CheckRouteDto {
  pickup: Coordinate;
  dropoff: Coordinate;
  vehicleTypeId?: number;
}

// ============================================================================
// CONTROLLER
// ============================================================================

@Controller('zones')
@UseGuards(JwtAuthGuard)
export class ZoneController {
  constructor(private readonly zoneService: ZoneService) {}

  // ==========================================================================
  // ZONE MANAGEMENT
  // ==========================================================================

  /**
   * Lister toutes les zones
   */
  @Get()
  async listZones(
    @MerchantId() merchantId: number,
    @Query('type') type?: ZoneType,
    @Query('active') active?: string,
    @Query('search') search?: string,
  ) {
    return this.zoneService.listZones(merchantId, {
      type,
      active: active !== undefined ? active === 'true' : undefined,
      search,
    });
  }

  /**
   * Obtenir les zones actives pour l'affichage sur carte
   */
  @Get('map')
  async getMapZones(
    @MerchantId() merchantId: number,
    @Query('types') types?: string,
  ) {
    const typeFilter = types?.split(',') as ZoneType[] | undefined;
    const zones = await this.zoneService.listZones(merchantId, { active: true });

    let filteredZones = zones;
    if (typeFilter && typeFilter.length > 0) {
      filteredZones = zones.filter((z) => typeFilter.includes(z.type));
    }

    // Format simplifié pour la carte
    return filteredZones.map((zone) => ({
      id: zone.id,
      name: zone.name,
      type: zone.type,
      polygon: zone.polygon,
      center: zone.center,
      radius: zone.radius,
      color: zone.color,
    }));
  }

  /**
   * Obtenir une zone
   */
  @Get(':id')
  async getZone(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const zone = await this.zoneService.getZone(id, merchantId);
    if (!zone) {
      return { error: 'Zone non trouvée' };
    }
    return zone;
  }

  /**
   * Créer une zone
   */
  @Post()
  async createZone(
    @MerchantId() merchantId: number,
    @Body() dto: CreateZoneDto,
  ) {
    const zone = await this.zoneService.createZone(merchantId, dto);
    return {
      success: true,
      zone,
    };
  }

  /**
   * Mettre à jour une zone
   */
  @Put(':id')
  async updateZone(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: UpdateZoneDto,
  ) {
    const zone = await this.zoneService.updateZone(id, merchantId, dto);
    if (!zone) {
      return { error: 'Zone non trouvée' };
    }
    return {
      success: true,
      zone,
    };
  }

  /**
   * Supprimer une zone
   */
  @Delete(':id')
  async deleteZone(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const deleted = await this.zoneService.deleteZone(id, merchantId);
    return {
      success: deleted,
      message: deleted ? 'Zone supprimée' : 'Zone non trouvée',
    };
  }

  /**
   * Activer/désactiver une zone
   */
  @Post(':id/toggle')
  async toggleZone(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const zone = await this.zoneService.getZone(id, merchantId);
    if (!zone) {
      return { error: 'Zone non trouvée' };
    }

    const updated = await this.zoneService.updateZone(id, merchantId, {
      active: !zone.active,
    });

    return {
      success: true,
      zone: updated,
      message: updated?.active ? 'Zone activée' : 'Zone désactivée',
    };
  }

  // ==========================================================================
  // GEOFENCING
  // ==========================================================================

  /**
   * Vérifier un point
   */
  @Post('check-point')
  async checkPoint(
    @MerchantId() merchantId: number,
    @Body() dto: CheckPointDto,
  ) {
    return this.zoneService.checkPoint(
      { lat: dto.lat, lng: dto.lng },
      merchantId,
      dto.vehicleTypeId,
    );
  }

  /**
   * Vérifier un trajet (pickup + dropoff)
   */
  @Post('check-route')
  async checkRoute(
    @MerchantId() merchantId: number,
    @Body() dto: CheckRouteDto,
  ) {
    return this.zoneService.checkRoute(
      dto.pickup,
      dto.dropoff,
      merchantId,
      dto.vehicleTypeId,
    );
  }

  /**
   * Trouver les zones contenant un point
   */
  @Get('containing')
  async findZonesContainingPoint(
    @MerchantId() merchantId: number,
    @Query('lat') lat: string,
    @Query('lng') lng: string,
  ) {
    if (!lat || !lng) {
      return { error: 'Coordonnées lat et lng requises' };
    }

    const zones = await this.zoneService.findZonesContainingPoint(
      { lat: parseFloat(lat), lng: parseFloat(lng) },
      merchantId,
    );

    return zones;
  }

  // ==========================================================================
  // ZONE TYPES
  // ==========================================================================

  /**
   * Obtenir les types de zones disponibles
   */
  @Get('meta/types')
  getZoneTypes() {
    return {
      types: [
        {
          type: 'service_area',
          name: 'Zone de service',
          description: 'Zone où les courses sont possibles',
          color: '#4CAF50',
        },
        {
          type: 'surge_zone',
          name: 'Zone de surge',
          description: 'Zone avec tarification dynamique',
          color: '#FF9800',
        },
        {
          type: 'restricted',
          name: 'Zone interdite',
          description: 'Zone où les courses sont interdites',
          color: '#F44336',
        },
        {
          type: 'airport',
          name: 'Aéroport',
          description: 'Zone aéroportuaire avec tarif spécial',
          color: '#2196F3',
        },
        {
          type: 'event',
          name: 'Événement',
          description: 'Zone événementielle temporaire',
          color: '#9C27B0',
        },
        {
          type: 'delivery_only',
          name: 'Livraison uniquement',
          description: 'Zone réservée aux livraisons',
          color: '#00BCD4',
        },
        {
          type: 'pickup_only',
          name: 'Prise en charge uniquement',
          description: 'Zone de prise en charge uniquement',
          color: '#8BC34A',
        },
        {
          type: 'dropoff_only',
          name: 'Dépose uniquement',
          description: 'Zone de dépose uniquement',
          color: '#CDDC39',
        },
      ],
    };
  }

  // ==========================================================================
  // STATISTICS
  // ==========================================================================

  /**
   * Statistiques des zones
   */
  @Get('stats/overview')
  async getStatistics(@MerchantId() merchantId: number) {
    return this.zoneService.getStatistics(merchantId);
  }

  // ==========================================================================
  // IMPORT/EXPORT
  // ==========================================================================

  /**
   * Exporter les zones (GeoJSON)
   */
  @Get('export/geojson')
  async exportGeoJson(@MerchantId() merchantId: number) {
    const zones = await this.zoneService.listZones(merchantId);

    const features = zones.map((zone) => {
      // Zone circulaire -> Point avec radius
      if (zone.radius) {
        return {
          type: 'Feature',
          properties: {
            id: zone.id,
            name: zone.name,
            type: zone.type,
            radius: zone.radius,
            color: zone.color,
            surgeMultiplier: zone.surgeMultiplier,
            extraFee: zone.extraFee,
            active: zone.active,
          },
          geometry: {
            type: 'Point',
            coordinates: [zone.center.lng, zone.center.lat],
          },
        };
      }

      // Zone polygonale
      return {
        type: 'Feature',
        properties: {
          id: zone.id,
          name: zone.name,
          type: zone.type,
          color: zone.color,
          surgeMultiplier: zone.surgeMultiplier,
          extraFee: zone.extraFee,
          active: zone.active,
        },
        geometry: {
          type: 'Polygon',
          coordinates: [
            [...zone.polygon.map((p) => [p.lng, p.lat]), [zone.polygon[0].lng, zone.polygon[0].lat]],
          ],
        },
      };
    });

    return {
      type: 'FeatureCollection',
      features,
    };
  }

  /**
   * Importer des zones (GeoJSON)
   */
  @Post('import/geojson')
  async importGeoJson(
    @MerchantId() merchantId: number,
    @Body() body: { geojson: any; overwrite?: boolean },
  ) {
    const { geojson, overwrite } = body;
    const results = { imported: 0, errors: [] as string[] };

    if (!geojson?.features || !Array.isArray(geojson.features)) {
      return { error: 'Format GeoJSON invalide' };
    }

    for (const feature of geojson.features) {
      try {
        const props = feature.properties || {};
        const geometry = feature.geometry;

        let createDto: CreateZoneDto;

        if (geometry.type === 'Point') {
          // Zone circulaire
          createDto = {
            name: props.name || `Zone importée ${results.imported + 1}`,
            slug: props.slug || `zone-import-${Date.now()}-${results.imported}`,
            type: props.type || 'service_area',
            description: props.description,
            center: {
              lat: geometry.coordinates[1],
              lng: geometry.coordinates[0],
            },
            radius: props.radius || 1000,
            color: props.color,
            surgeMultiplier: props.surgeMultiplier,
            extraFee: props.extraFee,
          };
        } else if (geometry.type === 'Polygon') {
          // Zone polygonale
          const coordinates = geometry.coordinates[0];
          const polygon = coordinates.slice(0, -1).map((coord: number[]) => ({
            lat: coord[1],
            lng: coord[0],
          }));

          createDto = {
            name: props.name || `Zone importée ${results.imported + 1}`,
            slug: props.slug || `zone-import-${Date.now()}-${results.imported}`,
            type: props.type || 'service_area',
            description: props.description,
            polygon,
            color: props.color,
            surgeMultiplier: props.surgeMultiplier,
            extraFee: props.extraFee,
          };
        } else {
          results.errors.push(`Type de géométrie non supporté: ${geometry.type}`);
          continue;
        }

        // Vérifier si existe déjà (par slug)
        if (overwrite) {
          const existing = await this.zoneService.getZoneBySlug(
            createDto.slug,
            merchantId,
          );
          if (existing) {
            await this.zoneService.updateZone(existing.id, merchantId, createDto);
            results.imported++;
            continue;
          }
        }

        await this.zoneService.createZone(merchantId, createDto);
        results.imported++;
      } catch (error: any) {
        results.errors.push(error.message);
      }
    }

    return {
      success: true,
      imported: results.imported,
      errors: results.errors,
    };
  }
}
