import { Injectable, Logger } from '@nestjs/common';
import { Server } from 'socket.io';
import {
  BOOKING_STATUS,
  getStatusLabel,
} from '../booking/booking.constants';

interface Connection {
  socketId: string;
  connectedAt: Date;
  lastActivity?: Date;
}

interface DriverLocation {
  latitude: number;
  longitude: number;
  heading?: number;
  speed?: number;
  updatedAt: Date;
}

@Injectable()
export class WebsocketService {
  private readonly logger = new Logger(WebsocketService.name);
  private server: Server;

  // Track connected users and drivers
  private userConnections: Map<number, Connection> = new Map();
  private driverConnections: Map<number, Connection> = new Map();

  // Track driver locations in memory (for fast access)
  private driverLocations: Map<number, DriverLocation> = new Map();

  setServer(server: Server) {
    this.server = server;
  }

  addConnection(userId: number, userType: 'user' | 'driver', socketId: string) {
    const connection: Connection = {
      socketId,
      connectedAt: new Date(),
      lastActivity: new Date(),
    };

    if (userType === 'user') {
      this.userConnections.set(userId, connection);
    } else {
      this.driverConnections.set(userId, connection);
    }

    this.logger.log(`${userType} #${userId} connected (socket: ${socketId})`);
  }

  removeConnection(userId: number, userType: 'user' | 'driver') {
    if (userType === 'user') {
      this.userConnections.delete(userId);
    } else {
      this.driverConnections.delete(userId);
      // Also clear driver location
      this.driverLocations.delete(userId);
    }

    this.logger.log(`${userType} #${userId} disconnected`);
  }

  isConnected(userId: number, userType: 'user' | 'driver'): boolean {
    if (userType === 'user') {
      return this.userConnections.has(userId);
    }
    return this.driverConnections.has(userId);
  }

  getConnectionStats() {
    return {
      users: this.userConnections.size,
      drivers: this.driverConnections.size,
      total: this.userConnections.size + this.driverConnections.size,
      driversWithLocation: this.driverLocations.size,
    };
  }

  // ============================================================================
  // DRIVER LOCATION MANAGEMENT
  // ============================================================================

  updateDriverLocation(driverId: number, location: Omit<DriverLocation, 'updatedAt'>) {
    this.driverLocations.set(driverId, {
      ...location,
      updatedAt: new Date(),
    });
  }

  getDriverLocation(driverId: number): DriverLocation | undefined {
    return this.driverLocations.get(driverId);
  }

  getOnlineDriversInArea(
    merchantId: number,
    latitude: number,
    longitude: number,
    radiusKm: number,
  ): Array<{ driverId: number; location: DriverLocation; distance: number }> {
    const results: Array<{ driverId: number; location: DriverLocation; distance: number }> = [];

    this.driverLocations.forEach((location, driverId) => {
      const distance = this.calculateDistance(
        latitude,
        longitude,
        location.latitude,
        location.longitude,
      );

      if (distance <= radiusKm) {
        results.push({ driverId, location, distance });
      }
    });

    // Sort by distance
    return results.sort((a, b) => a.distance - b.distance);
  }

  // ============================================================================
  // BROADCAST METHODS (called from other services)
  // ============================================================================

  /**
   * Send event to a specific user
   */
  sendToUser(userId: number, event: string, data: any) {
    if (!this.server) return;
    this.server.to(`user.${userId}`).emit(event, data);
  }

  /**
   * Send event to a specific driver
   */
  sendToDriver(driverId: number, event: string, data: any) {
    if (!this.server) return;
    this.server.to(`driver.${driverId}`).emit(event, data);
  }

  /**
   * Send event to all participants of a booking
   */
  sendToBooking(bookingId: number, event: string, data: any) {
    if (!this.server) return;
    this.server.to(`booking.${bookingId}`).emit(event, data);
  }

  /**
   * Send event to all users/drivers of a merchant
   */
  sendToMerchant(merchantId: number, event: string, data: any) {
    if (!this.server) return;
    this.server.to(`merchant.${merchantId}`).emit(event, data);
  }

  // ============================================================================
  // BOOKING EVENTS
  // ============================================================================

  /**
   * Notify about new booking request (to driver)
   */
  notifyNewBookingRequest(
    driverId: number,
    booking: any,
    expiresInSeconds: number = 30,
  ) {
    this.sendToDriver(driverId, 'booking:new_request', {
      bookingId: booking.id,
      merchantBookingId: booking.merchant_booking_id,
      pickup: {
        address: booking.pickup_address || booking.pickup_location,
        latitude: parseFloat(booking.pickup_latitude),
        longitude: parseFloat(booking.pickup_longitude),
      },
      drop: {
        address: booking.drop_address || booking.drop_location,
        latitude: parseFloat(booking.drop_latitude),
        longitude: parseFloat(booking.drop_longitude),
      },
      estimatedAmount: booking.estimate_amount,
      estimatedDistance: booking.travel_distance,
      estimatedTime: booking.travel_time,
      vehicleType: booking.vehicle_type_id,
      paymentMethod: booking.payment_method_id,
      user: booking.user ? {
        id: booking.user.id,
        name: `${booking.user.first_name || ''} ${booking.user.last_name || ''}`.trim(),
        phone: booking.user.UserPhone,
        rating: booking.user.rating,
        photo: booking.user.profile_image || booking.user.UserProfileImage,
      } : null,
      expiresAt: new Date(Date.now() + expiresInSeconds * 1000).toISOString(),
      timestamp: new Date().toISOString(),
    });
  }

  /**
   * Notify booking status change with Laravel-compatible status codes
   */
  notifyBookingStatusChanged(
    booking: any,
    previousStatus: number,
    metadata?: any,
  ) {
    const event = 'booking:status_changed';
    const data = {
      bookingId: booking.id,
      merchantBookingId: booking.merchant_booking_id,
      status: booking.booking_status,
      statusCode: booking.booking_status,
      statusLabel: getStatusLabel(booking.booking_status),
      previousStatus,
      previousStatusLabel: getStatusLabel(previousStatus),
      driver: booking.driver ? {
        id: booking.driver.id,
        name: `${booking.driver.first_name || ''} ${booking.driver.last_name || ''}`.trim(),
        phone: booking.driver.phoneNumber,
        photo: booking.driver.profile_image,
        location: booking.driver.latitude && booking.driver.longitude ? {
          latitude: parseFloat(booking.driver.latitude),
          longitude: parseFloat(booking.driver.longitude),
        } : null,
      } : null,
      metadata,
      timestamp: new Date().toISOString(),
    };

    // Notify user
    if (booking.user_id) {
      this.sendToUser(booking.user_id, event, data);
    }

    // Notify driver
    if (booking.driver_id) {
      this.sendToDriver(booking.driver_id, event, data);
    }

    // Notify booking room
    this.sendToBooking(booking.id, event, data);

    this.logger.log(
      `Booking ${booking.id} status changed: ${getStatusLabel(previousStatus)} -> ${getStatusLabel(booking.booking_status)}`,
    );
  }

  /**
   * Broadcast driver location update
   */
  broadcastDriverLocation(
    bookingId: number,
    driverId: number,
    location: {
      latitude: number;
      longitude: number;
      heading?: number;
      speed?: number;
    },
  ) {
    // Update in-memory location
    this.updateDriverLocation(driverId, location);

    // Broadcast to booking room
    this.sendToBooking(bookingId, 'driver:location:updated', {
      driverId,
      ...location,
      timestamp: new Date().toISOString(),
    });
  }

  /**
   * Notify booking accepted with full driver and vehicle info
   */
  notifyBookingAccepted(userId: number, booking: any, driver: any, vehicle?: any) {
    this.sendToUser(userId, 'booking:accepted', {
      bookingId: booking.id,
      merchantBookingId: booking.merchant_booking_id,
      status: BOOKING_STATUS.DRIVER_ACCEPTED,
      statusLabel: getStatusLabel(BOOKING_STATUS.DRIVER_ACCEPTED),
      driver: {
        id: driver.id,
        firstName: driver.first_name,
        lastName: driver.last_name,
        fullName: `${driver.first_name || ''} ${driver.last_name || ''}`.trim(),
        phone: driver.phoneNumber,
        photo: driver.profile_image,
        rating: driver.average_rating || driver.rating,
        totalTrips: driver.total_trips,
        location: driver.latitude && driver.longitude ? {
          latitude: parseFloat(driver.latitude),
          longitude: parseFloat(driver.longitude),
        } : null,
      },
      vehicle: vehicle ? {
        id: vehicle.id,
        make: vehicle.vehicle_make,
        model: vehicle.vehicle_model,
        color: vehicle.vehicle_color,
        plateNumber: vehicle.vehicle_plate_number,
        image: vehicle.vehicle_image,
      } : null,
      eta: booking.eta,
      otp: booking.ride_otp, // OTP for ride verification
      timestamp: new Date().toISOString(),
    });
  }

  /**
   * Notify driver on the way to pickup
   */
  notifyDriverOnTheWay(userId: number, bookingId: number, eta?: number) {
    this.sendToUser(userId, 'booking:driver_on_way', {
      bookingId,
      status: BOOKING_STATUS.DRIVER_ON_THE_WAY,
      statusLabel: getStatusLabel(BOOKING_STATUS.DRIVER_ON_THE_WAY),
      eta,
      timestamp: new Date().toISOString(),
    });
  }

  /**
   * Notify driver arrived at pickup
   */
  notifyDriverArrived(userId: number, bookingId: number) {
    this.sendToUser(userId, 'booking:driver_arrived', {
      bookingId,
      status: BOOKING_STATUS.DRIVER_ARRIVED,
      statusLabel: getStatusLabel(BOOKING_STATUS.DRIVER_ARRIVED),
      message: 'Your driver has arrived',
      timestamp: new Date().toISOString(),
    });
  }

  /**
   * Notify ride started
   */
  notifyRideStarted(userId: number, bookingId: number) {
    this.sendToUser(userId, 'booking:ride_started', {
      bookingId,
      status: BOOKING_STATUS.RIDE_STARTED,
      statusLabel: getStatusLabel(BOOKING_STATUS.RIDE_STARTED),
      message: 'Your ride has started',
      timestamp: new Date().toISOString(),
    });
  }

  /**
   * Notify ride completed with invoice details
   */
  notifyRideCompleted(userId: number, booking: any, transaction?: any) {
    this.sendToUser(userId, 'booking:ride_completed', {
      bookingId: booking.id,
      merchantBookingId: booking.merchant_booking_id,
      status: BOOKING_STATUS.RIDE_COMPLETED,
      statusLabel: getStatusLabel(BOOKING_STATUS.RIDE_COMPLETED),
      fare: {
        estimatedAmount: booking.estimate_amount,
        finalAmount: booking.final_amount,
        travelDistance: booking.travel_distance,
        travelTime: booking.travel_time,
        baseFare: transaction?.base_fare,
        distanceFare: transaction?.distance_fare,
        timeFare: transaction?.time_fare,
        surgeAmount: transaction?.surge_amount,
        discount: transaction?.discount_amount,
        tax: transaction?.tax_amount,
        tip: transaction?.tip,
        currency: 'XOF',
      },
      paymentStatus: booking.payment_status,
      paymentMethod: booking.payment_method_id,
      timestamp: new Date().toISOString(),
    });
  }

  /**
   * Notify booking cancelled
   */
  notifyBookingCancelled(
    booking: any,
    cancelledBy: 'user' | 'driver' | 'system',
    reason?: string,
    cancellationFee?: number,
  ) {
    const cancelStatus = cancelledBy === 'user'
      ? BOOKING_STATUS.CANCELLED_BY_USER
      : cancelledBy === 'driver'
        ? BOOKING_STATUS.CANCELLED_BY_DRIVER
        : BOOKING_STATUS.CANCELLED_BY_SYSTEM;

    const data = {
      bookingId: booking.id,
      merchantBookingId: booking.merchant_booking_id,
      status: cancelStatus,
      statusLabel: getStatusLabel(cancelStatus),
      cancelledBy,
      reason,
      cancellationFee,
      refundAmount: cancellationFee ? 0 : booking.estimate_amount,
      timestamp: new Date().toISOString(),
    };

    if (booking.user_id) {
      this.sendToUser(booking.user_id, 'booking:cancelled', data);
    }
    if (booking.driver_id) {
      this.sendToDriver(booking.driver_id, 'booking:cancelled', data);
    }

    this.sendToBooking(booking.id, 'booking:cancelled', data);
  }

  /**
   * Notify no driver found
   */
  notifyNoDriverFound(userId: number, bookingId: number) {
    this.sendToUser(userId, 'booking:no_driver_found', {
      bookingId,
      status: BOOKING_STATUS.NO_DRIVER_FOUND,
      statusLabel: getStatusLabel(BOOKING_STATUS.NO_DRIVER_FOUND),
      message: 'No drivers available at the moment. Please try again.',
      timestamp: new Date().toISOString(),
    });
  }

  /**
   * Notify driver request expired
   */
  notifyDriverRequestExpired(driverId: number, bookingId: number) {
    this.sendToDriver(driverId, 'booking:request_expired', {
      bookingId,
      message: 'Booking request has expired',
      timestamp: new Date().toISOString(),
    });
  }

  // ============================================================================
  // PAYMENT EVENTS
  // ============================================================================

  /**
   * Notify payment completed
   */
  notifyPaymentCompleted(userId: number, bookingId: number, amount: number, method: string) {
    this.sendToUser(userId, 'payment:completed', {
      bookingId,
      amount,
      method,
      status: BOOKING_STATUS.PAYMENT_COMPLETED,
      statusLabel: getStatusLabel(BOOKING_STATUS.PAYMENT_COMPLETED),
      timestamp: new Date().toISOString(),
    });
  }

  /**
   * Notify payment failed
   */
  notifyPaymentFailed(userId: number, bookingId: number, error: string) {
    this.sendToUser(userId, 'payment:failed', {
      bookingId,
      error,
      status: BOOKING_STATUS.PAYMENT_FAILED,
      statusLabel: getStatusLabel(BOOKING_STATUS.PAYMENT_FAILED),
      timestamp: new Date().toISOString(),
    });
  }

  // ============================================================================
  // DRIVER EARNINGS EVENTS
  // ============================================================================

  /**
   * Notify driver of new earnings
   */
  notifyDriverEarnings(driverId: number, booking: any, earnings: {
    totalFare: number;
    commission: number;
    netEarning: number;
    tip: number;
  }) {
    this.sendToDriver(driverId, 'earnings:new', {
      bookingId: booking.id,
      merchantBookingId: booking.merchant_booking_id,
      ...earnings,
      currency: 'XOF',
      timestamp: new Date().toISOString(),
    });
  }

  // ============================================================================
  // DELIVERY EVENTS
  // ============================================================================

  notifyDeliveryAccepted(userId: number, data: any) {
    this.sendToUser(userId, 'delivery:accepted', {
      ...data,
      timestamp: new Date().toISOString(),
    });
  }

  notifyDeliveryStatusChanged(userId: number, data: any) {
    this.sendToUser(userId, 'delivery:status_changed', {
      ...data,
      timestamp: new Date().toISOString(),
    });
  }

  notifyDeliveryCancelled(driverId: number, data: any) {
    this.sendToDriver(driverId, 'delivery:cancelled', {
      ...data,
      timestamp: new Date().toISOString(),
    });
  }

  broadcastDeliveryDriverLocation(
    deliveryId: number,
    userId: number,
    location: {
      latitude: number;
      longitude: number;
      heading?: number;
    },
  ) {
    this.sendToUser(userId, 'delivery:driver_location', {
      delivery_id: deliveryId,
      ...location,
      timestamp: new Date().toISOString(),
    });
  }

  // ============================================================================
  // HELPERS
  // ============================================================================

  private calculateDistance(
    lat1: number,
    lon1: number,
    lat2: number,
    lon2: number,
  ): number {
    const R = 6371; // Earth's radius in km
    const dLat = this.toRad(lat2 - lat1);
    const dLon = this.toRad(lon2 - lon1);
    const a =
      Math.sin(dLat / 2) * Math.sin(dLat / 2) +
      Math.cos(this.toRad(lat1)) *
        Math.cos(this.toRad(lat2)) *
        Math.sin(dLon / 2) *
        Math.sin(dLon / 2);
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
    return R * c;
  }

  private toRad(deg: number): number {
    return deg * (Math.PI / 180);
  }
}
