import {
  Controller,
  Get,
  Post,
  Body,
  Param,
  Query,
  UseGuards,
  ParseIntPipe,
} from '@nestjs/common';
import { ApiTags, ApiOperation, ApiBearerAuth } from '@nestjs/swagger';
import {
  WalletService,
  TransactionType,
  TransactionReason,
  WithdrawalStatus,
} from './wallet.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantGuard } from '../../common/guards/merchant.guard';
import { MerchantId, CurrentUser } from '../../common/decorators/merchant.decorator';

@ApiTags('Wallet')
@Controller('api/wallet')
@UseGuards(JwtAuthGuard, MerchantGuard)
@ApiBearerAuth()
export class WalletController {
  constructor(private walletService: WalletService) {}

  // ============================================================================
  // USER ENDPOINTS
  // ============================================================================

  @Get()
  @ApiOperation({ summary: 'Get wallet details' })
  async getWalletDetails(@CurrentUser() user: any) {
    const userType = user.type || 'user';
    return this.walletService.getWalletDetails(user.id, userType);
  }

  @Get('balance')
  @ApiOperation({ summary: 'Get wallet balance' })
  async getBalance(@CurrentUser() user: any) {
    const userType = user.type || 'user';
    const balance = await this.walletService.getBalance(user.id, userType);
    return { balance };
  }

  @Get('transactions')
  @ApiOperation({ summary: 'Get transaction history' })
  async getTransactions(
    @CurrentUser() user: any,
    @Query('type') type?: TransactionType,
    @Query('reason') reason?: TransactionReason,
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
  ) {
    const userType = user.type || 'user';
    return this.walletService.getTransactionHistory(user.id, userType, {
      type,
      reason,
      startDate: startDate ? new Date(startDate) : undefined,
      endDate: endDate ? new Date(endDate) : undefined,
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 20,
    });
  }

  // ============================================================================
  // TOP-UP
  // ============================================================================

  @Post('topup')
  @ApiOperation({ summary: 'Create top-up request' })
  async createTopUp(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
    @Body() body: { amount: number; paymentMethod: string },
  ) {
    const userType = user.type || 'user';
    return this.walletService.createTopUpRequest(
      user.id,
      userType,
      merchantId,
      body.amount,
      body.paymentMethod,
    );
  }

  @Post('topup/:id/confirm')
  @ApiOperation({ summary: 'Confirm top-up (webhook)' })
  async confirmTopUp(
    @Param('id', ParseIntPipe) topupId: number,
    @Body() body: { paymentReference: string },
  ) {
    return this.walletService.confirmTopUp(topupId, body.paymentReference);
  }

  // ============================================================================
  // WITHDRAWAL
  // ============================================================================

  @Post('withdraw')
  @ApiOperation({ summary: 'Request withdrawal' })
  async requestWithdrawal(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
    @Body()
    body: {
      amount: number;
      withdrawalMethod: string;
      accountNumber?: string;
      accountName?: string;
      bankName?: string;
      mobileNumber?: string;
      provider?: string;
    },
  ) {
    const userType = user.type || 'user';
    return this.walletService.requestWithdrawal(
      user.id,
      userType,
      merchantId,
      body.amount,
      body.withdrawalMethod,
      {
        accountNumber: body.accountNumber,
        accountName: body.accountName,
        bankName: body.bankName,
        mobileNumber: body.mobileNumber,
        provider: body.provider,
      },
    );
  }

  @Get('withdrawals')
  @ApiOperation({ summary: 'Get my withdrawals' })
  async getMyWithdrawals(
    @CurrentUser() user: any,
    @Query('status') status?: WithdrawalStatus,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
  ) {
    const userType = user.type || 'user';
    return this.walletService.getWithdrawals(user.id, userType, {
      status,
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 20,
    });
  }

  @Post('withdrawals/:id/cancel')
  @ApiOperation({ summary: 'Cancel withdrawal request' })
  async cancelWithdrawal(
    @Param('id', ParseIntPipe) withdrawalId: number,
    @CurrentUser() user: any,
  ) {
    await this.walletService.cancelWithdrawal(withdrawalId, user.id);
    return { message: 'Demande de retrait annulee' };
  }

  // ============================================================================
  // TRANSFER
  // ============================================================================

  @Post('transfer')
  @ApiOperation({ summary: 'Transfer to another wallet' })
  async transfer(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
    @Body()
    body: {
      toUserId: number;
      toUserType: 'user' | 'driver';
      amount: number;
      description?: string;
    },
  ) {
    const userType = user.type || 'user';
    return this.walletService.transfer(
      user.id,
      userType,
      body.toUserId,
      body.toUserType,
      merchantId,
      body.amount,
      body.description,
    );
  }

  // ============================================================================
  // ADMIN ENDPOINTS
  // ============================================================================

  @Get('admin/withdrawals/pending')
  @ApiOperation({ summary: 'Get pending withdrawals (admin)' })
  async getPendingWithdrawals(
    @MerchantId() merchantId: number,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
  ) {
    return this.walletService.getPendingWithdrawals(merchantId, {
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 20,
    });
  }

  @Post('admin/withdrawals/:id/process')
  @ApiOperation({ summary: 'Process withdrawal (admin)' })
  async processWithdrawal(
    @Param('id', ParseIntPipe) withdrawalId: number,
    @CurrentUser() admin: any,
  ) {
    return this.walletService.processWithdrawal(withdrawalId, admin.id);
  }

  @Post('admin/withdrawals/:id/complete')
  @ApiOperation({ summary: 'Complete withdrawal (admin)' })
  async completeWithdrawal(
    @Param('id', ParseIntPipe) withdrawalId: number,
    @CurrentUser() admin: any,
    @Body() body?: { transactionReference?: string },
  ) {
    return this.walletService.completeWithdrawal(
      withdrawalId,
      admin.id,
      body?.transactionReference,
    );
  }

  @Post('admin/withdrawals/:id/reject')
  @ApiOperation({ summary: 'Reject withdrawal (admin)' })
  async rejectWithdrawal(
    @Param('id', ParseIntPipe) withdrawalId: number,
    @CurrentUser() admin: any,
    @Body() body: { reason: string },
  ) {
    return this.walletService.rejectWithdrawal(withdrawalId, admin.id, body.reason);
  }

  @Post('admin/adjustment')
  @ApiOperation({ summary: 'Manual wallet adjustment (admin)' })
  async manualAdjustment(
    @CurrentUser() admin: any,
    @MerchantId() merchantId: number,
    @Body()
    body: {
      userId: number;
      userType: 'user' | 'driver';
      amount: number;
      type: TransactionType;
      reason: string;
    },
  ) {
    return this.walletService.manualAdjustment(
      admin.id,
      body.userId,
      body.userType,
      merchantId,
      body.amount,
      body.type,
      body.reason,
    );
  }
}
