import { Injectable, NotFoundException, BadRequestException } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';

// ============================================================================
// TYPES
// ============================================================================

export interface VehicleType {
  id: number;
  name: string;
  slug: string;
  description?: string;
  icon?: string;
  image?: string;
  capacity: number;
  baseFare: number;
  perKmRate: number;
  perMinRate: number;
  minFare: number;
  active: boolean;
  sortOrder: number;
}

export interface Vehicle {
  id: number;
  driverId: number;
  vehicleTypeId: number;
  make: string;
  model: string;
  year: number;
  color: string;
  plateNumber: string;
  vin?: string;
  registrationNumber?: string;
  insuranceNumber?: string;
  insuranceExpiry?: Date;
  inspectionExpiry?: Date;
  capacity: number;
  features: string[];
  images: string[];
  status: VehicleStatus;
  isDefault: boolean;
  verified: boolean;
  verifiedAt?: Date;
  merchantId: number;
  createdAt: Date;
  updatedAt: Date;
}

export type VehicleStatus = 'pending' | 'active' | 'inactive' | 'suspended' | 'rejected';

export interface VehicleDocument {
  id: number;
  vehicleId: number;
  type: string;
  fileUrl: string;
  expiryDate?: Date;
  status: 'pending' | 'approved' | 'rejected';
  rejectionReason?: string;
  verifiedAt?: Date;
  verifiedBy?: number;
}

export interface CreateVehicleDto {
  vehicleTypeId: number;
  make: string;
  model: string;
  year: number;
  color: string;
  plateNumber: string;
  vin?: string;
  registrationNumber?: string;
  insuranceNumber?: string;
  insuranceExpiry?: Date;
  inspectionExpiry?: Date;
  capacity?: number;
  features?: string[];
  images?: string[];
}

export interface UpdateVehicleDto extends Partial<CreateVehicleDto> {
  status?: VehicleStatus;
  isDefault?: boolean;
}

// ============================================================================
// SERVICE
// ============================================================================

@Injectable()
export class VehicleService {
  constructor(private readonly prisma: PrismaService) {}

  // ==========================================================================
  // VEHICLE TYPES
  // ==========================================================================

  /**
   * Obtenir tous les types de véhicules
   */
  async getVehicleTypes(merchantId: number, activeOnly = true): Promise<VehicleType[]> {
    const where: any = { merchantId };
    if (activeOnly) {
      where.active = true;
    }

    const types = await this.prisma.vehicleType.findMany({
      where,
      orderBy: { sortOrder: 'asc' },
    });

    return types.map(this.mapVehicleType);
  }

  /**
   * Obtenir un type de véhicule par ID
   */
  async getVehicleType(id: number, merchantId: number): Promise<VehicleType | null> {
    const type = await this.prisma.vehicleType.findFirst({
      where: { id, merchantId },
    });

    return type ? this.mapVehicleType(type) : null;
  }

  /**
   * Créer un type de véhicule
   */
  async createVehicleType(
    merchantId: number,
    data: {
      name: string;
      slug: string;
      description?: string;
      icon?: string;
      image?: string;
      capacity: number;
      baseFare: number;
      perKmRate: number;
      perMinRate: number;
      minFare: number;
      sortOrder?: number;
    },
  ): Promise<VehicleType> {
    // Vérifier slug unique
    const existing = await this.prisma.vehicleType.findFirst({
      where: { slug: data.slug, merchantId },
    });

    if (existing) {
      throw new BadRequestException('Ce slug existe déjà');
    }

    const type = await this.prisma.vehicleType.create({
      data: {
        ...data,
        merchantId,
        active: true,
        sortOrder: data.sortOrder ?? 0,
      },
    });

    return this.mapVehicleType(type);
  }

  /**
   * Mettre à jour un type de véhicule
   */
  async updateVehicleType(
    id: number,
    merchantId: number,
    data: Partial<{
      name: string;
      description: string;
      icon: string;
      image: string;
      capacity: number;
      baseFare: number;
      perKmRate: number;
      perMinRate: number;
      minFare: number;
      sortOrder: number;
      active: boolean;
    }>,
  ): Promise<VehicleType | null> {
    const type = await this.prisma.vehicleType.findFirst({
      where: { id, merchantId },
    });

    if (!type) {
      return null;
    }

    const updated = await this.prisma.vehicleType.update({
      where: { id },
      data,
    });

    return this.mapVehicleType(updated);
  }

  /**
   * Supprimer un type de véhicule
   */
  async deleteVehicleType(id: number, merchantId: number): Promise<boolean> {
    const type = await this.prisma.vehicleType.findFirst({
      where: { id, merchantId },
    });

    if (!type) {
      return false;
    }

    // Vérifier si des véhicules utilisent ce type
    const vehicleCount = await this.prisma.vehicle.count({
      where: { vehicleTypeId: id },
    });

    if (vehicleCount > 0) {
      throw new BadRequestException(
        `Ce type est utilisé par ${vehicleCount} véhicule(s). Désactivez-le plutôt.`,
      );
    }

    await this.prisma.vehicleType.delete({ where: { id } });
    return true;
  }

  // ==========================================================================
  // VEHICLES (DRIVER)
  // ==========================================================================

  /**
   * Obtenir les véhicules d'un chauffeur
   */
  async getDriverVehicles(driverId: number, merchantId: number): Promise<Vehicle[]> {
    const vehicles = await this.prisma.vehicle.findMany({
      where: { driverId, merchantId },
      include: {
        vehicleType: true,
      },
      orderBy: [
        { isDefault: 'desc' },
        { createdAt: 'desc' },
      ],
    });

    return vehicles.map(this.mapVehicle);
  }

  /**
   * Obtenir un véhicule par ID
   */
  async getVehicle(id: number, merchantId: number): Promise<Vehicle | null> {
    const vehicle = await this.prisma.vehicle.findFirst({
      where: { id, merchantId },
      include: {
        vehicleType: true,
        driver: {
          select: {
            id: true,
            firstName: true,
            lastName: true,
            phone: true,
          },
        },
      },
    });

    return vehicle ? this.mapVehicle(vehicle) : null;
  }

  /**
   * Créer un véhicule
   */
  async createVehicle(
    driverId: number,
    merchantId: number,
    data: CreateVehicleDto,
  ): Promise<Vehicle> {
    // Vérifier le type de véhicule
    const vehicleType = await this.prisma.vehicleType.findFirst({
      where: { id: data.vehicleTypeId, merchantId, active: true },
    });

    if (!vehicleType) {
      throw new BadRequestException('Type de véhicule invalide');
    }

    // Vérifier la plaque unique
    const existingPlate = await this.prisma.vehicle.findFirst({
      where: { plateNumber: data.plateNumber, merchantId },
    });

    if (existingPlate) {
      throw new BadRequestException('Cette plaque d\'immatriculation existe déjà');
    }

    // Vérifier si c'est le premier véhicule (sera par défaut)
    const vehicleCount = await this.prisma.vehicle.count({
      where: { driverId },
    });

    const vehicle = await this.prisma.vehicle.create({
      data: {
        driverId,
        merchantId,
        vehicleTypeId: data.vehicleTypeId,
        make: data.make,
        model: data.model,
        year: data.year,
        color: data.color,
        plateNumber: data.plateNumber.toUpperCase(),
        vin: data.vin,
        registrationNumber: data.registrationNumber,
        insuranceNumber: data.insuranceNumber,
        insuranceExpiry: data.insuranceExpiry,
        inspectionExpiry: data.inspectionExpiry,
        capacity: data.capacity ?? vehicleType.capacity,
        features: data.features ?? [],
        images: data.images ?? [],
        status: 'pending',
        isDefault: vehicleCount === 0,
        verified: false,
      },
      include: {
        vehicleType: true,
      },
    });

    return this.mapVehicle(vehicle);
  }

  /**
   * Mettre à jour un véhicule
   */
  async updateVehicle(
    id: number,
    driverId: number,
    merchantId: number,
    data: UpdateVehicleDto,
  ): Promise<Vehicle | null> {
    const vehicle = await this.prisma.vehicle.findFirst({
      where: { id, driverId, merchantId },
    });

    if (!vehicle) {
      return null;
    }

    // Vérifier la plaque si modifiée
    if (data.plateNumber && data.plateNumber !== vehicle.plateNumber) {
      const existingPlate = await this.prisma.vehicle.findFirst({
        where: {
          plateNumber: data.plateNumber,
          merchantId,
          id: { not: id },
        },
      });

      if (existingPlate) {
        throw new BadRequestException('Cette plaque d\'immatriculation existe déjà');
      }
    }

    const updated = await this.prisma.vehicle.update({
      where: { id },
      data: {
        ...data,
        plateNumber: data.plateNumber?.toUpperCase(),
        // Reset verification si données critiques changées
        ...(data.plateNumber || data.vin
          ? { verified: false, verifiedAt: null }
          : {}),
      },
      include: {
        vehicleType: true,
      },
    });

    return this.mapVehicle(updated);
  }

  /**
   * Définir le véhicule par défaut
   */
  async setDefaultVehicle(
    id: number,
    driverId: number,
    merchantId: number,
  ): Promise<boolean> {
    const vehicle = await this.prisma.vehicle.findFirst({
      where: { id, driverId, merchantId, status: 'active' },
    });

    if (!vehicle) {
      throw new NotFoundException('Véhicule non trouvé ou inactif');
    }

    // Transaction pour mettre à jour
    await this.prisma.$transaction([
      // Retirer le défaut des autres
      this.prisma.vehicle.updateMany({
        where: { driverId, isDefault: true },
        data: { isDefault: false },
      }),
      // Définir ce véhicule comme défaut
      this.prisma.vehicle.update({
        where: { id },
        data: { isDefault: true },
      }),
    ]);

    return true;
  }

  /**
   * Supprimer un véhicule
   */
  async deleteVehicle(
    id: number,
    driverId: number,
    merchantId: number,
  ): Promise<boolean> {
    const vehicle = await this.prisma.vehicle.findFirst({
      where: { id, driverId, merchantId },
    });

    if (!vehicle) {
      return false;
    }

    // Vérifier si des courses sont en cours
    const activeBookings = await this.prisma.booking.count({
      where: {
        vehicleId: id,
        status: { in: ['pending', 'accepted', 'arrived', 'started'] },
      },
    });

    if (activeBookings > 0) {
      throw new BadRequestException(
        'Ce véhicule est utilisé dans des courses en cours',
      );
    }

    await this.prisma.vehicle.delete({ where: { id } });

    // Si c'était le véhicule par défaut, en définir un autre
    if (vehicle.isDefault) {
      const nextVehicle = await this.prisma.vehicle.findFirst({
        where: { driverId, status: 'active' },
        orderBy: { createdAt: 'asc' },
      });

      if (nextVehicle) {
        await this.prisma.vehicle.update({
          where: { id: nextVehicle.id },
          data: { isDefault: true },
        });
      }
    }

    return true;
  }

  // ==========================================================================
  // ADMIN OPERATIONS
  // ==========================================================================

  /**
   * Lister tous les véhicules (admin)
   */
  async listVehicles(
    merchantId: number,
    options: {
      page?: number;
      limit?: number;
      status?: VehicleStatus;
      vehicleTypeId?: number;
      verified?: boolean;
      search?: string;
    } = {},
  ): Promise<{
    vehicles: Vehicle[];
    total: number;
    page: number;
    totalPages: number;
  }> {
    const page = options.page || 1;
    const limit = options.limit || 20;
    const skip = (page - 1) * limit;

    const where: any = { merchantId };

    if (options.status) {
      where.status = options.status;
    }

    if (options.vehicleTypeId) {
      where.vehicleTypeId = options.vehicleTypeId;
    }

    if (options.verified !== undefined) {
      where.verified = options.verified;
    }

    if (options.search) {
      where.OR = [
        { plateNumber: { contains: options.search } },
        { make: { contains: options.search } },
        { model: { contains: options.search } },
        { driver: { firstName: { contains: options.search } } },
        { driver: { lastName: { contains: options.search } } },
      ];
    }

    const [vehicles, total] = await Promise.all([
      this.prisma.vehicle.findMany({
        where,
        include: {
          vehicleType: true,
          driver: {
            select: {
              id: true,
              firstName: true,
              lastName: true,
              phone: true,
              profilePicture: true,
            },
          },
        },
        skip,
        take: limit,
        orderBy: { createdAt: 'desc' },
      }),
      this.prisma.vehicle.count({ where }),
    ]);

    return {
      vehicles: vehicles.map(this.mapVehicle),
      total,
      page,
      totalPages: Math.ceil(total / limit),
    };
  }

  /**
   * Vérifier un véhicule (admin)
   */
  async verifyVehicle(
    id: number,
    merchantId: number,
    adminId: number,
    approved: boolean,
    rejectionReason?: string,
  ): Promise<Vehicle | null> {
    const vehicle = await this.prisma.vehicle.findFirst({
      where: { id, merchantId },
    });

    if (!vehicle) {
      return null;
    }

    const updated = await this.prisma.vehicle.update({
      where: { id },
      data: {
        verified: approved,
        verifiedAt: approved ? new Date() : null,
        status: approved ? 'active' : 'rejected',
        verifiedBy: adminId,
        rejectionReason: approved ? null : rejectionReason,
      },
      include: {
        vehicleType: true,
      },
    });

    return this.mapVehicle(updated);
  }

  /**
   * Changer le statut d'un véhicule (admin)
   */
  async updateVehicleStatus(
    id: number,
    merchantId: number,
    status: VehicleStatus,
    reason?: string,
  ): Promise<Vehicle | null> {
    const vehicle = await this.prisma.vehicle.findFirst({
      where: { id, merchantId },
    });

    if (!vehicle) {
      return null;
    }

    const updated = await this.prisma.vehicle.update({
      where: { id },
      data: {
        status,
        statusReason: reason,
        statusChangedAt: new Date(),
      },
      include: {
        vehicleType: true,
      },
    });

    return this.mapVehicle(updated);
  }

  // ==========================================================================
  // VEHICLE DOCUMENTS
  // ==========================================================================

  /**
   * Obtenir les documents d'un véhicule
   */
  async getVehicleDocuments(vehicleId: number): Promise<VehicleDocument[]> {
    const documents = await this.prisma.vehicleDocument.findMany({
      where: { vehicleId },
      orderBy: { createdAt: 'desc' },
    });

    return documents.map(this.mapVehicleDocument);
  }

  /**
   * Ajouter un document
   */
  async addDocument(
    vehicleId: number,
    data: {
      type: string;
      fileUrl: string;
      expiryDate?: Date;
    },
  ): Promise<VehicleDocument> {
    // Supprimer l'ancien document du même type
    await this.prisma.vehicleDocument.deleteMany({
      where: { vehicleId, type: data.type },
    });

    const document = await this.prisma.vehicleDocument.create({
      data: {
        vehicleId,
        type: data.type,
        fileUrl: data.fileUrl,
        expiryDate: data.expiryDate,
        status: 'pending',
      },
    });

    return this.mapVehicleDocument(document);
  }

  /**
   * Vérifier un document (admin)
   */
  async verifyDocument(
    documentId: number,
    adminId: number,
    approved: boolean,
    rejectionReason?: string,
  ): Promise<VehicleDocument | null> {
    const document = await this.prisma.vehicleDocument.findUnique({
      where: { id: documentId },
    });

    if (!document) {
      return null;
    }

    const updated = await this.prisma.vehicleDocument.update({
      where: { id: documentId },
      data: {
        status: approved ? 'approved' : 'rejected',
        verifiedAt: new Date(),
        verifiedBy: adminId,
        rejectionReason: approved ? null : rejectionReason,
      },
    });

    return this.mapVehicleDocument(updated);
  }

  // ==========================================================================
  // STATISTICS
  // ==========================================================================

  /**
   * Statistiques véhicules
   */
  async getStatistics(merchantId: number): Promise<{
    total: number;
    byStatus: Record<VehicleStatus, number>;
    byType: { typeId: number; typeName: string; count: number }[];
    pendingVerification: number;
    expiringDocuments: number;
  }> {
    const [
      total,
      statusCounts,
      typeCounts,
      pendingVerification,
      expiringDocuments,
    ] = await Promise.all([
      this.prisma.vehicle.count({ where: { merchantId } }),
      this.prisma.vehicle.groupBy({
        by: ['status'],
        where: { merchantId },
        _count: { status: true },
      }),
      this.prisma.vehicle.groupBy({
        by: ['vehicleTypeId'],
        where: { merchantId },
        _count: { vehicleTypeId: true },
      }),
      this.prisma.vehicle.count({
        where: { merchantId, status: 'pending', verified: false },
      }),
      this.prisma.vehicleDocument.count({
        where: {
          vehicle: { merchantId },
          expiryDate: {
            lte: new Date(Date.now() + 30 * 24 * 60 * 60 * 1000), // 30 jours
            gte: new Date(),
          },
        },
      }),
    ]);

    // Récupérer les noms des types
    const vehicleTypes = await this.prisma.vehicleType.findMany({
      where: { merchantId },
      select: { id: true, name: true },
    });

    const typeMap = new Map(vehicleTypes.map((t) => [t.id, t.name]));

    const byStatus: Record<string, number> = {
      pending: 0,
      active: 0,
      inactive: 0,
      suspended: 0,
      rejected: 0,
    };

    statusCounts.forEach((s) => {
      byStatus[s.status] = s._count.status;
    });

    const byType = typeCounts.map((t) => ({
      typeId: t.vehicleTypeId,
      typeName: typeMap.get(t.vehicleTypeId) || 'Unknown',
      count: t._count.vehicleTypeId,
    }));

    return {
      total,
      byStatus: byStatus as Record<VehicleStatus, number>,
      byType,
      pendingVerification,
      expiringDocuments,
    };
  }

  // ==========================================================================
  // HELPERS
  // ==========================================================================

  private mapVehicleType(type: any): VehicleType {
    return {
      id: type.id,
      name: type.name,
      slug: type.slug,
      description: type.description,
      icon: type.icon,
      image: type.image,
      capacity: type.capacity,
      baseFare: parseFloat(type.baseFare || 0),
      perKmRate: parseFloat(type.perKmRate || 0),
      perMinRate: parseFloat(type.perMinRate || 0),
      minFare: parseFloat(type.minFare || 0),
      active: type.active,
      sortOrder: type.sortOrder,
    };
  }

  private mapVehicle(vehicle: any): Vehicle {
    return {
      id: vehicle.id,
      driverId: vehicle.driverId,
      vehicleTypeId: vehicle.vehicleTypeId,
      make: vehicle.make,
      model: vehicle.model,
      year: vehicle.year,
      color: vehicle.color,
      plateNumber: vehicle.plateNumber,
      vin: vehicle.vin,
      registrationNumber: vehicle.registrationNumber,
      insuranceNumber: vehicle.insuranceNumber,
      insuranceExpiry: vehicle.insuranceExpiry,
      inspectionExpiry: vehicle.inspectionExpiry,
      capacity: vehicle.capacity,
      features: vehicle.features || [],
      images: vehicle.images || [],
      status: vehicle.status,
      isDefault: vehicle.isDefault,
      verified: vehicle.verified,
      verifiedAt: vehicle.verifiedAt,
      merchantId: vehicle.merchantId,
      createdAt: vehicle.createdAt,
      updatedAt: vehicle.updatedAt,
    };
  }

  private mapVehicleDocument(doc: any): VehicleDocument {
    return {
      id: doc.id,
      vehicleId: doc.vehicleId,
      type: doc.type,
      fileUrl: doc.fileUrl,
      expiryDate: doc.expiryDate,
      status: doc.status,
      rejectionReason: doc.rejectionReason,
      verifiedAt: doc.verifiedAt,
      verifiedBy: doc.verifiedBy,
    };
  }
}
