import {
  Controller,
  Get,
  Post,
  Put,
  Delete,
  Body,
  Param,
  Query,
  UseGuards,
  ParseIntPipe,
  DefaultValuePipe,
  ParseBoolPipe,
} from '@nestjs/common';
import {
  VehicleService,
  CreateVehicleDto,
  UpdateVehicleDto,
  VehicleStatus,
} from './vehicle.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantId } from '../../common/decorators/merchant.decorator';
import { CurrentUser } from '../../common/decorators/current-user.decorator';

// ============================================================================
// DTOs
// ============================================================================

class CreateVehicleTypeDto {
  name: string;
  slug: string;
  description?: string;
  icon?: string;
  image?: string;
  capacity: number;
  baseFare: number;
  perKmRate: number;
  perMinRate: number;
  minFare: number;
  sortOrder?: number;
}

class UpdateVehicleTypeDto {
  name?: string;
  description?: string;
  icon?: string;
  image?: string;
  capacity?: number;
  baseFare?: number;
  perKmRate?: number;
  perMinRate?: number;
  minFare?: number;
  sortOrder?: number;
  active?: boolean;
}

class VerifyVehicleDto {
  approved: boolean;
  rejectionReason?: string;
}

class AddDocumentDto {
  type: string;
  fileUrl: string;
  expiryDate?: string;
}

// ============================================================================
// CONTROLLER
// ============================================================================

@Controller('vehicles')
@UseGuards(JwtAuthGuard)
export class VehicleController {
  constructor(private readonly vehicleService: VehicleService) {}

  // ==========================================================================
  // VEHICLE TYPES (PUBLIC)
  // ==========================================================================

  /**
   * Lister les types de véhicules
   */
  @Get('types')
  async getVehicleTypes(
    @MerchantId() merchantId: number,
    @Query('all', new DefaultValuePipe(false), ParseBoolPipe) all: boolean,
  ) {
    return this.vehicleService.getVehicleTypes(merchantId, !all);
  }

  /**
   * Obtenir un type de véhicule
   */
  @Get('types/:id')
  async getVehicleType(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const type = await this.vehicleService.getVehicleType(id, merchantId);
    if (!type) {
      return { error: 'Type de véhicule non trouvé' };
    }
    return type;
  }

  /**
   * Créer un type de véhicule (admin)
   */
  @Post('types')
  async createVehicleType(
    @MerchantId() merchantId: number,
    @Body() dto: CreateVehicleTypeDto,
  ) {
    const type = await this.vehicleService.createVehicleType(merchantId, dto);
    return {
      success: true,
      vehicleType: type,
    };
  }

  /**
   * Mettre à jour un type de véhicule (admin)
   */
  @Put('types/:id')
  async updateVehicleType(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: UpdateVehicleTypeDto,
  ) {
    const type = await this.vehicleService.updateVehicleType(id, merchantId, dto);
    if (!type) {
      return { error: 'Type de véhicule non trouvé' };
    }
    return {
      success: true,
      vehicleType: type,
    };
  }

  /**
   * Supprimer un type de véhicule (admin)
   */
  @Delete('types/:id')
  async deleteVehicleType(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const deleted = await this.vehicleService.deleteVehicleType(id, merchantId);
    return {
      success: deleted,
      message: deleted ? 'Type de véhicule supprimé' : 'Type de véhicule non trouvé',
    };
  }

  // ==========================================================================
  // DRIVER VEHICLES
  // ==========================================================================

  /**
   * Mes véhicules (chauffeur)
   */
  @Get('my')
  async getMyVehicles(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
  ) {
    return this.vehicleService.getDriverVehicles(user.driverId, merchantId);
  }

  /**
   * Ajouter un véhicule (chauffeur)
   */
  @Post('my')
  async addMyVehicle(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
    @Body() dto: CreateVehicleDto,
  ) {
    const vehicle = await this.vehicleService.createVehicle(
      user.driverId,
      merchantId,
      dto,
    );
    return {
      success: true,
      vehicle,
      message: 'Véhicule ajouté, en attente de vérification',
    };
  }

  /**
   * Mettre à jour mon véhicule (chauffeur)
   */
  @Put('my/:id')
  async updateMyVehicle(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: UpdateVehicleDto,
  ) {
    const vehicle = await this.vehicleService.updateVehicle(
      id,
      user.driverId,
      merchantId,
      dto,
    );
    if (!vehicle) {
      return { error: 'Véhicule non trouvé' };
    }
    return {
      success: true,
      vehicle,
    };
  }

  /**
   * Définir véhicule par défaut (chauffeur)
   */
  @Post('my/:id/default')
  async setDefaultVehicle(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    await this.vehicleService.setDefaultVehicle(id, user.driverId, merchantId);
    return {
      success: true,
      message: 'Véhicule défini par défaut',
    };
  }

  /**
   * Supprimer mon véhicule (chauffeur)
   */
  @Delete('my/:id')
  async deleteMyVehicle(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const deleted = await this.vehicleService.deleteVehicle(
      id,
      user.driverId,
      merchantId,
    );
    return {
      success: deleted,
      message: deleted ? 'Véhicule supprimé' : 'Véhicule non trouvé',
    };
  }

  // ==========================================================================
  // VEHICLE DOCUMENTS (DRIVER)
  // ==========================================================================

  /**
   * Documents de mon véhicule
   */
  @Get('my/:id/documents')
  async getMyVehicleDocuments(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) vehicleId: number,
  ) {
    // Vérifier que le véhicule appartient au chauffeur
    const vehicle = await this.vehicleService.getVehicle(vehicleId, merchantId);
    if (!vehicle || vehicle.driverId !== user.driverId) {
      return { error: 'Véhicule non trouvé' };
    }

    return this.vehicleService.getVehicleDocuments(vehicleId);
  }

  /**
   * Ajouter un document
   */
  @Post('my/:id/documents')
  async addMyVehicleDocument(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) vehicleId: number,
    @Body() dto: AddDocumentDto,
  ) {
    // Vérifier que le véhicule appartient au chauffeur
    const vehicle = await this.vehicleService.getVehicle(vehicleId, merchantId);
    if (!vehicle || vehicle.driverId !== user.driverId) {
      return { error: 'Véhicule non trouvé' };
    }

    const document = await this.vehicleService.addDocument(vehicleId, {
      type: dto.type,
      fileUrl: dto.fileUrl,
      expiryDate: dto.expiryDate ? new Date(dto.expiryDate) : undefined,
    });

    return {
      success: true,
      document,
      message: 'Document ajouté, en attente de vérification',
    };
  }

  // ==========================================================================
  // ADMIN OPERATIONS
  // ==========================================================================

  /**
   * Lister tous les véhicules (admin)
   */
  @Get()
  async listVehicles(
    @MerchantId() merchantId: number,
    @Query('page', new DefaultValuePipe(1), ParseIntPipe) page: number,
    @Query('limit', new DefaultValuePipe(20), ParseIntPipe) limit: number,
    @Query('status') status?: VehicleStatus,
    @Query('vehicleTypeId') vehicleTypeId?: string,
    @Query('verified') verified?: string,
    @Query('search') search?: string,
  ) {
    return this.vehicleService.listVehicles(merchantId, {
      page,
      limit,
      status,
      vehicleTypeId: vehicleTypeId ? parseInt(vehicleTypeId) : undefined,
      verified: verified !== undefined ? verified === 'true' : undefined,
      search,
    });
  }

  /**
   * Obtenir un véhicule (admin)
   */
  @Get(':id')
  async getVehicle(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const vehicle = await this.vehicleService.getVehicle(id, merchantId);
    if (!vehicle) {
      return { error: 'Véhicule non trouvé' };
    }
    return vehicle;
  }

  /**
   * Documents d'un véhicule (admin)
   */
  @Get(':id/documents')
  async getVehicleDocuments(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) vehicleId: number,
  ) {
    const vehicle = await this.vehicleService.getVehicle(vehicleId, merchantId);
    if (!vehicle) {
      return { error: 'Véhicule non trouvé' };
    }

    return this.vehicleService.getVehicleDocuments(vehicleId);
  }

  /**
   * Vérifier un véhicule (admin)
   */
  @Post(':id/verify')
  async verifyVehicle(
    @CurrentUser() admin: any,
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: VerifyVehicleDto,
  ) {
    const vehicle = await this.vehicleService.verifyVehicle(
      id,
      merchantId,
      admin.id,
      dto.approved,
      dto.rejectionReason,
    );

    if (!vehicle) {
      return { error: 'Véhicule non trouvé' };
    }

    return {
      success: true,
      vehicle,
      message: dto.approved ? 'Véhicule approuvé' : 'Véhicule rejeté',
    };
  }

  /**
   * Changer le statut d'un véhicule (admin)
   */
  @Put(':id/status')
  async updateVehicleStatus(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() body: { status: VehicleStatus; reason?: string },
  ) {
    const vehicle = await this.vehicleService.updateVehicleStatus(
      id,
      merchantId,
      body.status,
      body.reason,
    );

    if (!vehicle) {
      return { error: 'Véhicule non trouvé' };
    }

    return {
      success: true,
      vehicle,
    };
  }

  /**
   * Vérifier un document (admin)
   */
  @Post('documents/:documentId/verify')
  async verifyDocument(
    @CurrentUser() admin: any,
    @Param('documentId', ParseIntPipe) documentId: number,
    @Body() dto: VerifyVehicleDto,
  ) {
    const document = await this.vehicleService.verifyDocument(
      documentId,
      admin.id,
      dto.approved,
      dto.rejectionReason,
    );

    if (!document) {
      return { error: 'Document non trouvé' };
    }

    return {
      success: true,
      document,
      message: dto.approved ? 'Document approuvé' : 'Document rejeté',
    };
  }

  // ==========================================================================
  // STATISTICS
  // ==========================================================================

  /**
   * Statistiques véhicules (admin)
   */
  @Get('stats/overview')
  async getStatistics(@MerchantId() merchantId: number) {
    return this.vehicleService.getStatistics(merchantId);
  }
}
