// =============================================================================
// UserService Unit Tests
// =============================================================================

import { Test, TestingModule } from '@nestjs/testing';
import { NotFoundException } from '@nestjs/common';
import { UserService } from './user.service';
import { PrismaService } from '../../shared/database/prisma.service';
import {
  mockPrismaService,
  createMockUser,
  createMockBooking,
  createMockWalletTransaction,
  resetAllMocks,
} from '../../../test/setup';

describe('UserService', () => {
  let service: UserService;
  let prisma: typeof mockPrismaService;

  beforeEach(async () => {
    resetAllMocks();

    // Add missing prisma mocks
    mockPrismaService.userWalletTransaction = {
      findMany: jest.fn(),
      aggregate: jest.fn(),
      count: jest.fn(),
    };

    const module: TestingModule = await Test.createTestingModule({
      providers: [
        UserService,
        { provide: PrismaService, useValue: mockPrismaService },
      ],
    }).compile();

    service = module.get<UserService>(UserService);
    prisma = mockPrismaService;
  });

  afterEach(() => {
    jest.clearAllMocks();
  });

  // ===========================================================================
  // GET PROFILE
  // ===========================================================================

  describe('getProfile', () => {
    it('devrait retourner le profil utilisateur sans données sensibles', async () => {
      const mockUser = createMockUser();
      prisma.user.findUnique.mockResolvedValue(mockUser);

      const result = await service.getProfile(1);

      expect(result.message).toBe('User profile');
      expect(result.data).not.toHaveProperty('password');
      expect(result.data).not.toHaveProperty('otp');
      expect(result.data).not.toHaveProperty('otp_expires_at');
    });

    it('devrait lever une erreur si utilisateur non trouvé', async () => {
      prisma.user.findUnique.mockResolvedValue(null);

      await expect(service.getProfile(999)).rejects.toThrow(NotFoundException);
    });
  });

  // ===========================================================================
  // UPDATE PROFILE
  // ===========================================================================

  describe('updateProfile', () => {
    it('devrait mettre à jour le profil utilisateur', async () => {
      const mockUser = createMockUser();
      prisma.user.update.mockResolvedValue({
        ...mockUser,
        first_name: 'Updated',
        last_name: 'Name',
        email: 'updated@example.com',
      });

      const result = await service.updateProfile(1, {
        first_name: 'Updated',
        last_name: 'Name',
        email: 'updated@example.com',
      });

      expect(result.message).toBe('Profile updated successfully');
      expect(result.data.first_name).toBe('Updated');
      expect(result.data).not.toHaveProperty('password');
    });

    it('devrait permettre de mettre à jour uniquement certains champs', async () => {
      const mockUser = createMockUser();
      prisma.user.update.mockResolvedValue({
        ...mockUser,
        first_name: 'OnlyFirstName',
      });

      const result = await service.updateProfile(1, {
        first_name: 'OnlyFirstName',
      });

      expect(result.data.first_name).toBe('OnlyFirstName');
    });
  });

  // ===========================================================================
  // GET BOOKINGS
  // ===========================================================================

  describe('getBookings', () => {
    it('devrait retourner les courses de l\'utilisateur avec pagination', async () => {
      const mockBookings = [
        createMockBooking({ id: 1 }),
        createMockBooking({ id: 2 }),
      ];
      prisma.booking.findMany.mockResolvedValue(mockBookings);
      prisma.booking.count.mockResolvedValue(2);

      const result = await service.getBookings(1, 1, 10);

      expect(result.data.bookings).toHaveLength(2);
      expect(result.data.pagination).toEqual({
        page: 1,
        limit: 10,
        total: 2,
        total_pages: 1,
      });
    });

    it('devrait paginer correctement', async () => {
      prisma.booking.findMany.mockResolvedValue([createMockBooking()]);
      prisma.booking.count.mockResolvedValue(25);

      const result = await service.getBookings(1, 2, 10);

      expect(result.data.pagination.page).toBe(2);
      expect(result.data.pagination.total_pages).toBe(3);
    });

    it('devrait retourner une liste vide si pas de courses', async () => {
      prisma.booking.findMany.mockResolvedValue([]);
      prisma.booking.count.mockResolvedValue(0);

      const result = await service.getBookings(1, 1, 10);

      expect(result.data.bookings).toHaveLength(0);
      expect(result.data.pagination.total).toBe(0);
    });
  });

  // ===========================================================================
  // GET WALLET
  // ===========================================================================

  describe('getWallet', () => {
    it('devrait retourner le solde et les transactions', async () => {
      const mockTransactions = [
        createMockWalletTransaction({ id: 1, amount: 10000 }),
        createMockWalletTransaction({ id: 2, amount: -5000 }),
      ];

      prisma.userWalletTransaction.aggregate.mockResolvedValue({
        _sum: { amount: 5000 },
      });
      prisma.userWalletTransaction.findMany.mockResolvedValue(mockTransactions);
      prisma.userWalletTransaction.count.mockResolvedValue(2);

      const result = await service.getWallet(1, 1, 10);

      expect(result.data.balance).toBe(5000);
      expect(result.data.transactions).toHaveLength(2);
    });

    it('devrait retourner 0 si pas de transactions', async () => {
      prisma.userWalletTransaction.aggregate.mockResolvedValue({
        _sum: { amount: null },
      });
      prisma.userWalletTransaction.findMany.mockResolvedValue([]);
      prisma.userWalletTransaction.count.mockResolvedValue(0);

      const result = await service.getWallet(1, 1, 10);

      expect(result.data.balance).toBe(0);
      expect(result.data.transactions).toHaveLength(0);
    });

    it('devrait paginer les transactions correctement', async () => {
      prisma.userWalletTransaction.aggregate.mockResolvedValue({
        _sum: { amount: 50000 },
      });
      prisma.userWalletTransaction.findMany.mockResolvedValue([]);
      prisma.userWalletTransaction.count.mockResolvedValue(50);

      const result = await service.getWallet(1, 3, 10);

      expect(result.data.pagination.page).toBe(3);
      expect(result.data.pagination.total_pages).toBe(5);
    });
  });

  // ===========================================================================
  // GET FAVORITE DRIVERS
  // ===========================================================================

  describe('getFavoriteDrivers', () => {
    it('devrait retourner une liste vide (feature à implémenter)', async () => {
      const result = await service.getFavoriteDrivers(1);

      expect(result.message).toBe('Favorite drivers');
      expect(result.data).toEqual([]);
    });
  });
});
