import {
  Controller,
  Get,
  Put,
  Body,
  UseGuards,
  Query,
} from '@nestjs/common';
import { ApiTags, ApiOperation, ApiBearerAuth } from '@nestjs/swagger';

import { UserService } from './user.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantGuard } from '../../common/guards/merchant.guard';
import { CurrentUser, MerchantId } from '../../common/decorators/merchant.decorator';

import { UpdateProfileDto } from './dto/update-profile.dto';

@ApiTags('users')
@Controller('user')
@UseGuards(JwtAuthGuard, MerchantGuard)
@ApiBearerAuth()
export class UserController {
  constructor(private readonly userService: UserService) {}

  @Get('profile')
  @ApiOperation({ summary: 'Get user profile' })
  async getProfile(@CurrentUser() user: any) {
    return this.userService.getProfile(user.id);
  }

  @Put('profile')
  @ApiOperation({ summary: 'Update user profile' })
  async updateProfile(
    @CurrentUser() user: any,
    @Body() dto: UpdateProfileDto,
  ) {
    return this.userService.updateProfile(user.id, dto);
  }

  @Get('bookings')
  @ApiOperation({ summary: 'Get user booking history' })
  async getBookings(
    @CurrentUser() user: any,
    @Query('page') page: number = 1,
    @Query('limit') limit: number = 20,
  ) {
    return this.userService.getBookings(user.id, page, limit);
  }

  @Get('wallet')
  @ApiOperation({ summary: 'Get user wallet balance and transactions' })
  async getWallet(
    @CurrentUser() user: any,
    @Query('page') page: number = 1,
    @Query('limit') limit: number = 20,
  ) {
    return this.userService.getWallet(user.id, page, limit);
  }

  @Get('favorite-drivers')
  @ApiOperation({ summary: 'Get user favorite drivers' })
  async getFavoriteDrivers(@CurrentUser() user: any) {
    return this.userService.getFavoriteDrivers(user.id);
  }
}
