import { Injectable, NotFoundException, BadRequestException } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';

// ============================================================================
// TYPES
// ============================================================================

export type TicketStatus = 'open' | 'in_progress' | 'waiting_customer' | 'resolved' | 'closed';
export type TicketPriority = 'low' | 'medium' | 'high' | 'urgent';
export type TicketType = 'user' | 'driver';

export interface SupportTicket {
  id: number;
  ticketNumber: string;
  type: TicketType;
  userId?: number;
  driverId?: number;
  bookingId?: number;
  categoryId?: number;
  subject: string;
  description: string;
  status: TicketStatus;
  priority: TicketPriority;
  assignedTo?: number;
  tags: string[];
  attachments: string[];
  rating?: number;
  ratingComment?: string;
  resolvedAt?: Date;
  closedAt?: Date;
  firstResponseAt?: Date;
  merchantId: number;
  createdAt: Date;
  updatedAt: Date;
}

export interface TicketMessage {
  id: number;
  ticketId: number;
  senderId: number;
  senderType: 'user' | 'driver' | 'agent' | 'system';
  senderName: string;
  message: string;
  attachments: string[];
  isInternal: boolean; // Note interne (visible uniquement par les agents)
  createdAt: Date;
}

export interface TicketCategory {
  id: number;
  name: string;
  slug: string;
  description?: string;
  icon?: string;
  parentId?: number;
  priority: TicketPriority; // Priorité par défaut
  autoAssignTo?: number;
  slaHours?: number; // Temps de réponse attendu
  active: boolean;
  sortOrder: number;
}

export interface CreateTicketDto {
  type: TicketType;
  userId?: number;
  driverId?: number;
  bookingId?: number;
  categoryId?: number;
  subject: string;
  description: string;
  priority?: TicketPriority;
  attachments?: string[];
}

export interface UpdateTicketDto {
  categoryId?: number;
  subject?: string;
  priority?: TicketPriority;
  status?: TicketStatus;
  assignedTo?: number | null;
  tags?: string[];
}

// ============================================================================
// SERVICE
// ============================================================================

@Injectable()
export class SupportService {
  constructor(private readonly prisma: PrismaService) {}

  // ==========================================================================
  // TICKET CRUD
  // ==========================================================================

  /**
   * Créer un ticket
   */
  async createTicket(merchantId: number, data: CreateTicketDto): Promise<SupportTicket> {
    // Générer le numéro de ticket
    const ticketNumber = await this.generateTicketNumber(merchantId);

    // Récupérer la catégorie pour la priorité par défaut
    let priority = data.priority || 'medium';
    let assignedTo: number | undefined;

    if (data.categoryId) {
      const category = await this.prisma.supportCategory.findUnique({
        where: { id: data.categoryId },
      });
      if (category) {
        priority = data.priority || category.priority || 'medium';
        assignedTo = category.autoAssignTo || undefined;
      }
    }

    const ticket = await this.prisma.supportTicket.create({
      data: {
        merchantId,
        ticketNumber,
        type: data.type,
        userId: data.userId,
        driverId: data.driverId,
        bookingId: data.bookingId,
        categoryId: data.categoryId,
        subject: data.subject,
        description: data.description,
        status: 'open',
        priority,
        assignedTo,
        tags: [],
        attachments: data.attachments || [],
      },
    });

    // Créer le premier message (description)
    await this.prisma.ticketMessage.create({
      data: {
        ticketId: ticket.id,
        senderId: data.userId || data.driverId || 0,
        senderType: data.type === 'user' ? 'user' : 'driver',
        senderName: 'Client',
        message: data.description,
        attachments: data.attachments || [],
        isInternal: false,
      },
    });

    return this.mapTicket(ticket);
  }

  /**
   * Obtenir un ticket par ID
   */
  async getTicket(id: number, merchantId: number): Promise<SupportTicket | null> {
    const ticket = await this.prisma.supportTicket.findFirst({
      where: { id, merchantId },
      include: {
        category: true,
        assignedAgent: {
          select: {
            id: true,
            firstName: true,
            lastName: true,
          },
        },
      },
    });

    return ticket ? this.mapTicket(ticket) : null;
  }

  /**
   * Obtenir un ticket par numéro
   */
  async getTicketByNumber(
    ticketNumber: string,
    merchantId: number,
  ): Promise<SupportTicket | null> {
    const ticket = await this.prisma.supportTicket.findFirst({
      where: { ticketNumber, merchantId },
    });

    return ticket ? this.mapTicket(ticket) : null;
  }

  /**
   * Mettre à jour un ticket
   */
  async updateTicket(
    id: number,
    merchantId: number,
    data: UpdateTicketDto,
    agentId?: number,
  ): Promise<SupportTicket | null> {
    const ticket = await this.prisma.supportTicket.findFirst({
      where: { id, merchantId },
    });

    if (!ticket) {
      return null;
    }

    const updateData: any = { ...data };

    // Gérer les changements de statut
    if (data.status && data.status !== ticket.status) {
      if (data.status === 'resolved') {
        updateData.resolvedAt = new Date();
      } else if (data.status === 'closed') {
        updateData.closedAt = new Date();
      }

      // Ajouter un message système
      await this.addSystemMessage(id, `Statut changé: ${ticket.status} → ${data.status}`);
    }

    // Gérer l'assignation
    if (data.assignedTo !== undefined && data.assignedTo !== ticket.assignedTo) {
      const agentName = data.assignedTo
        ? await this.getAgentName(data.assignedTo)
        : 'Non assigné';
      await this.addSystemMessage(id, `Ticket assigné à: ${agentName}`);
    }

    const updated = await this.prisma.supportTicket.update({
      where: { id },
      data: updateData,
    });

    return this.mapTicket(updated);
  }

  /**
   * Lister les tickets
   */
  async listTickets(
    merchantId: number,
    options: {
      page?: number;
      limit?: number;
      status?: TicketStatus;
      priority?: TicketPriority;
      type?: TicketType;
      categoryId?: number;
      assignedTo?: number;
      userId?: number;
      driverId?: number;
      search?: string;
      dateFrom?: Date;
      dateTo?: Date;
    } = {},
  ): Promise<{
    tickets: SupportTicket[];
    total: number;
    page: number;
    totalPages: number;
  }> {
    const page = options.page || 1;
    const limit = options.limit || 20;
    const skip = (page - 1) * limit;

    const where: any = { merchantId };

    if (options.status) where.status = options.status;
    if (options.priority) where.priority = options.priority;
    if (options.type) where.type = options.type;
    if (options.categoryId) where.categoryId = options.categoryId;
    if (options.assignedTo) where.assignedTo = options.assignedTo;
    if (options.userId) where.userId = options.userId;
    if (options.driverId) where.driverId = options.driverId;

    if (options.search) {
      where.OR = [
        { ticketNumber: { contains: options.search } },
        { subject: { contains: options.search } },
        { description: { contains: options.search } },
      ];
    }

    if (options.dateFrom || options.dateTo) {
      where.createdAt = {};
      if (options.dateFrom) where.createdAt.gte = options.dateFrom;
      if (options.dateTo) where.createdAt.lte = options.dateTo;
    }

    const [tickets, total] = await Promise.all([
      this.prisma.supportTicket.findMany({
        where,
        include: {
          category: true,
          assignedAgent: {
            select: { id: true, firstName: true, lastName: true },
          },
        },
        skip,
        take: limit,
        orderBy: [
          { priority: 'desc' },
          { createdAt: 'desc' },
        ],
      }),
      this.prisma.supportTicket.count({ where }),
    ]);

    return {
      tickets: tickets.map(this.mapTicket),
      total,
      page,
      totalPages: Math.ceil(total / limit),
    };
  }

  /**
   * Tickets d'un utilisateur
   */
  async getUserTickets(
    userId: number,
    merchantId: number,
    status?: TicketStatus,
  ): Promise<SupportTicket[]> {
    const where: any = { userId, merchantId, type: 'user' };
    if (status) where.status = status;

    const tickets = await this.prisma.supportTicket.findMany({
      where,
      include: { category: true },
      orderBy: { createdAt: 'desc' },
    });

    return tickets.map(this.mapTicket);
  }

  /**
   * Tickets d'un chauffeur
   */
  async getDriverTickets(
    driverId: number,
    merchantId: number,
    status?: TicketStatus,
  ): Promise<SupportTicket[]> {
    const where: any = { driverId, merchantId, type: 'driver' };
    if (status) where.status = status;

    const tickets = await this.prisma.supportTicket.findMany({
      where,
      include: { category: true },
      orderBy: { createdAt: 'desc' },
    });

    return tickets.map(this.mapTicket);
  }

  // ==========================================================================
  // MESSAGES
  // ==========================================================================

  /**
   * Obtenir les messages d'un ticket
   */
  async getTicketMessages(
    ticketId: number,
    includeInternal = false,
  ): Promise<TicketMessage[]> {
    const where: any = { ticketId };
    if (!includeInternal) {
      where.isInternal = false;
    }

    const messages = await this.prisma.ticketMessage.findMany({
      where,
      orderBy: { createdAt: 'asc' },
    });

    return messages.map(this.mapMessage);
  }

  /**
   * Ajouter un message
   */
  async addMessage(
    ticketId: number,
    data: {
      senderId: number;
      senderType: 'user' | 'driver' | 'agent';
      senderName: string;
      message: string;
      attachments?: string[];
      isInternal?: boolean;
    },
  ): Promise<TicketMessage> {
    const ticket = await this.prisma.supportTicket.findUnique({
      where: { id: ticketId },
    });

    if (!ticket) {
      throw new NotFoundException('Ticket non trouvé');
    }

    const message = await this.prisma.ticketMessage.create({
      data: {
        ticketId,
        senderId: data.senderId,
        senderType: data.senderType,
        senderName: data.senderName,
        message: data.message,
        attachments: data.attachments || [],
        isInternal: data.isInternal || false,
      },
    });

    // Mettre à jour le ticket
    const updateData: any = { updatedAt: new Date() };

    // Premier réponse d'un agent
    if (data.senderType === 'agent' && !ticket.firstResponseAt) {
      updateData.firstResponseAt = new Date();
    }

    // Changer le statut si nécessaire
    if (data.senderType === 'agent' && ticket.status === 'open') {
      updateData.status = 'in_progress';
    } else if (['user', 'driver'].includes(data.senderType) && ticket.status === 'waiting_customer') {
      updateData.status = 'in_progress';
    }

    await this.prisma.supportTicket.update({
      where: { id: ticketId },
      data: updateData,
    });

    return this.mapMessage(message);
  }

  /**
   * Ajouter un message système
   */
  private async addSystemMessage(ticketId: number, message: string): Promise<void> {
    await this.prisma.ticketMessage.create({
      data: {
        ticketId,
        senderId: 0,
        senderType: 'system',
        senderName: 'Système',
        message,
        attachments: [],
        isInternal: true,
      },
    });
  }

  // ==========================================================================
  // CATEGORIES
  // ==========================================================================

  /**
   * Lister les catégories
   */
  async listCategories(merchantId: number, activeOnly = true): Promise<TicketCategory[]> {
    const where: any = { merchantId };
    if (activeOnly) where.active = true;

    const categories = await this.prisma.supportCategory.findMany({
      where,
      orderBy: [{ parentId: 'asc' }, { sortOrder: 'asc' }],
    });

    return categories.map(this.mapCategory);
  }

  /**
   * Créer une catégorie
   */
  async createCategory(
    merchantId: number,
    data: {
      name: string;
      slug: string;
      description?: string;
      icon?: string;
      parentId?: number;
      priority?: TicketPriority;
      autoAssignTo?: number;
      slaHours?: number;
      sortOrder?: number;
    },
  ): Promise<TicketCategory> {
    const category = await this.prisma.supportCategory.create({
      data: {
        merchantId,
        name: data.name,
        slug: data.slug,
        description: data.description,
        icon: data.icon,
        parentId: data.parentId,
        priority: data.priority || 'medium',
        autoAssignTo: data.autoAssignTo,
        slaHours: data.slaHours,
        sortOrder: data.sortOrder || 0,
        active: true,
      },
    });

    return this.mapCategory(category);
  }

  /**
   * Mettre à jour une catégorie
   */
  async updateCategory(
    id: number,
    merchantId: number,
    data: Partial<{
      name: string;
      description: string;
      icon: string;
      priority: TicketPriority;
      autoAssignTo: number | null;
      slaHours: number | null;
      sortOrder: number;
      active: boolean;
    }>,
  ): Promise<TicketCategory | null> {
    const category = await this.prisma.supportCategory.findFirst({
      where: { id, merchantId },
    });

    if (!category) {
      return null;
    }

    const updated = await this.prisma.supportCategory.update({
      where: { id },
      data,
    });

    return this.mapCategory(updated);
  }

  /**
   * Supprimer une catégorie
   */
  async deleteCategory(id: number, merchantId: number): Promise<boolean> {
    const category = await this.prisma.supportCategory.findFirst({
      where: { id, merchantId },
    });

    if (!category) {
      return false;
    }

    // Vérifier si des tickets utilisent cette catégorie
    const ticketCount = await this.prisma.supportTicket.count({
      where: { categoryId: id },
    });

    if (ticketCount > 0) {
      throw new BadRequestException(
        `Cette catégorie est utilisée par ${ticketCount} ticket(s). Désactivez-la plutôt.`,
      );
    }

    await this.prisma.supportCategory.delete({ where: { id } });
    return true;
  }

  // ==========================================================================
  // ACTIONS
  // ==========================================================================

  /**
   * Assigner un ticket
   */
  async assignTicket(
    id: number,
    merchantId: number,
    agentId: number | null,
  ): Promise<SupportTicket | null> {
    return this.updateTicket(id, merchantId, { assignedTo: agentId });
  }

  /**
   * Résoudre un ticket
   */
  async resolveTicket(
    id: number,
    merchantId: number,
    resolution?: string,
  ): Promise<SupportTicket | null> {
    const ticket = await this.getTicket(id, merchantId);
    if (!ticket) return null;

    if (resolution) {
      await this.addSystemMessage(id, `Résolution: ${resolution}`);
    }

    return this.updateTicket(id, merchantId, { status: 'resolved' });
  }

  /**
   * Fermer un ticket
   */
  async closeTicket(id: number, merchantId: number): Promise<SupportTicket | null> {
    return this.updateTicket(id, merchantId, { status: 'closed' });
  }

  /**
   * Rouvrir un ticket
   */
  async reopenTicket(id: number, merchantId: number): Promise<SupportTicket | null> {
    const ticket = await this.getTicket(id, merchantId);
    if (!ticket) return null;

    if (!['resolved', 'closed'].includes(ticket.status)) {
      throw new BadRequestException('Ce ticket ne peut pas être rouvert');
    }

    return this.updateTicket(id, merchantId, { status: 'open' });
  }

  /**
   * Noter un ticket (satisfaction client)
   */
  async rateTicket(
    id: number,
    merchantId: number,
    rating: number,
    comment?: string,
  ): Promise<SupportTicket | null> {
    const ticket = await this.prisma.supportTicket.findFirst({
      where: { id, merchantId },
    });

    if (!ticket) return null;

    if (!['resolved', 'closed'].includes(ticket.status)) {
      throw new BadRequestException('Seuls les tickets résolus peuvent être notés');
    }

    if (rating < 1 || rating > 5) {
      throw new BadRequestException('La note doit être entre 1 et 5');
    }

    const updated = await this.prisma.supportTicket.update({
      where: { id },
      data: {
        rating,
        ratingComment: comment,
      },
    });

    return this.mapTicket(updated);
  }

  // ==========================================================================
  // STATISTICS
  // ==========================================================================

  /**
   * Statistiques des tickets
   */
  async getStatistics(
    merchantId: number,
    dateFrom?: Date,
    dateTo?: Date,
  ): Promise<{
    total: number;
    byStatus: Record<TicketStatus, number>;
    byPriority: Record<TicketPriority, number>;
    byCategory: { categoryId: number; categoryName: string; count: number }[];
    avgResolutionTime: number; // en heures
    avgFirstResponseTime: number; // en heures
    avgRating: number;
    openTickets: number;
    resolvedToday: number;
  }> {
    const where: any = { merchantId };
    if (dateFrom || dateTo) {
      where.createdAt = {};
      if (dateFrom) where.createdAt.gte = dateFrom;
      if (dateTo) where.createdAt.lte = dateTo;
    }

    const [
      total,
      statusCounts,
      priorityCounts,
      categoryCounts,
      openTickets,
      resolvedToday,
      ratings,
      resolutionTimes,
      responseTimes,
    ] = await Promise.all([
      this.prisma.supportTicket.count({ where }),
      this.prisma.supportTicket.groupBy({
        by: ['status'],
        where,
        _count: { status: true },
      }),
      this.prisma.supportTicket.groupBy({
        by: ['priority'],
        where,
        _count: { priority: true },
      }),
      this.prisma.supportTicket.groupBy({
        by: ['categoryId'],
        where: { ...where, categoryId: { not: null } },
        _count: { categoryId: true },
      }),
      this.prisma.supportTicket.count({
        where: { ...where, status: { in: ['open', 'in_progress', 'waiting_customer'] } },
      }),
      this.prisma.supportTicket.count({
        where: {
          ...where,
          resolvedAt: {
            gte: new Date(new Date().setHours(0, 0, 0, 0)),
          },
        },
      }),
      this.prisma.supportTicket.aggregate({
        where: { ...where, rating: { not: null } },
        _avg: { rating: true },
      }),
      this.prisma.supportTicket.findMany({
        where: { ...where, resolvedAt: { not: null } },
        select: { createdAt: true, resolvedAt: true },
      }),
      this.prisma.supportTicket.findMany({
        where: { ...where, firstResponseAt: { not: null } },
        select: { createdAt: true, firstResponseAt: true },
      }),
    ]);

    // Récupérer les noms des catégories
    const categories = await this.prisma.supportCategory.findMany({
      where: { merchantId },
      select: { id: true, name: true },
    });
    const categoryMap = new Map(categories.map((c) => [c.id, c.name]));

    // Calculer les temps moyens
    let avgResolutionTime = 0;
    if (resolutionTimes.length > 0) {
      const totalHours = resolutionTimes.reduce((sum, t) => {
        const diff = new Date(t.resolvedAt!).getTime() - new Date(t.createdAt).getTime();
        return sum + diff / (1000 * 60 * 60);
      }, 0);
      avgResolutionTime = Math.round(totalHours / resolutionTimes.length);
    }

    let avgFirstResponseTime = 0;
    if (responseTimes.length > 0) {
      const totalHours = responseTimes.reduce((sum, t) => {
        const diff = new Date(t.firstResponseAt!).getTime() - new Date(t.createdAt).getTime();
        return sum + diff / (1000 * 60 * 60);
      }, 0);
      avgFirstResponseTime = Math.round((totalHours / responseTimes.length) * 10) / 10;
    }

    // Formater les résultats
    const byStatus: Record<string, number> = {
      open: 0,
      in_progress: 0,
      waiting_customer: 0,
      resolved: 0,
      closed: 0,
    };
    statusCounts.forEach((s) => {
      byStatus[s.status] = s._count.status;
    });

    const byPriority: Record<string, number> = {
      low: 0,
      medium: 0,
      high: 0,
      urgent: 0,
    };
    priorityCounts.forEach((p) => {
      byPriority[p.priority] = p._count.priority;
    });

    const byCategory = categoryCounts.map((c) => ({
      categoryId: c.categoryId!,
      categoryName: categoryMap.get(c.categoryId!) || 'Unknown',
      count: c._count.categoryId,
    }));

    return {
      total,
      byStatus: byStatus as Record<TicketStatus, number>,
      byPriority: byPriority as Record<TicketPriority, number>,
      byCategory,
      avgResolutionTime,
      avgFirstResponseTime,
      avgRating: ratings._avg.rating || 0,
      openTickets,
      resolvedToday,
    };
  }

  // ==========================================================================
  // HELPERS
  // ==========================================================================

  /**
   * Générer un numéro de ticket unique
   */
  private async generateTicketNumber(merchantId: number): Promise<string> {
    const today = new Date();
    const prefix = `TKT-${today.getFullYear()}${(today.getMonth() + 1).toString().padStart(2, '0')}`;

    const lastTicket = await this.prisma.supportTicket.findFirst({
      where: {
        merchantId,
        ticketNumber: { startsWith: prefix },
      },
      orderBy: { ticketNumber: 'desc' },
    });

    let sequence = 1;
    if (lastTicket) {
      const lastSequence = parseInt(lastTicket.ticketNumber.split('-').pop() || '0');
      sequence = lastSequence + 1;
    }

    return `${prefix}-${sequence.toString().padStart(5, '0')}`;
  }

  /**
   * Obtenir le nom d'un agent
   */
  private async getAgentName(agentId: number): Promise<string> {
    const agent = await this.prisma.admin.findUnique({
      where: { id: agentId },
      select: { firstName: true, lastName: true },
    });

    return agent ? `${agent.firstName} ${agent.lastName}` : 'Agent inconnu';
  }

  private mapTicket(ticket: any): SupportTicket {
    return {
      id: ticket.id,
      ticketNumber: ticket.ticketNumber,
      type: ticket.type,
      userId: ticket.userId,
      driverId: ticket.driverId,
      bookingId: ticket.bookingId,
      categoryId: ticket.categoryId,
      subject: ticket.subject,
      description: ticket.description,
      status: ticket.status,
      priority: ticket.priority,
      assignedTo: ticket.assignedTo,
      tags: ticket.tags || [],
      attachments: ticket.attachments || [],
      rating: ticket.rating,
      ratingComment: ticket.ratingComment,
      resolvedAt: ticket.resolvedAt,
      closedAt: ticket.closedAt,
      firstResponseAt: ticket.firstResponseAt,
      merchantId: ticket.merchantId,
      createdAt: ticket.createdAt,
      updatedAt: ticket.updatedAt,
    };
  }

  private mapMessage(message: any): TicketMessage {
    return {
      id: message.id,
      ticketId: message.ticketId,
      senderId: message.senderId,
      senderType: message.senderType,
      senderName: message.senderName,
      message: message.message,
      attachments: message.attachments || [],
      isInternal: message.isInternal,
      createdAt: message.createdAt,
    };
  }

  private mapCategory(category: any): TicketCategory {
    return {
      id: category.id,
      name: category.name,
      slug: category.slug,
      description: category.description,
      icon: category.icon,
      parentId: category.parentId,
      priority: category.priority,
      autoAssignTo: category.autoAssignTo,
      slaHours: category.slaHours,
      active: category.active,
      sortOrder: category.sortOrder,
    };
  }
}
