import {
  Controller,
  Get,
  Post,
  Put,
  Delete,
  Body,
  Param,
  Query,
  UseGuards,
  ParseIntPipe,
  DefaultValuePipe,
} from '@nestjs/common';
import {
  SupportService,
  TicketStatus,
  TicketPriority,
  TicketType,
  CreateTicketDto,
  UpdateTicketDto,
} from './support.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantId } from '../../common/decorators/merchant.decorator';
import { CurrentUser } from '../../common/decorators/current-user.decorator';

// ============================================================================
// DTOs
// ============================================================================

class CreateCategoryDto {
  name: string;
  slug: string;
  description?: string;
  icon?: string;
  parentId?: number;
  priority?: TicketPriority;
  autoAssignTo?: number;
  slaHours?: number;
  sortOrder?: number;
}

class AddMessageDto {
  message: string;
  attachments?: string[];
  isInternal?: boolean;
}

class RateTicketDto {
  rating: number;
  comment?: string;
}

// ============================================================================
// CONTROLLER
// ============================================================================

@Controller('support')
@UseGuards(JwtAuthGuard)
export class SupportController {
  constructor(private readonly supportService: SupportService) {}

  // ==========================================================================
  // CATEGORIES (PUBLIC)
  // ==========================================================================

  /**
   * Lister les catégories de support
   */
  @Get('categories')
  async listCategories(@MerchantId() merchantId: number) {
    return this.supportService.listCategories(merchantId);
  }

  // ==========================================================================
  // USER TICKETS
  // ==========================================================================

  /**
   * Mes tickets (utilisateur)
   */
  @Get('my-tickets')
  async getMyTickets(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
    @Query('status') status?: TicketStatus,
  ) {
    if (user.driverId) {
      return this.supportService.getDriverTickets(user.driverId, merchantId, status);
    }
    return this.supportService.getUserTickets(user.id, merchantId, status);
  }

  /**
   * Créer un ticket (utilisateur/chauffeur)
   */
  @Post('tickets')
  async createTicket(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
    @Body() dto: CreateTicketDto,
  ) {
    // Déterminer le type automatiquement
    const createData: CreateTicketDto = {
      ...dto,
      type: user.driverId ? 'driver' : 'user',
      userId: user.driverId ? undefined : user.id,
      driverId: user.driverId || undefined,
    };

    const ticket = await this.supportService.createTicket(merchantId, createData);

    return {
      success: true,
      ticket,
      message: `Ticket #${ticket.ticketNumber} créé avec succès`,
    };
  }

  /**
   * Voir un de mes tickets
   */
  @Get('my-tickets/:id')
  async getMyTicket(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const ticket = await this.supportService.getTicket(id, merchantId);

    if (!ticket) {
      return { error: 'Ticket non trouvé' };
    }

    // Vérifier que le ticket appartient à l'utilisateur
    if (user.driverId) {
      if (ticket.driverId !== user.driverId) {
        return { error: 'Ticket non trouvé' };
      }
    } else {
      if (ticket.userId !== user.id) {
        return { error: 'Ticket non trouvé' };
      }
    }

    const messages = await this.supportService.getTicketMessages(id, false);

    return {
      ...ticket,
      messages,
    };
  }

  /**
   * Ajouter un message à mon ticket
   */
  @Post('my-tickets/:id/messages')
  async addMessageToMyTicket(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: AddMessageDto,
  ) {
    const ticket = await this.supportService.getTicket(id, merchantId);

    if (!ticket) {
      return { error: 'Ticket non trouvé' };
    }

    // Vérifier que le ticket appartient à l'utilisateur
    const isOwner = user.driverId
      ? ticket.driverId === user.driverId
      : ticket.userId === user.id;

    if (!isOwner) {
      return { error: 'Ticket non trouvé' };
    }

    const message = await this.supportService.addMessage(id, {
      senderId: user.driverId || user.id,
      senderType: user.driverId ? 'driver' : 'user',
      senderName: `${user.firstName} ${user.lastName}`,
      message: dto.message,
      attachments: dto.attachments,
    });

    return {
      success: true,
      message,
    };
  }

  /**
   * Noter un ticket résolu
   */
  @Post('my-tickets/:id/rate')
  async rateMyTicket(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: RateTicketDto,
  ) {
    const ticket = await this.supportService.getTicket(id, merchantId);

    if (!ticket) {
      return { error: 'Ticket non trouvé' };
    }

    // Vérifier que le ticket appartient à l'utilisateur
    const isOwner = user.driverId
      ? ticket.driverId === user.driverId
      : ticket.userId === user.id;

    if (!isOwner) {
      return { error: 'Ticket non trouvé' };
    }

    const updated = await this.supportService.rateTicket(
      id,
      merchantId,
      dto.rating,
      dto.comment,
    );

    return {
      success: true,
      ticket: updated,
      message: 'Merci pour votre retour!',
    };
  }

  // ==========================================================================
  // ADMIN - TICKETS
  // ==========================================================================

  /**
   * Lister tous les tickets (admin)
   */
  @Get('admin/tickets')
  async listTickets(
    @MerchantId() merchantId: number,
    @Query('page', new DefaultValuePipe(1), ParseIntPipe) page: number,
    @Query('limit', new DefaultValuePipe(20), ParseIntPipe) limit: number,
    @Query('status') status?: TicketStatus,
    @Query('priority') priority?: TicketPriority,
    @Query('type') type?: TicketType,
    @Query('categoryId') categoryId?: string,
    @Query('assignedTo') assignedTo?: string,
    @Query('search') search?: string,
    @Query('dateFrom') dateFrom?: string,
    @Query('dateTo') dateTo?: string,
  ) {
    return this.supportService.listTickets(merchantId, {
      page,
      limit,
      status,
      priority,
      type,
      categoryId: categoryId ? parseInt(categoryId) : undefined,
      assignedTo: assignedTo ? parseInt(assignedTo) : undefined,
      search,
      dateFrom: dateFrom ? new Date(dateFrom) : undefined,
      dateTo: dateTo ? new Date(dateTo) : undefined,
    });
  }

  /**
   * Voir un ticket (admin)
   */
  @Get('admin/tickets/:id')
  async getTicket(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const ticket = await this.supportService.getTicket(id, merchantId);

    if (!ticket) {
      return { error: 'Ticket non trouvé' };
    }

    const messages = await this.supportService.getTicketMessages(id, true);

    return {
      ...ticket,
      messages,
    };
  }

  /**
   * Mettre à jour un ticket (admin)
   */
  @Put('admin/tickets/:id')
  async updateTicket(
    @CurrentUser() admin: any,
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: UpdateTicketDto,
  ) {
    const ticket = await this.supportService.updateTicket(id, merchantId, dto, admin.id);

    if (!ticket) {
      return { error: 'Ticket non trouvé' };
    }

    return {
      success: true,
      ticket,
    };
  }

  /**
   * Ajouter un message (admin)
   */
  @Post('admin/tickets/:id/messages')
  async addAdminMessage(
    @CurrentUser() admin: any,
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: AddMessageDto,
  ) {
    const ticket = await this.supportService.getTicket(id, merchantId);

    if (!ticket) {
      return { error: 'Ticket non trouvé' };
    }

    const message = await this.supportService.addMessage(id, {
      senderId: admin.id,
      senderType: 'agent',
      senderName: `${admin.firstName} ${admin.lastName}`,
      message: dto.message,
      attachments: dto.attachments,
      isInternal: dto.isInternal,
    });

    return {
      success: true,
      message,
    };
  }

  /**
   * Assigner un ticket
   */
  @Post('admin/tickets/:id/assign')
  async assignTicket(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() body: { agentId: number | null },
  ) {
    const ticket = await this.supportService.assignTicket(id, merchantId, body.agentId);

    if (!ticket) {
      return { error: 'Ticket non trouvé' };
    }

    return {
      success: true,
      ticket,
      message: body.agentId ? 'Ticket assigné' : 'Ticket désassigné',
    };
  }

  /**
   * Résoudre un ticket
   */
  @Post('admin/tickets/:id/resolve')
  async resolveTicket(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() body: { resolution?: string },
  ) {
    const ticket = await this.supportService.resolveTicket(id, merchantId, body.resolution);

    if (!ticket) {
      return { error: 'Ticket non trouvé' };
    }

    return {
      success: true,
      ticket,
      message: 'Ticket résolu',
    };
  }

  /**
   * Fermer un ticket
   */
  @Post('admin/tickets/:id/close')
  async closeTicket(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const ticket = await this.supportService.closeTicket(id, merchantId);

    if (!ticket) {
      return { error: 'Ticket non trouvé' };
    }

    return {
      success: true,
      ticket,
      message: 'Ticket fermé',
    };
  }

  /**
   * Rouvrir un ticket
   */
  @Post('admin/tickets/:id/reopen')
  async reopenTicket(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const ticket = await this.supportService.reopenTicket(id, merchantId);

    if (!ticket) {
      return { error: 'Ticket non trouvé' };
    }

    return {
      success: true,
      ticket,
      message: 'Ticket rouvert',
    };
  }

  // ==========================================================================
  // ADMIN - CATEGORIES
  // ==========================================================================

  /**
   * Lister toutes les catégories (admin)
   */
  @Get('admin/categories')
  async listAllCategories(@MerchantId() merchantId: number) {
    return this.supportService.listCategories(merchantId, false);
  }

  /**
   * Créer une catégorie (admin)
   */
  @Post('admin/categories')
  async createCategory(
    @MerchantId() merchantId: number,
    @Body() dto: CreateCategoryDto,
  ) {
    const category = await this.supportService.createCategory(merchantId, dto);
    return {
      success: true,
      category,
    };
  }

  /**
   * Mettre à jour une catégorie (admin)
   */
  @Put('admin/categories/:id')
  async updateCategory(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: Partial<CreateCategoryDto & { active: boolean }>,
  ) {
    const category = await this.supportService.updateCategory(id, merchantId, dto);

    if (!category) {
      return { error: 'Catégorie non trouvée' };
    }

    return {
      success: true,
      category,
    };
  }

  /**
   * Supprimer une catégorie (admin)
   */
  @Delete('admin/categories/:id')
  async deleteCategory(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const deleted = await this.supportService.deleteCategory(id, merchantId);
    return {
      success: deleted,
      message: deleted ? 'Catégorie supprimée' : 'Catégorie non trouvée',
    };
  }

  // ==========================================================================
  // STATISTICS
  // ==========================================================================

  /**
   * Statistiques des tickets (admin)
   */
  @Get('admin/stats')
  async getStatistics(
    @MerchantId() merchantId: number,
    @Query('dateFrom') dateFrom?: string,
    @Query('dateTo') dateTo?: string,
  ) {
    return this.supportService.getStatistics(
      merchantId,
      dateFrom ? new Date(dateFrom) : undefined,
      dateTo ? new Date(dateTo) : undefined,
    );
  }

  /**
   * Dashboard rapide
   */
  @Get('admin/dashboard')
  async getDashboard(@MerchantId() merchantId: number) {
    const stats = await this.supportService.getStatistics(merchantId);

    // Tickets urgents non assignés
    const urgentUnassigned = await this.supportService.listTickets(merchantId, {
      priority: 'urgent',
      status: 'open',
      limit: 10,
    });

    // Tickets les plus anciens ouverts
    const oldestOpen = await this.supportService.listTickets(merchantId, {
      status: 'open',
      limit: 5,
    });

    return {
      stats: {
        openTickets: stats.openTickets,
        resolvedToday: stats.resolvedToday,
        avgResponseTime: stats.avgFirstResponseTime,
        avgRating: stats.avgRating,
      },
      urgentUnassigned: urgentUnassigned.tickets,
      oldestOpen: oldestOpen.tickets,
    };
  }
}
