import {
  Controller,
  Post,
  Get,
  Delete,
  Body,
  Param,
  Query,
  UseGuards,
  UseInterceptors,
  UploadedFile,
  UploadedFiles,
  ParseFilePipe,
  MaxFileSizeValidator,
  FileTypeValidator,
} from '@nestjs/common';
import { FileInterceptor, FilesInterceptor } from '@nestjs/platform-express';
import { StorageService } from './storage.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantGuard } from '../../common/guards/merchant.guard';
import { CurrentMerchant } from '../../common/decorators/merchant.decorator';
import { CurrentUser } from '../../common/decorators/user.decorator';

// ============================================================================
// DTOs
// ============================================================================

class UploadFromUrlDto {
  url: string;
  folder?: string;
  filename?: string;
  provider?: string;
}

// ============================================================================
// CONTROLLER
// ============================================================================

@Controller('storage')
@UseGuards(JwtAuthGuard)
export class StorageController {
  constructor(private readonly storageService: StorageService) {}

  // ============================================================================
  // FILE UPLOAD
  // ============================================================================

  /**
   * Upload single file
   */
  @Post('upload')
  @UseInterceptors(FileInterceptor('file'))
  async uploadFile(
    @UploadedFile(
      new ParseFilePipe({
        validators: [
          new MaxFileSizeValidator({ maxSize: 10 * 1024 * 1024 }), // 10MB
        ],
      }),
    )
    file: Express.Multer.File,
    @CurrentUser() userId: number,
    @CurrentMerchant() merchantId: number,
    @Query('folder') folder?: string,
    @Query('provider') provider?: string,
  ) {
    return this.storageService.uploadFile(
      file.buffer,
      file.originalname,
      file.mimetype,
      {
        userId,
        merchantId,
        folder,
        provider,
      },
    );
  }

  /**
   * Upload multiple files
   */
  @Post('upload-multiple')
  @UseInterceptors(FilesInterceptor('files', 10))
  async uploadFiles(
    @UploadedFiles(
      new ParseFilePipe({
        validators: [
          new MaxFileSizeValidator({ maxSize: 10 * 1024 * 1024 }),
        ],
      }),
    )
    files: Express.Multer.File[],
    @CurrentUser() userId: number,
    @CurrentMerchant() merchantId: number,
    @Query('folder') folder?: string,
    @Query('provider') provider?: string,
  ) {
    return this.storageService.uploadFiles(
      files.map(f => ({
        buffer: f.buffer,
        originalName: f.originalname,
        mimeType: f.mimetype,
      })),
      {
        userId,
        merchantId,
        folder,
        provider,
      },
    );
  }

  /**
   * Upload image with optimization
   */
  @Post('upload-image')
  @UseInterceptors(FileInterceptor('file'))
  async uploadImage(
    @UploadedFile(
      new ParseFilePipe({
        validators: [
          new MaxFileSizeValidator({ maxSize: 10 * 1024 * 1024 }),
          new FileTypeValidator({ fileType: /image\/(jpeg|png|gif|webp)/ }),
        ],
      }),
    )
    file: Express.Multer.File,
    @CurrentUser() userId: number,
    @CurrentMerchant() merchantId: number,
    @Query('folder') folder?: string,
    @Query('width') width?: string,
    @Query('height') height?: string,
    @Query('quality') quality?: string,
  ) {
    return this.storageService.uploadFile(
      file.buffer,
      file.originalname,
      file.mimetype,
      {
        userId,
        merchantId,
        folder,
        resize: width || height ? {
          width: width ? parseInt(width) : undefined,
          height: height ? parseInt(height) : undefined,
          fit: 'inside',
        } : undefined,
        compress: true,
        quality: quality ? parseInt(quality) : 80,
      },
    );
  }

  /**
   * Upload from URL
   */
  @Post('upload-url')
  async uploadFromUrl(
    @Body() dto: UploadFromUrlDto,
    @CurrentUser() userId: number,
    @CurrentMerchant() merchantId: number,
  ) {
    return this.storageService.uploadFromUrl(dto.url, {
      userId,
      merchantId,
      folder: dto.folder,
      filename: dto.filename,
      provider: dto.provider,
    });
  }

  // ============================================================================
  // PROFILE & DOCUMENTS
  // ============================================================================

  /**
   * Upload profile picture
   */
  @Post('profile-picture')
  @UseInterceptors(FileInterceptor('file'))
  async uploadProfilePicture(
    @UploadedFile(
      new ParseFilePipe({
        validators: [
          new MaxFileSizeValidator({ maxSize: 5 * 1024 * 1024 }),
          new FileTypeValidator({ fileType: /image\/(jpeg|png|webp)/ }),
        ],
      }),
    )
    file: Express.Multer.File,
    @CurrentUser() userId: number,
  ) {
    return this.storageService.uploadProfilePicture(
      file.buffer,
      file.originalname,
      file.mimetype,
      userId,
      'user',
    );
  }

  /**
   * Upload document
   */
  @Post('document')
  @UseInterceptors(FileInterceptor('file'))
  async uploadDocument(
    @UploadedFile(
      new ParseFilePipe({
        validators: [
          new MaxFileSizeValidator({ maxSize: 20 * 1024 * 1024 }),
          new FileTypeValidator({ fileType: /image\/(jpeg|png)|application\/pdf/ }),
        ],
      }),
    )
    file: Express.Multer.File,
    @CurrentUser() userId: number,
    @CurrentMerchant() merchantId: number,
    @Query('entityType') entityType?: string,
    @Query('entityId') entityId?: string,
  ) {
    return this.storageService.uploadFile(
      file.buffer,
      file.originalname,
      file.mimetype,
      {
        userId,
        merchantId,
        folder: 'documents',
        entityType,
        entityId: entityId ? parseInt(entityId) : undefined,
      },
    );
  }

  // ============================================================================
  // FILE MANAGEMENT
  // ============================================================================

  /**
   * Delete file
   */
  @Delete(':id')
  async deleteFile(@Param('id') id: string) {
    const deleted = await this.storageService.deleteFileById(parseInt(id));
    return { success: deleted };
  }

  /**
   * Get signed URL for private file
   */
  @Get('signed-url/:key')
  async getSignedUrl(
    @Param('key') key: string,
    @Query('provider') provider?: string,
    @Query('expiresIn') expiresIn?: string,
  ) {
    const url = await this.storageService.getSignedUrl(
      key,
      provider,
      expiresIn ? parseInt(expiresIn) : undefined,
    );

    if (!url) {
      return { success: false, error: 'Could not generate signed URL' };
    }

    return { success: true, url };
  }

  /**
   * Get files for entity
   */
  @Get('entity/:type/:id')
  async getFilesForEntity(
    @Param('type') entityType: string,
    @Param('id') entityId: string,
  ) {
    return this.storageService.getFilesForEntity(entityType, parseInt(entityId));
  }

  /**
   * Get user files
   */
  @Get('my-files')
  async getMyFiles(
    @CurrentUser() userId: number,
    @Query('mimeType') mimeType?: string,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
  ) {
    return this.storageService.getUserFiles(userId, {
      mimeType,
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 50,
    });
  }

  /**
   * Get available providers
   */
  @Get('providers')
  @UseGuards(MerchantGuard)
  async getProviders() {
    return this.storageService.getAvailableProviders();
  }
}
