/**
 * Storage Provider Interface
 */

export interface UploadOptions {
  folder?: string;
  filename?: string;
  contentType?: string;
  isPublic?: boolean;
  metadata?: Record<string, string>;
}

export interface UploadResult {
  success: boolean;
  url?: string;
  key?: string;
  bucket?: string;
  size?: number;
  contentType?: string;
  error?: string;
}

export interface DeleteResult {
  success: boolean;
  error?: string;
}

export interface SignedUrlOptions {
  expiresIn?: number; // seconds
  contentType?: string;
  action?: 'read' | 'write';
}

export interface StorageProvider {
  readonly name: string;
  readonly displayName: string;

  /**
   * Upload a file
   */
  upload(
    buffer: Buffer,
    filename: string,
    options?: UploadOptions,
  ): Promise<UploadResult>;

  /**
   * Upload from URL
   */
  uploadFromUrl?(
    url: string,
    filename: string,
    options?: UploadOptions,
  ): Promise<UploadResult>;

  /**
   * Delete a file
   */
  delete(key: string): Promise<DeleteResult>;

  /**
   * Get signed URL for temporary access
   */
  getSignedUrl?(
    key: string,
    options?: SignedUrlOptions,
  ): Promise<string>;

  /**
   * Check if file exists
   */
  exists?(key: string): Promise<boolean>;

  /**
   * Get file metadata
   */
  getMetadata?(key: string): Promise<Record<string, any>>;

  /**
   * Copy file
   */
  copy?(sourceKey: string, destKey: string): Promise<UploadResult>;
}
