import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import * as fs from 'fs/promises';
import * as path from 'path';
import * as crypto from 'crypto';
import axios from 'axios';
import {
  StorageProvider,
  UploadOptions,
  UploadResult,
  DeleteResult,
} from './storage.interface';

@Injectable()
export class LocalStorageProvider implements StorageProvider {
  readonly name = 'local';
  readonly displayName = 'Local Storage';

  private readonly logger = new Logger(LocalStorageProvider.name);
  private basePath: string;
  private baseUrl: string;

  constructor(private configService: ConfigService) {
    this.basePath = this.configService.get<string>('STORAGE_LOCAL_PATH') || './uploads';
    this.baseUrl = this.configService.get<string>('STORAGE_LOCAL_URL') || '/uploads';
  }

  async upload(
    buffer: Buffer,
    filename: string,
    options?: UploadOptions,
  ): Promise<UploadResult> {
    try {
      const folder = options?.folder || 'general';
      const dirPath = path.join(this.basePath, folder);

      // Create directory if it doesn't exist
      await fs.mkdir(dirPath, { recursive: true });

      // Generate unique filename if not provided
      const ext = path.extname(filename);
      const name = options?.filename || `${Date.now()}_${crypto.randomBytes(8).toString('hex')}${ext}`;
      const filePath = path.join(dirPath, name);

      // Write file
      await fs.writeFile(filePath, buffer);

      const key = `${folder}/${name}`;
      const url = `${this.baseUrl}/${key}`;

      return {
        success: true,
        url,
        key,
        size: buffer.length,
        contentType: options?.contentType,
      };
    } catch (error) {
      this.logger.error(`Local upload error: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  async uploadFromUrl(
    url: string,
    filename: string,
    options?: UploadOptions,
  ): Promise<UploadResult> {
    try {
      const response = await axios.get(url, { responseType: 'arraybuffer' });
      const buffer = Buffer.from(response.data);
      const contentType = response.headers['content-type'] || options?.contentType;

      return this.upload(buffer, filename, { ...options, contentType });
    } catch (error) {
      this.logger.error(`Local uploadFromUrl error: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  async delete(key: string): Promise<DeleteResult> {
    try {
      const filePath = path.join(this.basePath, key);
      await fs.unlink(filePath);
      return { success: true };
    } catch (error) {
      if ((error as any).code === 'ENOENT') {
        return { success: true }; // File doesn't exist, consider it deleted
      }
      this.logger.error(`Local delete error: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  async exists(key: string): Promise<boolean> {
    try {
      const filePath = path.join(this.basePath, key);
      await fs.access(filePath);
      return true;
    } catch {
      return false;
    }
  }

  async getMetadata(key: string): Promise<Record<string, any>> {
    try {
      const filePath = path.join(this.basePath, key);
      const stats = await fs.stat(filePath);

      return {
        size: stats.size,
        lastModified: stats.mtime,
        created: stats.birthtime,
      };
    } catch (error) {
      this.logger.error(`Local getMetadata error: ${error.message}`);
      throw error;
    }
  }

  async copy(sourceKey: string, destKey: string): Promise<UploadResult> {
    try {
      const sourcePath = path.join(this.basePath, sourceKey);
      const destPath = path.join(this.basePath, destKey);

      // Ensure destination directory exists
      await fs.mkdir(path.dirname(destPath), { recursive: true });

      // Copy file
      await fs.copyFile(sourcePath, destPath);

      const url = `${this.baseUrl}/${destKey}`;

      return {
        success: true,
        url,
        key: destKey,
      };
    } catch (error) {
      this.logger.error(`Local copy error: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  /**
   * Read file content
   */
  async read(key: string): Promise<Buffer> {
    const filePath = path.join(this.basePath, key);
    return fs.readFile(filePath);
  }

  /**
   * List files in a folder
   */
  async list(folder: string): Promise<string[]> {
    try {
      const dirPath = path.join(this.basePath, folder);
      const files = await fs.readdir(dirPath);
      return files.map(f => `${folder}/${f}`);
    } catch {
      return [];
    }
  }
}
