import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { v2 as cloudinary, UploadApiResponse } from 'cloudinary';
import {
  StorageProvider,
  UploadOptions,
  UploadResult,
  DeleteResult,
} from './storage.interface';

interface CloudinaryUploadOptions extends UploadOptions {
  transformation?: {
    width?: number;
    height?: number;
    crop?: string;
    quality?: string | number;
    format?: string;
  };
  eager?: Array<{
    width?: number;
    height?: number;
    crop?: string;
  }>;
}

@Injectable()
export class CloudinaryProvider implements StorageProvider {
  readonly name = 'cloudinary';
  readonly displayName = 'Cloudinary';

  private readonly logger = new Logger(CloudinaryProvider.name);

  constructor(private configService: ConfigService) {
    cloudinary.config({
      cloud_name: this.configService.get<string>('CLOUDINARY_CLOUD_NAME'),
      api_key: this.configService.get<string>('CLOUDINARY_API_KEY'),
      api_secret: this.configService.get<string>('CLOUDINARY_API_SECRET'),
    });
  }

  async upload(
    buffer: Buffer,
    filename: string,
    options?: CloudinaryUploadOptions,
  ): Promise<UploadResult> {
    try {
      const result = await new Promise<UploadApiResponse>((resolve, reject) => {
        const uploadStream = cloudinary.uploader.upload_stream(
          {
            folder: options?.folder || 'monkapi',
            public_id: options?.filename || filename.replace(/\.[^/.]+$/, ''),
            resource_type: 'auto',
            transformation: options?.transformation,
            eager: options?.eager,
          },
          (error, result) => {
            if (error) reject(error);
            else resolve(result);
          },
        );

        uploadStream.end(buffer);
      });

      return {
        success: true,
        url: result.secure_url,
        key: result.public_id,
        size: result.bytes,
        contentType: result.format,
      };
    } catch (error) {
      this.logger.error(`Cloudinary upload error: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  async uploadFromUrl(
    url: string,
    filename: string,
    options?: CloudinaryUploadOptions,
  ): Promise<UploadResult> {
    try {
      const result = await cloudinary.uploader.upload(url, {
        folder: options?.folder || 'monkapi',
        public_id: options?.filename || filename.replace(/\.[^/.]+$/, ''),
        resource_type: 'auto',
        transformation: options?.transformation,
        eager: options?.eager,
      });

      return {
        success: true,
        url: result.secure_url,
        key: result.public_id,
        size: result.bytes,
        contentType: result.format,
      };
    } catch (error) {
      this.logger.error(`Cloudinary uploadFromUrl error: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  async delete(publicId: string): Promise<DeleteResult> {
    try {
      await cloudinary.uploader.destroy(publicId);
      return { success: true };
    } catch (error) {
      this.logger.error(`Cloudinary delete error: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  async exists(publicId: string): Promise<boolean> {
    try {
      await cloudinary.api.resource(publicId);
      return true;
    } catch {
      return false;
    }
  }

  async getMetadata(publicId: string): Promise<Record<string, any>> {
    try {
      const result = await cloudinary.api.resource(publicId);
      return {
        format: result.format,
        width: result.width,
        height: result.height,
        bytes: result.bytes,
        created_at: result.created_at,
        url: result.secure_url,
      };
    } catch (error) {
      this.logger.error(`Cloudinary getMetadata error: ${error.message}`);
      throw error;
    }
  }

  /**
   * Generate URL with transformations
   */
  getTransformedUrl(
    publicId: string,
    transformations: {
      width?: number;
      height?: number;
      crop?: string;
      quality?: string | number;
      format?: string;
      gravity?: string;
      effect?: string;
    },
  ): string {
    return cloudinary.url(publicId, {
      transformation: [transformations],
      secure: true,
    });
  }

  /**
   * Generate thumbnail URL
   */
  getThumbnailUrl(publicId: string, size: number = 150): string {
    return this.getTransformedUrl(publicId, {
      width: size,
      height: size,
      crop: 'fill',
      gravity: 'auto',
    });
  }

  /**
   * Generate optimized URL for web
   */
  getOptimizedUrl(publicId: string, maxWidth: number = 1200): string {
    return this.getTransformedUrl(publicId, {
      width: maxWidth,
      crop: 'limit',
      quality: 'auto',
      format: 'auto',
    });
  }

  /**
   * Upload and generate multiple sizes
   */
  async uploadWithVariants(
    buffer: Buffer,
    filename: string,
    options?: UploadOptions,
  ): Promise<{
    original: UploadResult;
    thumbnail?: string;
    medium?: string;
    large?: string;
  }> {
    const result = await this.upload(buffer, filename, {
      ...options,
      eager: [
        { width: 150, height: 150, crop: 'fill' },
        { width: 600, height: 600, crop: 'limit' },
        { width: 1200, height: 1200, crop: 'limit' },
      ],
    } as CloudinaryUploadOptions);

    if (!result.success || !result.key) {
      return { original: result };
    }

    return {
      original: result,
      thumbnail: this.getThumbnailUrl(result.key, 150),
      medium: this.getTransformedUrl(result.key, { width: 600, crop: 'limit' }),
      large: this.getTransformedUrl(result.key, { width: 1200, crop: 'limit' }),
    };
  }
}
