import {
  Controller,
  Post,
  Get,
  Body,
  Param,
  Query,
  UseGuards,
} from '@nestjs/common';
import { SmsService } from './sms.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantGuard } from '../../common/guards/merchant.guard';
import { CurrentMerchant } from '../../common/decorators/merchant.decorator';

// ============================================================================
// DTOs
// ============================================================================

class SendSmsDto {
  to: string;
  message: string;
  provider?: string;
  from?: string;
  reference?: string;
}

class SendBulkSmsDto {
  recipients: string[];
  message: string;
  provider?: string;
  from?: string;
  reference?: string;
}

class SendOtpDto {
  to: string;
  otp: string;
  provider?: string;
}

class SendSmartSmsDto {
  to: string;
  message: string;
  countryCode: string;
  provider?: string;
  from?: string;
}

// ============================================================================
// CONTROLLER
// ============================================================================

@Controller('sms')
@UseGuards(JwtAuthGuard, MerchantGuard)
export class SmsController {
  constructor(private readonly smsService: SmsService) {}

  // ============================================================================
  // SEND SMS
  // ============================================================================

  /**
   * Send single SMS
   */
  @Post('send')
  async sendSms(@Body() dto: SendSmsDto) {
    return this.smsService.send(dto.to, dto.message, {
      provider: dto.provider,
      from: dto.from,
      reference: dto.reference,
    });
  }

  /**
   * Send SMS with automatic provider selection
   */
  @Post('send-smart')
  async sendSmartSms(@Body() dto: SendSmartSmsDto) {
    return this.smsService.sendSmart(dto.to, dto.message, dto.countryCode, {
      provider: dto.provider,
      from: dto.from,
    });
  }

  /**
   * Send bulk SMS
   */
  @Post('send-bulk')
  async sendBulkSms(@Body() dto: SendBulkSmsDto) {
    return this.smsService.sendBulk(dto.recipients, dto.message, {
      provider: dto.provider,
      from: dto.from,
      reference: dto.reference,
    });
  }

  /**
   * Send OTP
   */
  @Post('send-otp')
  async sendOtp(@Body() dto: SendOtpDto) {
    return this.smsService.sendOtp(dto.to, dto.otp, {
      provider: dto.provider,
    });
  }

  // ============================================================================
  // DELIVERY STATUS
  // ============================================================================

  /**
   * Check delivery status
   */
  @Get('status/:messageId')
  async checkDeliveryStatus(
    @Param('messageId') messageId: string,
    @Query('provider') provider?: string,
  ) {
    return this.smsService.checkDeliveryStatus(messageId, provider);
  }

  // ============================================================================
  // PROVIDER MANAGEMENT
  // ============================================================================

  /**
   * Get available providers
   */
  @Get('providers')
  async getProviders() {
    return this.smsService.getAvailableProviders();
  }

  /**
   * Get provider balance
   */
  @Get('providers/:provider/balance')
  async getProviderBalance(@Param('provider') provider: string) {
    return this.smsService.getProviderBalance(provider);
  }

  // ============================================================================
  // SMS LOGS
  // ============================================================================

  /**
   * Get SMS logs
   */
  @Get('logs')
  async getSmsLogs(
    @CurrentMerchant() merchantId: number,
    @Query('status') status?: string,
    @Query('provider') provider?: string,
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
  ) {
    return this.smsService.getSmsLogs(merchantId, {
      status,
      provider,
      startDate: startDate ? new Date(startDate) : undefined,
      endDate: endDate ? new Date(endDate) : undefined,
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 50,
    });
  }
}
