import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import axios from 'axios';
import {
  SmsProvider,
  SendSmsParams,
  SendSmsResult,
  SendBulkSmsParams,
  SendBulkSmsResult,
  DeliveryStatusResult,
} from './provider.interface';

@Injectable()
export class PlivoProvider implements SmsProvider {
  readonly name = 'plivo';
  readonly displayName = 'Plivo';
  readonly supportedCountries = ['*']; // Plivo supports most countries

  private readonly logger = new Logger(PlivoProvider.name);
  private readonly baseUrl = 'https://api.plivo.com/v1';
  private authId: string;
  private authToken: string;
  private senderId: string;

  constructor(private configService: ConfigService) {
    this.authId = this.configService.get<string>('PLIVO_AUTH_ID');
    this.authToken = this.configService.get<string>('PLIVO_AUTH_TOKEN');
    this.senderId = this.configService.get<string>('PLIVO_SENDER_ID');
  }

  isSupported(country: string): boolean {
    return true;
  }

  private getAuth(): string {
    return Buffer.from(`${this.authId}:${this.authToken}`).toString('base64');
  }

  async sendSms(params: SendSmsParams): Promise<SendSmsResult> {
    try {
      const response = await axios.post(
        `${this.baseUrl}/Account/${this.authId}/Message/`,
        {
          src: params.from || this.senderId,
          dst: this.formatPhoneNumber(params.to),
          text: params.message,
        },
        {
          headers: {
            Authorization: `Basic ${this.getAuth()}`,
            'Content-Type': 'application/json',
          },
        },
      );

      if (response.data.message_uuid) {
        return {
          success: true,
          messageId: Array.isArray(response.data.message_uuid)
            ? response.data.message_uuid[0]
            : response.data.message_uuid,
          status: 'sent',
          rawResponse: response.data,
        };
      }

      return {
        success: false,
        message: response.data.error || 'Failed to send',
        rawResponse: response.data,
      };
    } catch (error) {
      this.logger.error(`Plivo sendSms error: ${error.message}`);
      return {
        success: false,
        message: error.response?.data?.error || error.message,
      };
    }
  }

  async sendBulkSms(params: SendBulkSmsParams): Promise<SendBulkSmsResult> {
    try {
      const destinations = params.recipients
        .map(r => this.formatPhoneNumber(r))
        .join('<');

      const response = await axios.post(
        `${this.baseUrl}/Account/${this.authId}/Message/`,
        {
          src: params.from || this.senderId,
          dst: destinations,
          text: params.message,
        },
        {
          headers: {
            Authorization: `Basic ${this.getAuth()}`,
            'Content-Type': 'application/json',
          },
        },
      );

      const messageUuids = response.data.message_uuid || [];
      const results: SendBulkSmsResult['results'] = params.recipients.map((to, index) => ({
        to,
        messageId: messageUuids[index],
        status: messageUuids[index] ? 'sent' : 'failed',
      }));

      const totalSent = results.filter(r => r.status === 'sent').length;
      const totalFailed = results.filter(r => r.status === 'failed').length;

      return {
        success: totalSent > 0,
        totalSent,
        totalFailed,
        results,
      };
    } catch (error) {
      this.logger.error(`Plivo sendBulkSms error: ${error.message}`);
      return {
        success: false,
        totalSent: 0,
        totalFailed: params.recipients.length,
        results: params.recipients.map(to => ({
          to,
          status: 'failed' as const,
          error: error.message,
        })),
      };
    }
  }

  async checkDeliveryStatus(messageId: string): Promise<DeliveryStatusResult> {
    try {
      const response = await axios.get(
        `${this.baseUrl}/Account/${this.authId}/Message/${messageId}/`,
        {
          headers: {
            Authorization: `Basic ${this.getAuth()}`,
          },
        },
      );

      const statusMap: Record<string, DeliveryStatusResult['status']> = {
        queued: 'pending',
        sent: 'sent',
        delivered: 'delivered',
        undelivered: 'failed',
        failed: 'failed',
        rejected: 'failed',
      };

      return {
        success: true,
        messageId,
        status: statusMap[response.data.message_state] || 'unknown',
      };
    } catch (error) {
      return {
        success: false,
        messageId,
        status: 'unknown',
        error: error.message,
      };
    }
  }

  async getBalance(): Promise<{ balance: number; currency: string }> {
    try {
      const response = await axios.get(
        `${this.baseUrl}/Account/${this.authId}/`,
        {
          headers: {
            Authorization: `Basic ${this.getAuth()}`,
          },
        },
      );

      return {
        balance: parseFloat(response.data.cash_credits || '0'),
        currency: 'USD',
      };
    } catch (error) {
      this.logger.error(`Plivo getBalance error: ${error.message}`);
      return { balance: 0, currency: 'USD' };
    }
  }

  private formatPhoneNumber(phone: string): string {
    return phone.replace(/[^0-9]/g, '');
  }

  // ============================================================================
  // VOICE CALL
  // ============================================================================

  async makeCall(
    to: string,
    answerUrl: string,
    from?: string,
  ): Promise<{ success: boolean; callUuid?: string; message?: string }> {
    try {
      const response = await axios.post(
        `${this.baseUrl}/Account/${this.authId}/Call/`,
        {
          from: from || this.senderId,
          to: this.formatPhoneNumber(to),
          answer_url: answerUrl,
          answer_method: 'GET',
        },
        {
          headers: {
            Authorization: `Basic ${this.getAuth()}`,
            'Content-Type': 'application/json',
          },
        },
      );

      if (response.data.request_uuid) {
        return {
          success: true,
          callUuid: response.data.request_uuid,
        };
      }

      return { success: false, message: 'Failed to initiate call' };
    } catch (error) {
      return { success: false, message: error.message };
    }
  }
}
