import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import axios from 'axios';
import {
  SmsProvider,
  SendSmsParams,
  SendSmsResult,
  SendBulkSmsParams,
  SendBulkSmsResult,
  DeliveryStatusResult,
} from './provider.interface';

@Injectable()
export class AfricasTalkingProvider implements SmsProvider {
  readonly name = 'africastalking';
  readonly displayName = "Africa's Talking";
  readonly supportedCountries = ['KE', 'UG', 'TZ', 'RW', 'NG', 'ET', 'MW', 'ZM', 'GH', 'SN', 'CI', 'BJ'];

  private readonly logger = new Logger(AfricasTalkingProvider.name);
  private readonly baseUrl: string;
  private apiKey: string;
  private username: string;
  private senderId: string;

  constructor(private configService: ConfigService) {
    const isLive = this.configService.get<string>('AFRICASTALKING_MODE') === 'live';
    this.baseUrl = isLive
      ? 'https://api.africastalking.com/version1'
      : 'https://api.sandbox.africastalking.com/version1';

    this.apiKey = this.configService.get<string>('AFRICASTALKING_API_KEY');
    this.username = this.configService.get<string>('AFRICASTALKING_USERNAME');
    this.senderId = this.configService.get<string>('AFRICASTALKING_SENDER_ID');
  }

  isSupported(country: string): boolean {
    return this.supportedCountries.includes(country.toUpperCase());
  }

  async sendSms(params: SendSmsParams): Promise<SendSmsResult> {
    try {
      const formData = new URLSearchParams();
      formData.append('username', this.username);
      formData.append('to', this.formatPhoneNumber(params.to));
      formData.append('message', params.message);

      if (params.from || this.senderId) {
        formData.append('from', params.from || this.senderId);
      }

      const response = await axios.post(
        `${this.baseUrl}/messaging`,
        formData.toString(),
        {
          headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
            apiKey: this.apiKey,
            Accept: 'application/json',
          },
        },
      );

      const result = response.data.SMSMessageData;
      const recipient = result.Recipients?.[0];

      if (recipient && recipient.statusCode === 101) {
        return {
          success: true,
          messageId: recipient.messageId,
          status: recipient.status,
          cost: parseFloat(recipient.cost?.replace(/[^0-9.]/g, '') || '0'),
          rawResponse: response.data,
        };
      }

      return {
        success: false,
        message: recipient?.status || result.Message,
        rawResponse: response.data,
      };
    } catch (error) {
      this.logger.error(`AfricasTalking sendSms error: ${error.message}`);
      return {
        success: false,
        message: error.response?.data?.SMSMessageData?.Message || error.message,
      };
    }
  }

  async sendBulkSms(params: SendBulkSmsParams): Promise<SendBulkSmsResult> {
    try {
      const formattedRecipients = params.recipients
        .map(r => this.formatPhoneNumber(r))
        .join(',');

      const formData = new URLSearchParams();
      formData.append('username', this.username);
      formData.append('to', formattedRecipients);
      formData.append('message', params.message);

      if (params.from || this.senderId) {
        formData.append('from', params.from || this.senderId);
      }

      const response = await axios.post(
        `${this.baseUrl}/messaging`,
        formData.toString(),
        {
          headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
            apiKey: this.apiKey,
            Accept: 'application/json',
          },
        },
      );

      const result = response.data.SMSMessageData;
      const recipients = result.Recipients || [];

      const results: SendBulkSmsResult['results'] = recipients.map((r: any) => ({
        to: r.number,
        messageId: r.messageId,
        status: r.statusCode === 101 ? 'sent' : 'failed',
        error: r.statusCode !== 101 ? r.status : undefined,
      }));

      const totalSent = results.filter(r => r.status === 'sent').length;
      const totalFailed = results.filter(r => r.status === 'failed').length;

      return {
        success: totalSent > 0,
        totalSent,
        totalFailed,
        results,
      };
    } catch (error) {
      this.logger.error(`AfricasTalking sendBulkSms error: ${error.message}`);
      return {
        success: false,
        totalSent: 0,
        totalFailed: params.recipients.length,
        results: params.recipients.map(to => ({
          to,
          status: 'failed' as const,
          error: error.message,
        })),
      };
    }
  }

  async getBalance(): Promise<{ balance: number; currency: string }> {
    try {
      const response = await axios.get(
        `${this.baseUrl}/user?username=${this.username}`,
        {
          headers: {
            apiKey: this.apiKey,
            Accept: 'application/json',
          },
        },
      );

      const balance = response.data.UserData?.balance || '0';
      const match = balance.match(/([A-Z]+)\s*([\d.]+)/);

      if (match) {
        return {
          currency: match[1],
          balance: parseFloat(match[2]),
        };
      }

      return { balance: 0, currency: 'KES' };
    } catch (error) {
      this.logger.error(`AfricasTalking getBalance error: ${error.message}`);
      return { balance: 0, currency: 'KES' };
    }
  }

  private formatPhoneNumber(phone: string): string {
    let formatted = phone.replace(/[^0-9+]/g, '');
    if (!formatted.startsWith('+')) {
      formatted = '+' + formatted;
    }
    return formatted;
  }

  // ============================================================================
  // USSD (Africa's Talking)
  // ============================================================================

  async handleUssdRequest(
    sessionId: string,
    phoneNumber: string,
    text: string,
  ): Promise<string> {
    // This would be implemented based on your USSD menu logic
    // Returns the USSD response text prefixed with CON (continue) or END (end session)

    if (text === '') {
      return 'CON Bienvenue!\n1. Voir solde\n2. Recharger\n3. Historique';
    }

    const parts = text.split('*');
    const level = parts.length;

    switch (parts[0]) {
      case '1':
        return 'END Votre solde est de 5000 XOF';
      case '2':
        if (level === 1) {
          return 'CON Entrez le montant:';
        }
        return `END Rechargement de ${parts[1]} XOF initie`;
      case '3':
        return 'END Dernieres transactions:\n1. -500 XOF (Course)\n2. +1000 XOF (Recharge)';
      default:
        return 'END Option invalide';
    }
  }
}
