import { Injectable, Logger, OnModuleInit } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';
import { ConfigService } from '@nestjs/config';

export type SettingType = 'string' | 'number' | 'boolean' | 'json' | 'array';

export type SettingCategory =
  | 'general'
  | 'booking'
  | 'pricing'
  | 'payment'
  | 'driver'
  | 'user'
  | 'notification'
  | 'sms'
  | 'email'
  | 'app'
  | 'security'
  | 'integration';

export interface Setting {
  id: number;
  key: string;
  value: any;
  type: SettingType;
  category: SettingCategory;
  label: string;
  description?: string;
  merchantId?: number;
  isPublic: boolean;
  isEditable: boolean;
  validationRules?: Record<string, any>;
  options?: any[];
  updatedAt: Date;
}

export interface SettingDefinition {
  key: string;
  defaultValue: any;
  type: SettingType;
  category: SettingCategory;
  label: string;
  description?: string;
  isPublic?: boolean;
  isEditable?: boolean;
  validationRules?: Record<string, any>;
  options?: any[];
}

@Injectable()
export class SettingsService implements OnModuleInit {
  private readonly logger = new Logger(SettingsService.name);
  private cache: Map<string, { value: any; expires: number }> = new Map();
  private readonly cacheTTL = 60000; // 1 minute

  // Définitions des paramètres par défaut
  private readonly defaultSettings: SettingDefinition[] = [
    // ============================================================================
    // GENERAL
    // ============================================================================
    {
      key: 'app.name',
      defaultValue: 'MonkAPI',
      type: 'string',
      category: 'general',
      label: 'Nom de l\'application',
      isPublic: true,
    },
    {
      key: 'app.currency',
      defaultValue: 'XOF',
      type: 'string',
      category: 'general',
      label: 'Devise',
      options: ['XOF', 'XAF', 'NGN', 'GHS', 'KES', 'USD', 'EUR'],
    },
    {
      key: 'app.timezone',
      defaultValue: 'Africa/Dakar',
      type: 'string',
      category: 'general',
      label: 'Fuseau horaire',
    },
    {
      key: 'app.language',
      defaultValue: 'fr',
      type: 'string',
      category: 'general',
      label: 'Langue par défaut',
      options: ['fr', 'en', 'ar', 'pt', 'es'],
      isPublic: true,
    },
    {
      key: 'app.country_code',
      defaultValue: 'SN',
      type: 'string',
      category: 'general',
      label: 'Code pays',
    },
    {
      key: 'app.phone_prefix',
      defaultValue: '+221',
      type: 'string',
      category: 'general',
      label: 'Préfixe téléphone',
      isPublic: true,
    },

    // ============================================================================
    // BOOKING
    // ============================================================================
    {
      key: 'booking.auto_cancel_timeout',
      defaultValue: 300,
      type: 'number',
      category: 'booking',
      label: 'Délai annulation auto (secondes)',
      description: 'Délai avant annulation automatique si aucun chauffeur accepte',
      validationRules: { min: 60, max: 900 },
    },
    {
      key: 'booking.driver_acceptance_timeout',
      defaultValue: 30,
      type: 'number',
      category: 'booking',
      label: 'Délai acceptation chauffeur (secondes)',
      validationRules: { min: 10, max: 120 },
    },
    {
      key: 'booking.max_search_radius_km',
      defaultValue: 10,
      type: 'number',
      category: 'booking',
      label: 'Rayon de recherche max (km)',
      validationRules: { min: 1, max: 50 },
    },
    {
      key: 'booking.initial_search_radius_km',
      defaultValue: 3,
      type: 'number',
      category: 'booking',
      label: 'Rayon de recherche initial (km)',
      validationRules: { min: 1, max: 10 },
    },
    {
      key: 'booking.allow_cash_payment',
      defaultValue: true,
      type: 'boolean',
      category: 'booking',
      label: 'Autoriser paiement cash',
    },
    {
      key: 'booking.allow_scheduled',
      defaultValue: true,
      type: 'boolean',
      category: 'booking',
      label: 'Autoriser courses programmées',
    },
    {
      key: 'booking.max_scheduled_days',
      defaultValue: 7,
      type: 'number',
      category: 'booking',
      label: 'Jours max pour programmation',
      validationRules: { min: 1, max: 30 },
    },
    {
      key: 'booking.free_cancellation_minutes',
      defaultValue: 5,
      type: 'number',
      category: 'booking',
      label: 'Minutes annulation gratuite',
      validationRules: { min: 0, max: 30 },
    },
    {
      key: 'booking.cancellation_fee',
      defaultValue: 500,
      type: 'number',
      category: 'booking',
      label: 'Frais annulation',
      validationRules: { min: 0 },
    },

    // ============================================================================
    // PRICING
    // ============================================================================
    {
      key: 'pricing.surge_enabled',
      defaultValue: true,
      type: 'boolean',
      category: 'pricing',
      label: 'Activer surge pricing',
    },
    {
      key: 'pricing.max_surge_multiplier',
      defaultValue: 3.0,
      type: 'number',
      category: 'pricing',
      label: 'Multiplicateur surge max',
      validationRules: { min: 1, max: 5 },
    },
    {
      key: 'pricing.commission_rate',
      defaultValue: 20,
      type: 'number',
      category: 'pricing',
      label: 'Taux de commission (%)',
      validationRules: { min: 0, max: 50 },
    },
    {
      key: 'pricing.minimum_fare',
      defaultValue: 1000,
      type: 'number',
      category: 'pricing',
      label: 'Tarif minimum',
      validationRules: { min: 0 },
    },
    {
      key: 'pricing.waiting_fee_per_minute',
      defaultValue: 50,
      type: 'number',
      category: 'pricing',
      label: 'Frais attente par minute',
      validationRules: { min: 0 },
    },
    {
      key: 'pricing.free_waiting_minutes',
      defaultValue: 5,
      type: 'number',
      category: 'pricing',
      label: 'Minutes attente gratuites',
      validationRules: { min: 0, max: 15 },
    },

    // ============================================================================
    // DRIVER
    // ============================================================================
    {
      key: 'driver.auto_offline_minutes',
      defaultValue: 30,
      type: 'number',
      category: 'driver',
      label: 'Minutes avant mise hors-ligne auto',
      validationRules: { min: 5, max: 120 },
    },
    {
      key: 'driver.min_acceptance_rate',
      defaultValue: 70,
      type: 'number',
      category: 'driver',
      label: 'Taux acceptation minimum (%)',
      validationRules: { min: 0, max: 100 },
    },
    {
      key: 'driver.daily_target_trips',
      defaultValue: 10,
      type: 'number',
      category: 'driver',
      label: 'Objectif courses journalier',
      validationRules: { min: 1 },
    },
    {
      key: 'driver.location_update_interval',
      defaultValue: 10,
      type: 'number',
      category: 'driver',
      label: 'Intervalle MAJ position (secondes)',
      validationRules: { min: 5, max: 60 },
    },
    {
      key: 'driver.min_rating_threshold',
      defaultValue: 4.0,
      type: 'number',
      category: 'driver',
      label: 'Note minimum requise',
      validationRules: { min: 1, max: 5 },
    },
    {
      key: 'driver.require_selfie_verification',
      defaultValue: false,
      type: 'boolean',
      category: 'driver',
      label: 'Exiger vérification selfie',
    },

    // ============================================================================
    // USER
    // ============================================================================
    {
      key: 'user.referral_bonus_amount',
      defaultValue: 1000,
      type: 'number',
      category: 'user',
      label: 'Bonus parrainage',
      validationRules: { min: 0 },
    },
    {
      key: 'user.first_ride_discount',
      defaultValue: 20,
      type: 'number',
      category: 'user',
      label: 'Réduction première course (%)',
      validationRules: { min: 0, max: 100 },
    },
    {
      key: 'user.max_saved_addresses',
      defaultValue: 10,
      type: 'number',
      category: 'user',
      label: 'Adresses favorites max',
      validationRules: { min: 1, max: 50 },
    },
    {
      key: 'user.require_email_verification',
      defaultValue: false,
      type: 'boolean',
      category: 'user',
      label: 'Exiger vérification email',
    },
    {
      key: 'user.require_phone_verification',
      defaultValue: true,
      type: 'boolean',
      category: 'user',
      label: 'Exiger vérification téléphone',
    },

    // ============================================================================
    // NOTIFICATION
    // ============================================================================
    {
      key: 'notification.push_enabled',
      defaultValue: true,
      type: 'boolean',
      category: 'notification',
      label: 'Notifications push activées',
    },
    {
      key: 'notification.sms_enabled',
      defaultValue: true,
      type: 'boolean',
      category: 'notification',
      label: 'Notifications SMS activées',
    },
    {
      key: 'notification.email_enabled',
      defaultValue: true,
      type: 'boolean',
      category: 'notification',
      label: 'Notifications email activées',
    },
    {
      key: 'notification.whatsapp_enabled',
      defaultValue: false,
      type: 'boolean',
      category: 'notification',
      label: 'Notifications WhatsApp activées',
    },

    // ============================================================================
    // SECURITY
    // ============================================================================
    {
      key: 'security.otp_expiry_minutes',
      defaultValue: 5,
      type: 'number',
      category: 'security',
      label: 'Expiration OTP (minutes)',
      validationRules: { min: 1, max: 30 },
    },
    {
      key: 'security.max_login_attempts',
      defaultValue: 5,
      type: 'number',
      category: 'security',
      label: 'Tentatives connexion max',
      validationRules: { min: 3, max: 10 },
    },
    {
      key: 'security.lockout_duration_minutes',
      defaultValue: 30,
      type: 'number',
      category: 'security',
      label: 'Durée blocage (minutes)',
      validationRules: { min: 5, max: 1440 },
    },
    {
      key: 'security.session_timeout_hours',
      defaultValue: 24,
      type: 'number',
      category: 'security',
      label: 'Expiration session (heures)',
      validationRules: { min: 1, max: 720 },
    },

    // ============================================================================
    // APP FEATURES
    // ============================================================================
    {
      key: 'app.maintenance_mode',
      defaultValue: false,
      type: 'boolean',
      category: 'app',
      label: 'Mode maintenance',
    },
    {
      key: 'app.maintenance_message',
      defaultValue: 'L\'application est en maintenance. Veuillez réessayer plus tard.',
      type: 'string',
      category: 'app',
      label: 'Message maintenance',
      isPublic: true,
    },
    {
      key: 'app.min_android_version',
      defaultValue: '1.0.0',
      type: 'string',
      category: 'app',
      label: 'Version Android minimum',
      isPublic: true,
    },
    {
      key: 'app.min_ios_version',
      defaultValue: '1.0.0',
      type: 'string',
      category: 'app',
      label: 'Version iOS minimum',
      isPublic: true,
    },
    {
      key: 'app.force_update',
      defaultValue: false,
      type: 'boolean',
      category: 'app',
      label: 'Forcer mise à jour',
      isPublic: true,
    },
  ];

  constructor(
    private prisma: PrismaService,
    private configService: ConfigService,
  ) {}

  async onModuleInit() {
    // Initialiser les paramètres par défaut si nécessaire
    await this.initializeDefaults();
  }

  /**
   * Initialiser les paramètres par défaut
   */
  private async initializeDefaults(): Promise<void> {
    for (const setting of this.defaultSettings) {
      const exists = await this.prisma.setting.findFirst({
        where: { key: setting.key, merchant_id: null },
      });

      if (!exists) {
        await this.prisma.setting.create({
          data: {
            key: setting.key,
            value: JSON.stringify(setting.defaultValue),
            type: setting.type,
            category: setting.category,
            label: setting.label,
            description: setting.description,
            is_public: setting.isPublic ?? false,
            is_editable: setting.isEditable ?? true,
            validation_rules: setting.validationRules
              ? JSON.stringify(setting.validationRules)
              : null,
            options: setting.options ? JSON.stringify(setting.options) : null,
            created_at: new Date(),
            updated_at: new Date(),
          },
        });
      }
    }

    this.logger.log('Paramètres par défaut initialisés');
  }

  /**
   * Obtenir une valeur de paramètre
   */
  async get<T>(key: string, merchantId?: number): Promise<T> {
    const cacheKey = `${merchantId || 'global'}:${key}`;

    // Vérifier le cache
    const cached = this.cache.get(cacheKey);
    if (cached && cached.expires > Date.now()) {
      return cached.value as T;
    }

    // Chercher le paramètre merchant-specific d'abord
    let setting = null;
    if (merchantId) {
      setting = await this.prisma.setting.findFirst({
        where: { key, merchant_id: merchantId },
      });
    }

    // Sinon, chercher le paramètre global
    if (!setting) {
      setting = await this.prisma.setting.findFirst({
        where: { key, merchant_id: null },
      });
    }

    if (!setting) {
      // Retourner la valeur par défaut si définie
      const defaultSetting = this.defaultSettings.find((s) => s.key === key);
      return (defaultSetting?.defaultValue ?? null) as T;
    }

    const value = this.parseValue(setting.value, setting.type);

    // Mettre en cache
    this.cache.set(cacheKey, { value, expires: Date.now() + this.cacheTTL });

    return value as T;
  }

  /**
   * Obtenir plusieurs paramètres
   */
  async getMany(
    keys: string[],
    merchantId?: number,
  ): Promise<Record<string, any>> {
    const result: Record<string, any> = {};

    for (const key of keys) {
      result[key] = await this.get(key, merchantId);
    }

    return result;
  }

  /**
   * Obtenir tous les paramètres d'une catégorie
   */
  async getByCategory(
    category: SettingCategory,
    merchantId?: number,
  ): Promise<Setting[]> {
    const where: any = { category };

    if (merchantId) {
      where.OR = [{ merchant_id: merchantId }, { merchant_id: null }];
    } else {
      where.merchant_id = null;
    }

    const settings = await this.prisma.setting.findMany({
      where,
      orderBy: { key: 'asc' },
    });

    return settings.map((s) => this.mapToSetting(s));
  }

  /**
   * Obtenir tous les paramètres publics
   */
  async getPublicSettings(merchantId?: number): Promise<Record<string, any>> {
    const where: any = { is_public: true };

    if (merchantId) {
      where.OR = [{ merchant_id: merchantId }, { merchant_id: null }];
    } else {
      where.merchant_id = null;
    }

    const settings = await this.prisma.setting.findMany({ where });

    const result: Record<string, any> = {};
    for (const setting of settings) {
      result[setting.key] = this.parseValue(setting.value, setting.type);
    }

    return result;
  }

  /**
   * Définir une valeur de paramètre
   */
  async set(
    key: string,
    value: any,
    merchantId?: number,
  ): Promise<Setting | null> {
    // Trouver le paramètre existant ou créer
    const where: any = { key };
    if (merchantId) {
      where.merchant_id = merchantId;
    } else {
      where.merchant_id = null;
    }

    let setting = await this.prisma.setting.findFirst({ where });

    // Valider la valeur
    if (setting?.validation_rules) {
      const rules = JSON.parse(setting.validation_rules);
      if (!this.validateValue(value, rules)) {
        throw new Error(`Valeur invalide pour ${key}`);
      }
    }

    const serializedValue = JSON.stringify(value);

    if (setting) {
      setting = await this.prisma.setting.update({
        where: { id: setting.id },
        data: { value: serializedValue, updated_at: new Date() },
      });
    } else {
      // Copier depuis le paramètre global
      const globalSetting = await this.prisma.setting.findFirst({
        where: { key, merchant_id: null },
      });

      if (!globalSetting) {
        throw new Error(`Paramètre ${key} non trouvé`);
      }

      setting = await this.prisma.setting.create({
        data: {
          key,
          value: serializedValue,
          type: globalSetting.type,
          category: globalSetting.category,
          label: globalSetting.label,
          description: globalSetting.description,
          merchant_id: merchantId,
          is_public: globalSetting.is_public,
          is_editable: globalSetting.is_editable,
          validation_rules: globalSetting.validation_rules,
          options: globalSetting.options,
          created_at: new Date(),
          updated_at: new Date(),
        },
      });
    }

    // Invalider le cache
    this.invalidateCache(key, merchantId);

    return this.mapToSetting(setting);
  }

  /**
   * Définir plusieurs paramètres
   */
  async setMany(
    settings: Record<string, any>,
    merchantId?: number,
  ): Promise<number> {
    let updated = 0;

    for (const [key, value] of Object.entries(settings)) {
      try {
        await this.set(key, value, merchantId);
        updated++;
      } catch (error) {
        this.logger.warn(`Erreur mise à jour ${key}: ${error.message}`);
      }
    }

    return updated;
  }

  /**
   * Réinitialiser un paramètre à sa valeur par défaut
   */
  async reset(key: string, merchantId?: number): Promise<boolean> {
    if (merchantId) {
      // Supprimer le paramètre merchant-specific
      const result = await this.prisma.setting.deleteMany({
        where: { key, merchant_id: merchantId },
      });
      this.invalidateCache(key, merchantId);
      return result.count > 0;
    }

    // Réinitialiser à la valeur par défaut
    const defaultSetting = this.defaultSettings.find((s) => s.key === key);
    if (!defaultSetting) {
      return false;
    }

    await this.prisma.setting.updateMany({
      where: { key, merchant_id: null },
      data: {
        value: JSON.stringify(defaultSetting.defaultValue),
        updated_at: new Date(),
      },
    });

    this.invalidateCache(key, merchantId);
    return true;
  }

  /**
   * Obtenir toutes les catégories
   */
  getCategories(): SettingCategory[] {
    return [
      'general',
      'booking',
      'pricing',
      'payment',
      'driver',
      'user',
      'notification',
      'sms',
      'email',
      'app',
      'security',
      'integration',
    ];
  }

  /**
   * Obtenir les définitions de paramètres
   */
  getDefinitions(): SettingDefinition[] {
    return this.defaultSettings;
  }

  /**
   * Vider le cache
   */
  clearCache(): void {
    this.cache.clear();
  }

  // ============================================================================
  // MÉTHODES PRIVÉES
  // ============================================================================

  private parseValue(value: string, type: string): any {
    try {
      const parsed = JSON.parse(value);

      switch (type) {
        case 'number':
          return Number(parsed);
        case 'boolean':
          return Boolean(parsed);
        case 'json':
        case 'array':
          return parsed;
        default:
          return parsed;
      }
    } catch {
      return value;
    }
  }

  private validateValue(value: any, rules: Record<string, any>): boolean {
    if (rules.min !== undefined && value < rules.min) {
      return false;
    }
    if (rules.max !== undefined && value > rules.max) {
      return false;
    }
    if (rules.pattern && !new RegExp(rules.pattern).test(value)) {
      return false;
    }
    return true;
  }

  private invalidateCache(key: string, merchantId?: number): void {
    const cacheKey = `${merchantId || 'global'}:${key}`;
    this.cache.delete(cacheKey);
    // Aussi invalider le cache global si c'est un paramètre global
    if (!merchantId) {
      this.cache.delete(`global:${key}`);
    }
  }

  private mapToSetting(setting: any): Setting {
    return {
      id: setting.id,
      key: setting.key,
      value: this.parseValue(setting.value, setting.type),
      type: setting.type,
      category: setting.category,
      label: setting.label,
      description: setting.description,
      merchantId: setting.merchant_id,
      isPublic: setting.is_public,
      isEditable: setting.is_editable,
      validationRules: setting.validation_rules
        ? JSON.parse(setting.validation_rules)
        : null,
      options: setting.options ? JSON.parse(setting.options) : null,
      updatedAt: setting.updated_at,
    };
  }
}
