import {
  Controller,
  Get,
  Put,
  Post,
  Delete,
  Body,
  Param,
  Query,
  UseGuards,
} from '@nestjs/common';
import { SettingsService } from './settings.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantId } from '../../common/decorators/merchant.decorator';

// ============================================================================
// DTOs
// ============================================================================

class UpdateSettingDto {
  value: any;
}

class BulkUpdateSettingsDto {
  settings: { key: string; value: any }[];
}

class CreateSettingDto {
  key: string;
  value: any;
  category: string;
  description?: string;
  type?: 'string' | 'number' | 'boolean' | 'json' | 'array';
  validation?: {
    min?: number;
    max?: number;
    pattern?: string;
    options?: any[];
  };
}

// ============================================================================
// CONTROLLER
// ============================================================================

@Controller('settings')
@UseGuards(JwtAuthGuard)
export class SettingsController {
  constructor(private readonly settingsService: SettingsService) {}

  // ==========================================================================
  // PUBLIC SETTINGS (App Config)
  // ==========================================================================

  /**
   * Obtenir les paramètres publics de l'app (sans auth)
   */
  @Get('public')
  async getPublicSettings(@MerchantId() merchantId: number) {
    const publicKeys = [
      'app_name',
      'currency',
      'currency_symbol',
      'country_code',
      'phone_code',
      'distance_unit',
      'support_email',
      'support_phone',
      'min_app_version_android',
      'min_app_version_ios',
      'force_update',
    ];

    const settings: Record<string, any> = {};
    for (const key of publicKeys) {
      settings[key] = await this.settingsService.get(key, merchantId);
    }

    return settings;
  }

  // ==========================================================================
  // SETTINGS BY CATEGORY
  // ==========================================================================

  /**
   * Lister toutes les catégories
   */
  @Get('categories')
  getCategories() {
    return {
      categories: [
        { key: 'general', name: 'Général', description: 'Paramètres généraux de l\'application' },
        { key: 'booking', name: 'Réservations', description: 'Configuration des réservations' },
        { key: 'pricing', name: 'Tarification', description: 'Paramètres de tarification' },
        { key: 'payment', name: 'Paiements', description: 'Configuration des paiements' },
        { key: 'driver', name: 'Chauffeurs', description: 'Paramètres des chauffeurs' },
        { key: 'user', name: 'Utilisateurs', description: 'Paramètres des utilisateurs' },
        { key: 'notification', name: 'Notifications', description: 'Configuration des notifications' },
        { key: 'sms', name: 'SMS', description: 'Paramètres SMS' },
        { key: 'email', name: 'Email', description: 'Paramètres email' },
        { key: 'security', name: 'Sécurité', description: 'Paramètres de sécurité' },
        { key: 'app', name: 'Application', description: 'Paramètres de l\'app mobile' },
        { key: 'integration', name: 'Intégrations', description: 'Intégrations tierces' },
      ],
    };
  }

  /**
   * Obtenir les paramètres par catégorie
   */
  @Get('category/:category')
  async getByCategory(
    @MerchantId() merchantId: number,
    @Param('category') category: string,
  ) {
    return this.settingsService.getByCategory(category, merchantId);
  }

  // ==========================================================================
  // CRUD OPERATIONS
  // ==========================================================================

  /**
   * Obtenir tous les paramètres
   */
  @Get()
  async getAllSettings(
    @MerchantId() merchantId: number,
    @Query('category') category?: string,
    @Query('search') search?: string,
  ) {
    if (category) {
      return this.settingsService.getByCategory(category, merchantId);
    }

    return this.settingsService.getAll(merchantId);
  }

  /**
   * Obtenir un paramètre spécifique
   */
  @Get(':key')
  async getSetting(
    @MerchantId() merchantId: number,
    @Param('key') key: string,
  ) {
    const value = await this.settingsService.get(key, merchantId);
    const metadata = await this.settingsService.getMetadata(key, merchantId);

    return {
      key,
      value,
      ...metadata,
    };
  }

  /**
   * Mettre à jour un paramètre
   */
  @Put(':key')
  async updateSetting(
    @MerchantId() merchantId: number,
    @Param('key') key: string,
    @Body() dto: UpdateSettingDto,
  ) {
    const updated = await this.settingsService.set(key, dto.value, merchantId);

    return {
      success: true,
      key,
      value: updated,
      message: 'Paramètre mis à jour',
    };
  }

  /**
   * Mise à jour en masse
   */
  @Put()
  async bulkUpdate(
    @MerchantId() merchantId: number,
    @Body() dto: BulkUpdateSettingsDto,
  ) {
    const results = await this.settingsService.setMany(dto.settings, merchantId);

    return {
      success: true,
      updated: results.length,
      settings: results,
    };
  }

  /**
   * Créer un nouveau paramètre personnalisé
   */
  @Post()
  async createSetting(
    @MerchantId() merchantId: number,
    @Body() dto: CreateSettingDto,
  ) {
    const created = await this.settingsService.create(merchantId, {
      key: dto.key,
      value: dto.value,
      category: dto.category,
      description: dto.description,
      type: dto.type || 'string',
      validation: dto.validation,
    });

    return {
      success: true,
      setting: created,
    };
  }

  /**
   * Supprimer un paramètre personnalisé
   */
  @Delete(':key')
  async deleteSetting(
    @MerchantId() merchantId: number,
    @Param('key') key: string,
  ) {
    const deleted = await this.settingsService.delete(key, merchantId);

    return {
      success: deleted,
      message: deleted ? 'Paramètre supprimé' : 'Paramètre non trouvé ou non supprimable',
    };
  }

  // ==========================================================================
  // RESET & DEFAULTS
  // ==========================================================================

  /**
   * Réinitialiser un paramètre à sa valeur par défaut
   */
  @Post(':key/reset')
  async resetSetting(
    @MerchantId() merchantId: number,
    @Param('key') key: string,
  ) {
    const value = await this.settingsService.reset(key, merchantId);

    return {
      success: true,
      key,
      value,
      message: 'Paramètre réinitialisé',
    };
  }

  /**
   * Réinitialiser tous les paramètres d'une catégorie
   */
  @Post('category/:category/reset')
  async resetCategory(
    @MerchantId() merchantId: number,
    @Param('category') category: string,
  ) {
    const count = await this.settingsService.resetCategory(category, merchantId);

    return {
      success: true,
      category,
      resetCount: count,
      message: `${count} paramètres réinitialisés`,
    };
  }

  /**
   * Réinitialiser tous les paramètres
   */
  @Post('reset-all')
  async resetAll(@MerchantId() merchantId: number) {
    const count = await this.settingsService.resetAll(merchantId);

    return {
      success: true,
      resetCount: count,
      message: 'Tous les paramètres ont été réinitialisés',
    };
  }

  // ==========================================================================
  // EXPORT/IMPORT
  // ==========================================================================

  /**
   * Exporter tous les paramètres
   */
  @Get('export/json')
  async exportSettings(@MerchantId() merchantId: number) {
    const settings = await this.settingsService.export(merchantId);

    return {
      merchantId,
      exportedAt: new Date().toISOString(),
      settings,
    };
  }

  /**
   * Importer des paramètres
   */
  @Post('import')
  async importSettings(
    @MerchantId() merchantId: number,
    @Body() body: { settings: Record<string, any>; overwrite?: boolean },
  ) {
    const result = await this.settingsService.import(
      merchantId,
      body.settings,
      body.overwrite ?? false,
    );

    return {
      success: true,
      imported: result.imported,
      skipped: result.skipped,
      errors: result.errors,
    };
  }

  // ==========================================================================
  // FEATURE FLAGS
  // ==========================================================================

  /**
   * Obtenir les feature flags
   */
  @Get('features/flags')
  async getFeatureFlags(@MerchantId() merchantId: number) {
    const flags = [
      'booking_enabled',
      'delivery_enabled',
      'wallet_enabled',
      'referral_enabled',
      'promo_enabled',
      'surge_pricing_enabled',
      'scheduled_booking_enabled',
      'driver_tips_enabled',
      'driver_subscription',
      'cash_payment_enabled',
      'card_payment_enabled',
      'wallet_payment_enabled',
    ];

    const result: Record<string, boolean> = {};
    for (const flag of flags) {
      result[flag] = await this.settingsService.get(flag, merchantId) === true;
    }

    return result;
  }

  /**
   * Activer/désactiver une feature
   */
  @Put('features/:feature')
  async toggleFeature(
    @MerchantId() merchantId: number,
    @Param('feature') feature: string,
    @Body() body: { enabled: boolean },
  ) {
    await this.settingsService.set(feature, body.enabled, merchantId);

    return {
      success: true,
      feature,
      enabled: body.enabled,
    };
  }

  // ==========================================================================
  // VALIDATION METADATA
  // ==========================================================================

  /**
   * Obtenir les règles de validation pour un paramètre
   */
  @Get(':key/validation')
  async getValidation(
    @MerchantId() merchantId: number,
    @Param('key') key: string,
  ) {
    return this.settingsService.getValidationRules(key, merchantId);
  }

  /**
   * Valider une valeur sans la sauvegarder
   */
  @Post(':key/validate')
  async validateValue(
    @MerchantId() merchantId: number,
    @Param('key') key: string,
    @Body() body: { value: any },
  ) {
    const result = await this.settingsService.validate(key, body.value, merchantId);

    return {
      valid: result.valid,
      errors: result.errors,
    };
  }
}
