import {
  Controller,
  Get,
  Post,
  Put,
  Delete,
  Body,
  Param,
  Query,
  UseGuards,
  ParseIntPipe,
  DefaultValuePipe,
  Res,
} from '@nestjs/common';
import { Response } from 'express';
import {
  ReportService,
  ReportType,
  ReportFormat,
  ReportPeriod,
  ReportOptions,
} from './report.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantId } from '../../common/decorators/merchant.decorator';

// ============================================================================
// DTOs
// ============================================================================

class GenerateReportDto {
  type: ReportType;
  format: ReportFormat;
  period: ReportPeriod;
  startDate?: string;
  endDate?: string;
  filters?: Record<string, any>;
  groupBy?: string;
}

class CreateScheduledReportDto {
  name: string;
  type: ReportType;
  format: ReportFormat;
  schedule: string;
  recipients: string[];
  filters?: Record<string, any>;
}

class UpdateScheduledReportDto {
  name?: string;
  schedule?: string;
  recipients?: string[];
  filters?: Record<string, any>;
  active?: boolean;
}

// ============================================================================
// CONTROLLER
// ============================================================================

@Controller('reports')
@UseGuards(JwtAuthGuard)
export class ReportController {
  constructor(private readonly reportService: ReportService) {}

  // ==========================================================================
  // GENERATION RAPPORTS
  // ==========================================================================

  /**
   * Générer un rapport
   */
  @Post('generate')
  async generateReport(
    @MerchantId() merchantId: number,
    @Body() dto: GenerateReportDto,
  ) {
    const options: ReportOptions = {
      type: dto.type,
      format: dto.format,
      period: dto.period,
      startDate: dto.startDate ? new Date(dto.startDate) : undefined,
      endDate: dto.endDate ? new Date(dto.endDate) : undefined,
      filters: dto.filters,
      groupBy: dto.groupBy,
    };

    const report = await this.reportService.generateReport(merchantId, options);

    return {
      success: true,
      report,
    };
  }

  /**
   * Rapport rapide - Réservations
   */
  @Get('bookings')
  async getBookingsReport(
    @MerchantId() merchantId: number,
    @Query('period', new DefaultValuePipe('month')) period: ReportPeriod,
    @Query('format', new DefaultValuePipe('json')) format: ReportFormat,
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
  ) {
    return this.reportService.generateReport(merchantId, {
      type: 'bookings',
      format,
      period,
      startDate: startDate ? new Date(startDate) : undefined,
      endDate: endDate ? new Date(endDate) : undefined,
    });
  }

  /**
   * Rapport rapide - Revenus
   */
  @Get('earnings')
  async getEarningsReport(
    @MerchantId() merchantId: number,
    @Query('period', new DefaultValuePipe('month')) period: ReportPeriod,
    @Query('format', new DefaultValuePipe('json')) format: ReportFormat,
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
  ) {
    return this.reportService.generateReport(merchantId, {
      type: 'earnings',
      format,
      period,
      startDate: startDate ? new Date(startDate) : undefined,
      endDate: endDate ? new Date(endDate) : undefined,
    });
  }

  /**
   * Rapport rapide - Chauffeurs
   */
  @Get('drivers')
  async getDriversReport(
    @MerchantId() merchantId: number,
    @Query('period', new DefaultValuePipe('month')) period: ReportPeriod,
    @Query('format', new DefaultValuePipe('json')) format: ReportFormat,
  ) {
    return this.reportService.generateReport(merchantId, {
      type: 'drivers',
      format,
      period,
    });
  }

  /**
   * Rapport rapide - Utilisateurs
   */
  @Get('users')
  async getUsersReport(
    @MerchantId() merchantId: number,
    @Query('period', new DefaultValuePipe('month')) period: ReportPeriod,
    @Query('format', new DefaultValuePipe('json')) format: ReportFormat,
  ) {
    return this.reportService.generateReport(merchantId, {
      type: 'users',
      format,
      period,
    });
  }

  /**
   * Rapport rapide - Paiements
   */
  @Get('payments')
  async getPaymentsReport(
    @MerchantId() merchantId: number,
    @Query('period', new DefaultValuePipe('month')) period: ReportPeriod,
    @Query('format', new DefaultValuePipe('json')) format: ReportFormat,
  ) {
    return this.reportService.generateReport(merchantId, {
      type: 'payments',
      format,
      period,
    });
  }

  /**
   * Rapport rapide - Financier
   */
  @Get('financial')
  async getFinancialReport(
    @MerchantId() merchantId: number,
    @Query('period', new DefaultValuePipe('month')) period: ReportPeriod,
    @Query('format', new DefaultValuePipe('json')) format: ReportFormat,
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
  ) {
    return this.reportService.generateReport(merchantId, {
      type: 'financial',
      format,
      period,
      startDate: startDate ? new Date(startDate) : undefined,
      endDate: endDate ? new Date(endDate) : undefined,
    });
  }

  /**
   * Rapport rapide - Opérationnel
   */
  @Get('operational')
  async getOperationalReport(
    @MerchantId() merchantId: number,
    @Query('period', new DefaultValuePipe('week')) period: ReportPeriod,
    @Query('format', new DefaultValuePipe('json')) format: ReportFormat,
  ) {
    return this.reportService.generateReport(merchantId, {
      type: 'operational',
      format,
      period,
    });
  }

  /**
   * Rapport rapide - Support
   */
  @Get('support')
  async getSupportReport(
    @MerchantId() merchantId: number,
    @Query('period', new DefaultValuePipe('month')) period: ReportPeriod,
    @Query('format', new DefaultValuePipe('json')) format: ReportFormat,
  ) {
    return this.reportService.generateReport(merchantId, {
      type: 'support',
      format,
      period,
    });
  }

  /**
   * Rapport rapide - Évaluations
   */
  @Get('ratings')
  async getRatingsReport(
    @MerchantId() merchantId: number,
    @Query('period', new DefaultValuePipe('month')) period: ReportPeriod,
    @Query('format', new DefaultValuePipe('json')) format: ReportFormat,
  ) {
    return this.reportService.generateReport(merchantId, {
      type: 'ratings',
      format,
      period,
    });
  }

  // ==========================================================================
  // RAPPORTS PLANIFIES
  // ==========================================================================

  /**
   * Lister les rapports planifiés
   */
  @Get('scheduled')
  async listScheduledReports(@MerchantId() merchantId: number) {
    return this.reportService.listScheduledReports(merchantId);
  }

  /**
   * Créer un rapport planifié
   */
  @Post('scheduled')
  async createScheduledReport(
    @MerchantId() merchantId: number,
    @Body() dto: CreateScheduledReportDto,
  ) {
    const report = await this.reportService.createScheduledReport(merchantId, dto);
    return {
      success: true,
      scheduledReport: report,
    };
  }

  /**
   * Mettre à jour un rapport planifié
   */
  @Put('scheduled/:id')
  async updateScheduledReport(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: UpdateScheduledReportDto,
  ) {
    const report = await this.reportService.updateScheduledReport(id, merchantId, dto);

    if (!report) {
      return { error: 'Rapport planifié non trouvé' };
    }

    return {
      success: true,
      scheduledReport: report,
    };
  }

  /**
   * Supprimer un rapport planifié
   */
  @Delete('scheduled/:id')
  async deleteScheduledReport(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const deleted = await this.reportService.deleteScheduledReport(id, merchantId);
    return {
      success: deleted,
      message: deleted ? 'Rapport planifié supprimé' : 'Rapport planifié non trouvé',
    };
  }

  // ==========================================================================
  // HISTORIQUE
  // ==========================================================================

  /**
   * Historique des rapports générés
   */
  @Get('history')
  async getReportHistory(
    @MerchantId() merchantId: number,
    @Query('page', new DefaultValuePipe(1), ParseIntPipe) page: number,
    @Query('limit', new DefaultValuePipe(20), ParseIntPipe) limit: number,
    @Query('type') type?: ReportType,
  ) {
    return this.reportService.getReportHistory(merchantId, { page, limit, type });
  }

  // ==========================================================================
  // METADATA
  // ==========================================================================

  /**
   * Types de rapports disponibles
   */
  @Get('meta/types')
  getReportTypes() {
    return {
      types: [
        { type: 'bookings', name: 'Réservations', description: 'Rapport des courses' },
        { type: 'earnings', name: 'Revenus', description: 'Rapport des revenus et gains' },
        { type: 'drivers', name: 'Chauffeurs', description: 'Performance des chauffeurs' },
        { type: 'users', name: 'Utilisateurs', description: 'Activité des utilisateurs' },
        { type: 'payments', name: 'Paiements', description: 'Transactions et paiements' },
        { type: 'deliveries', name: 'Livraisons', description: 'Rapport des livraisons' },
        { type: 'ratings', name: 'Évaluations', description: 'Notes et avis' },
        { type: 'support', name: 'Support', description: 'Tickets de support' },
        { type: 'financial', name: 'Financier', description: 'Rapport financier complet' },
        { type: 'operational', name: 'Opérationnel', description: 'Métriques opérationnelles' },
      ],
    };
  }

  /**
   * Formats disponibles
   */
  @Get('meta/formats')
  getFormats() {
    return {
      formats: [
        { format: 'json', name: 'JSON', description: 'Données brutes' },
        { format: 'csv', name: 'CSV', description: 'Fichier tableur CSV' },
        { format: 'excel', name: 'Excel', description: 'Fichier Excel (.xlsx)' },
        { format: 'pdf', name: 'PDF', description: 'Document PDF' },
      ],
    };
  }

  /**
   * Périodes disponibles
   */
  @Get('meta/periods')
  getPeriods() {
    return {
      periods: [
        { period: 'today', name: "Aujourd'hui" },
        { period: 'yesterday', name: 'Hier' },
        { period: 'week', name: '7 derniers jours' },
        { period: 'month', name: 'Ce mois' },
        { period: 'quarter', name: 'Ce trimestre' },
        { period: 'year', name: 'Cette année' },
        { period: 'custom', name: 'Personnalisé' },
      ],
    };
  }

  /**
   * Expressions cron courantes
   */
  @Get('meta/schedules')
  getScheduleExamples() {
    return {
      schedules: [
        { cron: '0 8 * * *', name: 'Tous les jours à 8h' },
        { cron: '0 8 * * 1', name: 'Tous les lundis à 8h' },
        { cron: '0 8 1 * *', name: 'Le 1er de chaque mois à 8h' },
        { cron: '0 8 1 1,4,7,10 *', name: 'Chaque trimestre' },
        { cron: '0 0 * * 0', name: 'Tous les dimanches à minuit' },
      ],
    };
  }
}
