import { Injectable, Logger, BadRequestException } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';
import { NotificationService } from '../notification/notification.service';

interface ReferralStats {
  totalReferrals: number;
  pendingReferrals: number;
  completedReferrals: number;
  totalEarned: number;
  currency: string;
}

@Injectable()
export class ReferralService {
  private readonly logger = new Logger(ReferralService.name);

  constructor(
    private prisma: PrismaService,
    private notificationService: NotificationService,
  ) {}

  /**
   * Generate unique referral code for user/driver
   */
  async generateReferralCode(
    userId: number,
    userType: 'user' | 'driver',
    merchantId: number,
  ): Promise<string> {
    // Check if user already has a referral code
    let referral: any;

    if (userType === 'user') {
      referral = await this.prisma.userReferral.findFirst({
        where: { user_id: userId },
      });

      if (referral?.referral_code) {
        return referral.referral_code;
      }
    } else {
      referral = await this.prisma.driverReferral.findFirst({
        where: { driver_id: userId },
      });

      if (referral?.referral_code) {
        return referral.referral_code;
      }
    }

    // Generate new unique code
    const code = await this.generateUniqueCode(merchantId);

    // Save to database
    if (userType === 'user') {
      await this.prisma.userReferral.upsert({
        where: { user_id: userId },
        create: {
          user_id: userId,
          merchant_id: merchantId,
          referral_code: code,
          total_referrals: 0,
          total_earned: 0,
          created_at: new Date(),
        },
        update: {
          referral_code: code,
        },
      });
    } else {
      await this.prisma.driverReferral.upsert({
        where: { driver_id: userId },
        create: {
          driver_id: userId,
          merchant_id: merchantId,
          referral_code: code,
          total_referrals: 0,
          total_earned: 0,
          created_at: new Date(),
        },
        update: {
          referral_code: code,
        },
      });
    }

    this.logger.log(`Generated referral code ${code} for ${userType} #${userId}`);
    return code;
  }

  /**
   * Apply referral code during signup
   */
  async applyReferralCode(
    newUserId: number,
    newUserType: 'user' | 'driver',
    referralCode: string,
    merchantId: number,
  ): Promise<{ success: boolean; message: string; referrerId?: number }> {
    // Find referrer
    const userReferral = await this.prisma.userReferral.findFirst({
      where: { referral_code: referralCode, merchant_id: merchantId },
    });

    const driverReferral = await this.prisma.driverReferral.findFirst({
      where: { referral_code: referralCode, merchant_id: merchantId },
    });

    if (!userReferral && !driverReferral) {
      return { success: false, message: 'Code de parrainage invalide' };
    }

    const referrerId = userReferral?.user_id || driverReferral?.driver_id;
    const referrerType = userReferral ? 'user' : 'driver';

    // Prevent self-referral
    if (referrerId === newUserId && referrerType === newUserType) {
      return { success: false, message: 'Vous ne pouvez pas utiliser votre propre code' };
    }

    // Check if already referred
    const existingReferral = await this.prisma.referralTransaction.findFirst({
      where: {
        referred_id: newUserId,
        referred_type: newUserType,
      },
    });

    if (existingReferral) {
      return { success: false, message: 'Vous avez deja utilise un code de parrainage' };
    }

    // Get referral bonus settings
    const settings = await this.getReferralSettings(merchantId);

    // Create referral transaction (pending)
    await this.prisma.referralTransaction.create({
      data: {
        merchant_id: merchantId,
        referrer_id: referrerId,
        referrer_type: referrerType,
        referred_id: newUserId,
        referred_type: newUserType,
        referral_code: referralCode,
        referrer_bonus: settings.referrerBonus,
        referred_bonus: settings.referredBonus,
        status: 'pending', // Will be completed after first ride
        created_at: new Date(),
      },
    });

    this.logger.log(
      `Referral code ${referralCode} applied by ${newUserType} #${newUserId}`,
    );

    return {
      success: true,
      message: `Code applique! Bonus de ${settings.referredBonus} XOF apres votre premiere course`,
      referrerId,
    };
  }

  /**
   * Complete referral after first ride/delivery
   */
  async completeReferral(
    userId: number,
    userType: 'user' | 'driver',
    bookingId: number,
    merchantId: number,
  ): Promise<void> {
    // Find pending referral
    const referral = await this.prisma.referralTransaction.findFirst({
      where: {
        referred_id: userId,
        referred_type: userType,
        status: 'pending',
        merchant_id: merchantId,
      },
    });

    if (!referral) {
      return; // No pending referral
    }

    // Update referral status
    await this.prisma.referralTransaction.update({
      where: { id: referral.id },
      data: {
        status: 'completed',
        completed_booking_id: bookingId,
        completed_at: new Date(),
      },
    });

    // Credit referrer bonus (to wallet)
    await this.creditWallet(
      referral.referrer_id,
      referral.referrer_type,
      Number(referral.referrer_bonus),
      `Bonus parrainage - ${userType === 'user' ? 'Utilisateur' : 'Chauffeur'} #${userId}`,
      merchantId,
    );

    // Credit referred bonus (to wallet)
    await this.creditWallet(
      userId,
      userType,
      Number(referral.referred_bonus),
      'Bonus premiere course',
      merchantId,
    );

    // Update referrer stats
    if (referral.referrer_type === 'user') {
      await this.prisma.userReferral.updateMany({
        where: { user_id: referral.referrer_id },
        data: {
          total_referrals: { increment: 1 },
          total_earned: { increment: Number(referral.referrer_bonus) },
        },
      });
    } else {
      await this.prisma.driverReferral.updateMany({
        where: { driver_id: referral.referrer_id },
        data: {
          total_referrals: { increment: 1 },
          total_earned: { increment: Number(referral.referrer_bonus) },
        },
      });
    }

    // Notify referrer
    await this.notificationService.sendToUser(
      referral.referrer_id,
      referral.referrer_type,
      {
        title: 'Bonus de parrainage!',
        body: `Vous avez gagne ${referral.referrer_bonus} XOF grace a votre filleul`,
        data: { type: 'referral_bonus' },
      },
    );

    this.logger.log(
      `Referral completed: referrer #${referral.referrer_id} -> referred #${userId}`,
    );
  }

  /**
   * Get user's referral stats
   */
  async getReferralStats(
    userId: number,
    userType: 'user' | 'driver',
    merchantId: number,
  ): Promise<ReferralStats> {
    // Get referral record
    let referral: any;

    if (userType === 'user') {
      referral = await this.prisma.userReferral.findFirst({
        where: { user_id: userId, merchant_id: merchantId },
      });
    } else {
      referral = await this.prisma.driverReferral.findFirst({
        where: { driver_id: userId, merchant_id: merchantId },
      });
    }

    // Count transactions
    const [pendingCount, completedCount] = await Promise.all([
      this.prisma.referralTransaction.count({
        where: {
          referrer_id: userId,
          referrer_type: userType,
          status: 'pending',
        },
      }),
      this.prisma.referralTransaction.count({
        where: {
          referrer_id: userId,
          referrer_type: userType,
          status: 'completed',
        },
      }),
    ]);

    return {
      totalReferrals: referral?.total_referrals || 0,
      pendingReferrals: pendingCount,
      completedReferrals: completedCount,
      totalEarned: referral?.total_earned || 0,
      currency: 'XOF',
    };
  }

  /**
   * Get referral history
   */
  async getReferralHistory(
    userId: number,
    userType: 'user' | 'driver',
    options?: { page?: number; limit?: number },
  ): Promise<{ data: any[]; total: number }> {
    const page = options?.page || 1;
    const limit = options?.limit || 20;
    const skip = (page - 1) * limit;

    const where = {
      referrer_id: userId,
      referrer_type: userType,
    };

    const [data, total] = await Promise.all([
      this.prisma.referralTransaction.findMany({
        where,
        skip,
        take: limit,
        orderBy: { created_at: 'desc' },
      }),
      this.prisma.referralTransaction.count({ where }),
    ]);

    return { data, total };
  }

  /**
   * Get referral settings for merchant
   */
  async getReferralSettings(merchantId: number): Promise<{
    referrerBonus: number;
    referredBonus: number;
    isEnabled: boolean;
  }> {
    const merchant = await this.prisma.merchant.findUnique({
      where: { id: merchantId },
      select: {
        referral_enabled: true,
        referral_referrer_bonus: true,
        referral_referred_bonus: true,
      },
    });

    return {
      referrerBonus: Number(merchant?.referral_referrer_bonus) || 500,
      referredBonus: Number(merchant?.referral_referred_bonus) || 500,
      isEnabled: merchant?.referral_enabled === 1,
    };
  }

  /**
   * Generate unique referral code
   */
  private async generateUniqueCode(merchantId: number): Promise<string> {
    const characters = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789';
    const codeLength = 8;

    let code: string;
    let isUnique = false;

    while (!isUnique) {
      code = '';
      for (let i = 0; i < codeLength; i++) {
        code += characters.charAt(Math.floor(Math.random() * characters.length));
      }

      // Check uniqueness
      const existingUser = await this.prisma.userReferral.findFirst({
        where: { referral_code: code, merchant_id: merchantId },
      });
      const existingDriver = await this.prisma.driverReferral.findFirst({
        where: { referral_code: code, merchant_id: merchantId },
      });

      if (!existingUser && !existingDriver) {
        isUnique = true;
      }
    }

    return code;
  }

  /**
   * Credit wallet balance
   */
  private async creditWallet(
    userId: number,
    userType: 'user' | 'driver',
    amount: number,
    description: string,
    merchantId: number,
  ): Promise<void> {
    if (userType === 'user') {
      // Update user wallet
      await this.prisma.user.update({
        where: { id: userId },
        data: {
          wallet_balance: { increment: amount },
        },
      });

      // Create wallet transaction
      await this.prisma.userWalletTransaction.create({
        data: {
          user_id: userId,
          merchant_id: merchantId,
          type: 'credit',
          amount,
          description,
          status: 'completed',
          created_at: new Date(),
        },
      });
    } else {
      // Update driver wallet
      await this.prisma.driver.update({
        where: { id: userId },
        data: {
          wallet_balance: { increment: amount },
        },
      });

      // Create wallet transaction
      await this.prisma.driverWalletTransaction.create({
        data: {
          driver_id: userId,
          merchant_id: merchantId,
          type: 'credit',
          amount,
          description,
          status: 'completed',
          created_at: new Date(),
        },
      });
    }
  }
}
