import { Injectable, Logger, BadRequestException, NotFoundException } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';

interface RatingData {
  bookingId: number;
  rating: number;
  comment?: string;
  tags?: string[];
}

interface ReviewResponse {
  id: number;
  bookingId: number;
  rating: number;
  comment: string;
  tags: string[];
  createdAt: Date;
  reviewer: { id: number; name: string; photo?: string };
  response?: { comment: string; createdAt: Date };
}

@Injectable()
export class RatingService {
  private readonly logger = new Logger(RatingService.name);

  // Predefined rating tags
  private readonly POSITIVE_TAGS = {
    user: ['Ponctuel', 'Poli', 'Propre', 'Bonne conversation', 'Recommande'],
    driver: ['Conduite sure', 'Vehicule propre', 'Ponctuel', 'Agreable', 'Route optimale'],
  };

  private readonly NEGATIVE_TAGS = {
    user: ['En retard', 'Impoli', 'Sale', 'Problematique'],
    driver: ['Conduite dangereuse', 'Vehicule sale', 'En retard', 'Desagreable', 'Mauvaise route'],
  };

  constructor(private prisma: PrismaService) {}

  /**
   * Submit rating from user to driver
   */
  async rateDriver(
    userId: number,
    data: RatingData,
  ): Promise<ReviewResponse> {
    // Verify booking
    const booking = await this.prisma.booking.findFirst({
      where: {
        id: data.bookingId,
        user_id: userId,
        booking_status: 'completed',
      },
    });

    if (!booking) {
      throw new NotFoundException('Course non trouvee ou non completee');
    }

    if (!booking.driver_id) {
      throw new BadRequestException('Pas de chauffeur assigne');
    }

    // Check if already rated
    const existingRating = await this.prisma.userRating.findFirst({
      where: {
        booking_id: data.bookingId,
        from_user_id: userId,
      },
    });

    if (existingRating) {
      throw new BadRequestException('Vous avez deja evalue cette course');
    }

    // Validate rating
    if (data.rating < 1 || data.rating > 5) {
      throw new BadRequestException('Note invalide (1-5)');
    }

    // Create rating
    const rating = await this.prisma.userRating.create({
      data: {
        booking_id: data.bookingId,
        from_user_id: userId,
        to_driver_id: booking.driver_id,
        rating: data.rating,
        comment: data.comment,
        tags: data.tags ? JSON.stringify(data.tags) : null,
        rating_type: 'user_to_driver',
        created_at: new Date(),
      },
    });

    // Update driver average rating
    await this.updateDriverRating(booking.driver_id);

    // Update booking
    await this.prisma.booking.update({
      where: { id: data.bookingId },
      data: { user_rating: data.rating },
    });

    this.logger.log(`User #${userId} rated driver #${booking.driver_id}: ${data.rating}/5`);

    return this.formatRatingResponse(rating);
  }

  /**
   * Submit rating from driver to user
   */
  async rateUser(
    driverId: number,
    data: RatingData,
  ): Promise<ReviewResponse> {
    // Verify booking
    const booking = await this.prisma.booking.findFirst({
      where: {
        id: data.bookingId,
        driver_id: driverId,
        booking_status: 'completed',
      },
    });

    if (!booking) {
      throw new NotFoundException('Course non trouvee ou non completee');
    }

    // Check if already rated
    const existingRating = await this.prisma.userRating.findFirst({
      where: {
        booking_id: data.bookingId,
        from_driver_id: driverId,
      },
    });

    if (existingRating) {
      throw new BadRequestException('Vous avez deja evalue cette course');
    }

    // Create rating
    const rating = await this.prisma.userRating.create({
      data: {
        booking_id: data.bookingId,
        from_driver_id: driverId,
        to_user_id: booking.user_id,
        rating: data.rating,
        comment: data.comment,
        tags: data.tags ? JSON.stringify(data.tags) : null,
        rating_type: 'driver_to_user',
        created_at: new Date(),
      },
    });

    // Update user average rating
    await this.updateUserRating(booking.user_id);

    // Update booking
    await this.prisma.booking.update({
      where: { id: data.bookingId },
      data: { driver_rating: data.rating },
    });

    this.logger.log(`Driver #${driverId} rated user #${booking.user_id}: ${data.rating}/5`);

    return this.formatRatingResponse(rating);
  }

  /**
   * Get driver reviews
   */
  async getDriverReviews(
    driverId: number,
    options?: { page?: number; limit?: number; minRating?: number },
  ): Promise<{ data: ReviewResponse[]; total: number; average: number }> {
    const page = options?.page || 1;
    const limit = options?.limit || 20;
    const skip = (page - 1) * limit;

    const where: any = { to_driver_id: driverId };
    if (options?.minRating) {
      where.rating = { gte: options.minRating };
    }

    const [ratings, total, avgResult] = await Promise.all([
      this.prisma.userRating.findMany({
        where,
        skip,
        take: limit,
        orderBy: { created_at: 'desc' },
        include: {
          from_user: {
            select: { id: true, first_name: true, last_name: true, profile_picture: true },
          },
          rating_response: true,
        },
      }),
      this.prisma.userRating.count({ where }),
      this.prisma.userRating.aggregate({
        where: { to_driver_id: driverId },
        _avg: { rating: true },
      }),
    ]);

    return {
      data: ratings.map((r) => this.formatRatingResponse(r)),
      total,
      average: Math.round((avgResult._avg?.rating || 0) * 10) / 10,
    };
  }

  /**
   * Get user reviews
   */
  async getUserReviews(
    userId: number,
    options?: { page?: number; limit?: number },
  ): Promise<{ data: ReviewResponse[]; total: number; average: number }> {
    const page = options?.page || 1;
    const limit = options?.limit || 20;
    const skip = (page - 1) * limit;

    const [ratings, total, avgResult] = await Promise.all([
      this.prisma.userRating.findMany({
        where: { to_user_id: userId },
        skip,
        take: limit,
        orderBy: { created_at: 'desc' },
        include: {
          from_driver: {
            select: { id: true, first_name: true, last_name: true, profile_picture: true },
          },
        },
      }),
      this.prisma.userRating.count({ where: { to_user_id: userId } }),
      this.prisma.userRating.aggregate({
        where: { to_user_id: userId },
        _avg: { rating: true },
      }),
    ]);

    return {
      data: ratings.map((r) => this.formatRatingResponse(r)),
      total,
      average: Math.round((avgResult._avg?.rating || 0) * 10) / 10,
    };
  }

  /**
   * Driver responds to a review
   */
  async respondToReview(
    driverId: number,
    ratingId: number,
    comment: string,
  ): Promise<any> {
    const rating = await this.prisma.userRating.findFirst({
      where: { id: ratingId, to_driver_id: driverId },
    });

    if (!rating) {
      throw new NotFoundException('Evaluation non trouvee');
    }

    // Check if response already exists
    const existing = await this.prisma.ratingResponse.findFirst({
      where: { rating_id: ratingId },
    });

    if (existing) {
      throw new BadRequestException('Vous avez deja repondu a cette evaluation');
    }

    return this.prisma.ratingResponse.create({
      data: {
        rating_id: ratingId,
        comment,
        created_at: new Date(),
      },
    });
  }

  /**
   * Report a review (flag as inappropriate)
   */
  async reportReview(
    reporterId: number,
    reporterType: 'user' | 'driver',
    ratingId: number,
    reason: string,
  ): Promise<void> {
    const rating = await this.prisma.userRating.findUnique({
      where: { id: ratingId },
    });

    if (!rating) {
      throw new NotFoundException('Evaluation non trouvee');
    }

    await this.prisma.ratingReport.create({
      data: {
        rating_id: ratingId,
        reporter_id: reporterId,
        reporter_type: reporterType,
        reason,
        status: 'pending',
        created_at: new Date(),
      },
    });

    this.logger.log(`Review #${ratingId} reported by ${reporterType} #${reporterId}`);
  }

  /**
   * Get rating tags for UI
   */
  getRatingTags(ratingValue: number, forType: 'user' | 'driver'): string[] {
    if (ratingValue >= 4) {
      return this.POSITIVE_TAGS[forType];
    }
    return this.NEGATIVE_TAGS[forType];
  }

  /**
   * Get rating statistics for driver
   */
  async getDriverRatingStats(driverId: number): Promise<{
    average: number;
    total: number;
    distribution: { [key: number]: number };
    recentTrend: 'up' | 'down' | 'stable';
  }> {
    const ratings = await this.prisma.userRating.findMany({
      where: { to_driver_id: driverId },
      select: { rating: true, created_at: true },
      orderBy: { created_at: 'desc' },
    });

    if (ratings.length === 0) {
      return {
        average: 0,
        total: 0,
        distribution: { 1: 0, 2: 0, 3: 0, 4: 0, 5: 0 },
        recentTrend: 'stable',
      };
    }

    // Calculate distribution
    const distribution = { 1: 0, 2: 0, 3: 0, 4: 0, 5: 0 };
    let sum = 0;
    for (const r of ratings) {
      distribution[r.rating]++;
      sum += r.rating;
    }

    // Calculate recent trend (last 10 vs previous 10)
    let recentTrend: 'up' | 'down' | 'stable' = 'stable';
    if (ratings.length >= 20) {
      const recent10 = ratings.slice(0, 10).reduce((s, r) => s + r.rating, 0) / 10;
      const previous10 = ratings.slice(10, 20).reduce((s, r) => s + r.rating, 0) / 10;
      if (recent10 > previous10 + 0.2) recentTrend = 'up';
      else if (recent10 < previous10 - 0.2) recentTrend = 'down';
    }

    return {
      average: Math.round((sum / ratings.length) * 10) / 10,
      total: ratings.length,
      distribution,
      recentTrend,
    };
  }

  /**
   * Update driver average rating
   */
  private async updateDriverRating(driverId: number): Promise<void> {
    const result = await this.prisma.userRating.aggregate({
      where: { to_driver_id: driverId },
      _avg: { rating: true },
      _count: true,
    });

    await this.prisma.driver.update({
      where: { id: driverId },
      data: {
        rating: result._avg?.rating || 0,
        total_ratings: result._count || 0,
      },
    });
  }

  /**
   * Update user average rating
   */
  private async updateUserRating(userId: number): Promise<void> {
    const result = await this.prisma.userRating.aggregate({
      where: { to_user_id: userId },
      _avg: { rating: true },
      _count: true,
    });

    await this.prisma.user.update({
      where: { id: userId },
      data: {
        rating: result._avg?.rating || 0,
        total_ratings: result._count || 0,
      },
    });
  }

  /**
   * Format rating response
   */
  private formatRatingResponse(rating: any): ReviewResponse {
    const reviewer = rating.from_user || rating.from_driver;

    return {
      id: rating.id,
      bookingId: rating.booking_id,
      rating: rating.rating,
      comment: rating.comment || '',
      tags: rating.tags ? JSON.parse(rating.tags) : [],
      createdAt: rating.created_at,
      reviewer: {
        id: reviewer?.id,
        name: `${reviewer?.first_name || ''} ${reviewer?.last_name || ''}`.trim(),
        photo: reviewer?.profile_picture,
      },
      response: rating.rating_response
        ? {
            comment: rating.rating_response.comment,
            createdAt: rating.rating_response.created_at,
          }
        : undefined,
    };
  }
}
