import {
  Controller,
  Get,
  Post,
  Body,
  Param,
  Query,
  UseGuards,
  ParseIntPipe,
} from '@nestjs/common';
import { ApiTags, ApiOperation, ApiBearerAuth } from '@nestjs/swagger';
import { RatingService } from './rating.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantGuard } from '../../common/guards/merchant.guard';
import { CurrentUser } from '../../common/decorators/merchant.decorator';
import { Public } from '../../common/decorators/public.decorator';

@ApiTags('Rating')
@Controller('api/rating')
@UseGuards(JwtAuthGuard, MerchantGuard)
@ApiBearerAuth()
export class RatingController {
  constructor(private ratingService: RatingService) {}

  // ============================================================================
  // USER ENDPOINTS
  // ============================================================================

  @Post('driver')
  @ApiOperation({ summary: 'Rate a driver (user)' })
  async rateDriver(
    @CurrentUser() user: any,
    @Body()
    body: {
      bookingId: number;
      rating: number;
      comment?: string;
      tags?: string[];
    },
  ) {
    return this.ratingService.rateDriver(user.id, body);
  }

  @Get('driver/:id/reviews')
  @ApiOperation({ summary: 'Get driver reviews' })
  async getDriverReviews(
    @Param('id', ParseIntPipe) driverId: number,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
    @Query('min_rating') minRating?: string,
  ) {
    return this.ratingService.getDriverReviews(driverId, {
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 20,
      minRating: minRating ? parseInt(minRating) : undefined,
    });
  }

  @Get('my-reviews')
  @ApiOperation({ summary: 'Get reviews I received (user)' })
  async getMyReviews(
    @CurrentUser() user: any,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
  ) {
    return this.ratingService.getUserReviews(user.id, {
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 20,
    });
  }

  @Post('report/:id')
  @ApiOperation({ summary: 'Report a review' })
  async reportReview(
    @Param('id', ParseIntPipe) ratingId: number,
    @CurrentUser() user: any,
    @Body() body: { reason: string },
  ) {
    const userType = user.type || 'user';
    await this.ratingService.reportReview(user.id, userType, ratingId, body.reason);
    return { message: 'Signalement envoye' };
  }

  @Get('tags')
  @ApiOperation({ summary: 'Get rating tags for a rating value' })
  async getRatingTags(
    @Query('rating') rating: string,
    @Query('for') forType: 'user' | 'driver',
  ) {
    return {
      tags: this.ratingService.getRatingTags(
        parseInt(rating),
        forType || 'driver',
      ),
    };
  }

  // ============================================================================
  // DRIVER ENDPOINTS
  // ============================================================================

  @Post('user')
  @ApiOperation({ summary: 'Rate a user (driver)' })
  async rateUser(
    @CurrentUser() driver: any,
    @Body()
    body: {
      bookingId: number;
      rating: number;
      comment?: string;
      tags?: string[];
    },
  ) {
    return this.ratingService.rateUser(driver.id, body);
  }

  @Get('driver/my-reviews')
  @ApiOperation({ summary: 'Get reviews I received (driver)' })
  async getDriverMyReviews(
    @CurrentUser() driver: any,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
  ) {
    return this.ratingService.getDriverReviews(driver.id, {
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 20,
    });
  }

  @Get('driver/stats')
  @ApiOperation({ summary: 'Get my rating statistics (driver)' })
  async getDriverStats(@CurrentUser() driver: any) {
    return this.ratingService.getDriverRatingStats(driver.id);
  }

  @Post('respond/:id')
  @ApiOperation({ summary: 'Respond to a review (driver)' })
  async respondToReview(
    @Param('id', ParseIntPipe) ratingId: number,
    @CurrentUser() driver: any,
    @Body() body: { comment: string },
  ) {
    return this.ratingService.respondToReview(driver.id, ratingId, body.comment);
  }
}
