import { Injectable, Logger, OnModuleInit } from '@nestjs/common';
import { InjectQueue } from '@nestjs/bull';
import { Queue, JobOptions } from 'bull';
import { QUEUE_NAMES } from './queue.module';

@Injectable()
export class QueueService implements OnModuleInit {
  private readonly logger = new Logger(QueueService.name);

  constructor(
    @InjectQueue(QUEUE_NAMES.BOOKING) private bookingQueue: Queue,
    @InjectQueue(QUEUE_NAMES.NOTIFICATION) private notificationQueue: Queue,
    @InjectQueue(QUEUE_NAMES.EMAIL) private emailQueue: Queue,
    @InjectQueue(QUEUE_NAMES.DRIVER_SEARCH) private driverSearchQueue: Queue,
    @InjectQueue(QUEUE_NAMES.PAYMENT) private paymentQueue: Queue,
    @InjectQueue(QUEUE_NAMES.SCHEDULED_TASKS) private scheduledTasksQueue: Queue,
    @InjectQueue(QUEUE_NAMES.SMS) private smsQueue: Queue,
    @InjectQueue(QUEUE_NAMES.WHATSAPP) private whatsappQueue: Queue,
  ) {}

  async onModuleInit() {
    // Configurer les tâches récurrentes
    await this.setupRecurringJobs();
  }

  /**
   * Configurer les tâches planifiées récurrentes
   */
  private async setupRecurringJobs() {
    // Nettoyer les anciennes tâches répétitives
    const repeatableJobs = await this.scheduledTasksQueue.getRepeatableJobs();
    for (const job of repeatableJobs) {
      await this.scheduledTasksQueue.removeRepeatableByKey(job.key);
    }

    // Vérifier les réservations programmées toutes les minutes
    await this.scheduledTasksQueue.add(
      'process-scheduled-bookings',
      {},
      { repeat: { cron: '* * * * *' } }, // Chaque minute
    );

    // Envoyer les rappels toutes les 10 minutes
    await this.scheduledTasksQueue.add(
      'send-booking-reminders',
      {},
      { repeat: { cron: '*/10 * * * *' } },
    );

    // Nettoyer les réservations expirées toutes les 5 minutes
    await this.scheduledTasksQueue.add(
      'cleanup-expired-bookings',
      {},
      { repeat: { cron: '*/5 * * * *' } },
    );

    // Mettre à jour les stats chauffeurs chaque heure
    await this.scheduledTasksQueue.add(
      'update-driver-daily-stats',
      {},
      { repeat: { cron: '0 * * * *' } },
    );

    // Vérifier les documents expirés chaque jour à 6h
    await this.scheduledTasksQueue.add(
      'check-expiring-documents',
      {},
      { repeat: { cron: '0 6 * * *' } },
    );

    // Envoyer les résumés quotidiens à 8h
    await this.scheduledTasksQueue.add(
      'send-daily-summary',
      {},
      { repeat: { cron: '0 8 * * *' } },
    );

    // Désactiver les promos expirées à minuit
    await this.scheduledTasksQueue.add(
      'expire-promo-codes',
      {},
      { repeat: { cron: '0 0 * * *' } },
    );

    this.logger.log('Tâches récurrentes configurées');
  }

  // ============================================================================
  // BOOKING JOBS
  // ============================================================================

  /**
   * Queue driver assignment job
   */
  async queueDriverAssignment(bookingId: number, merchantId: number) {
    const job = await this.bookingQueue.add(
      'assign-driver',
      { bookingId, merchantId },
      {
        attempts: 5,
        backoff: { type: 'fixed', delay: 5000 }, // Retry every 5 seconds
        timeout: 60000, // 1 minute timeout
      },
    );
    this.logger.log(`Queued driver assignment for booking #${bookingId}, job: ${job.id}`);
    return job;
  }

  /**
   * Queue booking timeout check
   */
  async queueBookingTimeout(bookingId: number, timeoutSeconds: number) {
    const job = await this.bookingQueue.add(
      'check-timeout',
      { bookingId },
      {
        delay: timeoutSeconds * 1000,
        attempts: 1,
      },
    );
    this.logger.log(`Queued timeout check for booking #${bookingId} in ${timeoutSeconds}s`);
    return job;
  }

  /**
   * Queue booking completion tasks (invoice, rewards, etc.)
   */
  async queueBookingCompletion(bookingId: number) {
    const job = await this.bookingQueue.add(
      'complete-booking',
      { bookingId },
      { priority: 1 }, // High priority
    );
    this.logger.log(`Queued completion tasks for booking #${bookingId}`);
    return job;
  }

  /**
   * Queue driver location update processing
   */
  async queueDriverLocationUpdate(
    bookingId: number,
    driverId: number,
    location: { latitude: number; longitude: number },
  ) {
    // Use a shorter job ID to prevent duplicate processing
    const jobId = `driver-loc-${bookingId}-${driverId}`;

    const job = await this.bookingQueue.add(
      'update-driver-location',
      { bookingId, driverId, location },
      {
        jobId,
        removeOnComplete: true,
        removeOnFail: true,
      },
    );
    return job;
  }

  // ============================================================================
  // NOTIFICATION JOBS
  // ============================================================================

  /**
   * Queue push notification
   */
  async queuePushNotification(
    userId: number,
    userType: 'user' | 'driver',
    title: string,
    body: string,
    data?: Record<string, string>,
  ) {
    const job = await this.notificationQueue.add(
      'send-push',
      { userId, userType, title, body, data },
      { attempts: 3 },
    );
    this.logger.log(`Queued push notification for ${userType} #${userId}`);
    return job;
  }

  /**
   * Queue SMS notification
   */
  async queueSmsNotification(phone: string, message: string) {
    const job = await this.notificationQueue.add(
      'send-sms',
      { phone, message },
      { attempts: 2 },
    );
    this.logger.log(`Queued SMS to ${phone}`);
    return job;
  }

  /**
   * Queue bulk notification (to multiple users)
   */
  async queueBulkNotification(
    userIds: number[],
    userType: 'user' | 'driver',
    title: string,
    body: string,
    data?: Record<string, string>,
  ) {
    const job = await this.notificationQueue.add(
      'send-bulk-push',
      { userIds, userType, title, body, data },
      {
        attempts: 2,
        timeout: 300000, // 5 minutes for bulk
      },
    );
    this.logger.log(`Queued bulk notification to ${userIds.length} ${userType}s`);
    return job;
  }

  // ============================================================================
  // EMAIL JOBS
  // ============================================================================

  /**
   * Queue email
   */
  async queueEmail(
    to: string,
    subject: string,
    template: string,
    context: Record<string, any>,
  ) {
    const job = await this.emailQueue.add(
      'send-email',
      { to, subject, template, context },
      { attempts: 3 },
    );
    this.logger.log(`Queued email to ${to}: ${subject}`);
    return job;
  }

  /**
   * Queue booking invoice email
   */
  async queueInvoiceEmail(bookingId: number, userEmail: string) {
    const job = await this.emailQueue.add(
      'send-invoice',
      { bookingId, userEmail },
      { priority: 2 },
    );
    this.logger.log(`Queued invoice email for booking #${bookingId}`);
    return job;
  }

  /**
   * Queue welcome email
   */
  async queueWelcomeEmail(
    userId: number,
    userType: 'user' | 'driver',
    email: string,
    name: string,
  ) {
    const job = await this.emailQueue.add(
      'send-welcome',
      { userId, userType, email, name },
      { delay: 5000 }, // Send after 5 seconds
    );
    this.logger.log(`Queued welcome email to ${email}`);
    return job;
  }

  // ============================================================================
  // DRIVER SEARCH JOBS
  // ============================================================================

  /**
   * Lancer la recherche de chauffeur
   */
  async queueDriverSearch(
    bookingId: number,
    merchantId: number,
    vehicleTypeId: number,
    pickupLatitude: number,
    pickupLongitude: number,
    userId: number,
    initialRadius = 3,
    maxRadius = 15,
  ) {
    const job = await this.driverSearchQueue.add(
      'search-drivers',
      {
        bookingId,
        merchantId,
        vehicleTypeId,
        pickupLatitude,
        pickupLongitude,
        currentRadius: initialRadius,
        maxRadius,
        attempt: 1,
        excludedDriverIds: [],
        userId,
      },
      {
        attempts: 1, // Pas de retry automatique, géré manuellement
        timeout: 60000,
      },
    );
    this.logger.log(`Recherche chauffeur lancée pour réservation #${bookingId}`);
    return job;
  }

  /**
   * Traiter la réponse d'un chauffeur
   */
  async queueDriverResponse(
    bookingId: number,
    driverId: number,
    accepted: boolean,
    merchantId: number,
  ) {
    const job = await this.driverSearchQueue.add(
      'driver-response',
      { bookingId, driverId, accepted, merchantId },
      { priority: 1 }, // Haute priorité
    );
    this.logger.log(
      `Réponse chauffeur #${driverId} mise en queue pour réservation #${bookingId}`,
    );
    return job;
  }

  /**
   * Vérifier le timeout des demandes chauffeurs
   */
  async queueRequestTimeout(
    bookingId: number,
    driverIds: number[],
    merchantId: number,
    nextRadius: number,
    maxRadius: number,
    excludedDriverIds: number[],
    delayMs = 30000,
  ) {
    const job = await this.driverSearchQueue.add(
      'check-request-timeout',
      { bookingId, driverIds, merchantId, nextRadius, maxRadius, excludedDriverIds },
      { delay: delayMs },
    );
    return job;
  }

  // ============================================================================
  // PAYMENT JOBS
  // ============================================================================

  /**
   * Traiter un paiement
   */
  async queuePayment(
    bookingId: number,
    userId: number,
    driverId: number,
    amount: number,
    currency: string,
    paymentMethod: string,
    merchantId: number,
    paymentMethodId?: string,
  ) {
    const job = await this.paymentQueue.add(
      'process-payment',
      {
        bookingId,
        userId,
        driverId,
        amount,
        currency,
        paymentMethod,
        paymentMethodId,
        merchantId,
      },
      {
        attempts: 3,
        backoff: { type: 'exponential', delay: 2000 },
      },
    );
    this.logger.log(`Paiement mis en queue pour réservation #${bookingId}`);
    return job;
  }

  /**
   * Traiter un remboursement
   */
  async queueRefund(
    bookingId: number,
    transactionId: number,
    amount: number,
    reason: string,
  ) {
    const job = await this.paymentQueue.add(
      'process-refund',
      { bookingId, transactionId, amount, reason },
      { attempts: 3 },
    );
    this.logger.log(`Remboursement mis en queue pour réservation #${bookingId}`);
    return job;
  }

  /**
   * Distribuer les gains chauffeur
   */
  async queueEarningsDistribution(
    bookingId: number,
    driverId: number,
    totalAmount: number,
    merchantId: number,
    commissionRate?: number,
  ) {
    const job = await this.paymentQueue.add(
      'distribute-earnings',
      { bookingId, driverId, totalAmount, commissionRate, merchantId },
      { priority: 2 },
    );
    return job;
  }

  /**
   * Traiter un pourboire
   */
  async queueTip(
    bookingId: number,
    driverId: number,
    userId: number,
    amount: number,
    currency: string,
  ) {
    const job = await this.paymentQueue.add(
      'process-tip',
      { bookingId, driverId, userId, amount, currency },
    );
    this.logger.log(`Pourboire mis en queue pour réservation #${bookingId}`);
    return job;
  }

  // ============================================================================
  // SMS JOBS
  // ============================================================================

  /**
   * Envoyer un SMS
   */
  async queueSms(phone: string, message: string) {
    const job = await this.smsQueue.add(
      'send-sms',
      { phone, message },
      { attempts: 2 },
    );
    this.logger.log(`SMS mis en queue pour ${phone}`);
    return job;
  }

  /**
   * Envoyer un OTP par SMS
   */
  async queueOtpSms(phone: string, otp: string, merchantName?: string) {
    const job = await this.smsQueue.add(
      'send-otp',
      { phone, otp, merchantName },
      { attempts: 3, priority: 1 }, // Priorité haute pour les OTP
    );
    return job;
  }

  // ============================================================================
  // WHATSAPP JOBS
  // ============================================================================

  /**
   * Envoyer un message WhatsApp
   */
  async queueWhatsAppMessage(phone: string, message: string) {
    const job = await this.whatsappQueue.add(
      'send-message',
      { phone, message },
      { attempts: 2 },
    );
    this.logger.log(`WhatsApp mis en queue pour ${phone}`);
    return job;
  }

  /**
   * Envoyer un template WhatsApp
   */
  async queueWhatsAppTemplate(
    phone: string,
    templateName: string,
    language: string,
    parameters: string[],
  ) {
    const job = await this.whatsappQueue.add(
      'send-template',
      { phone, templateName, language, parameters },
      { attempts: 2 },
    );
    return job;
  }

  /**
   * Envoyer la confirmation de réservation via WhatsApp
   */
  async queueWhatsAppBookingConfirmation(
    phone: string,
    driverName: string,
    vehiclePlate: string,
    vehicleModel: string,
    eta: number,
  ) {
    const job = await this.whatsappQueue.add(
      'send-booking-confirmation',
      { phone, driverName, vehiclePlate, vehicleModel, eta },
    );
    return job;
  }

  // ============================================================================
  // SCHEDULED TASKS
  // ============================================================================

  /**
   * Générer un rapport hebdomadaire pour un merchant
   */
  async queueWeeklyReport(merchantId: number) {
    const job = await this.scheduledTasksQueue.add(
      'generate-weekly-report',
      { merchantId },
    );
    return job;
  }

  // ============================================================================
  // QUEUE STATS
  // ============================================================================

  async getQueueStats() {
    const [
      booking,
      notification,
      email,
      driverSearch,
      payment,
      scheduledTasks,
      sms,
      whatsapp,
    ] = await Promise.all([
      this.getQueueInfo(this.bookingQueue),
      this.getQueueInfo(this.notificationQueue),
      this.getQueueInfo(this.emailQueue),
      this.getQueueInfo(this.driverSearchQueue),
      this.getQueueInfo(this.paymentQueue),
      this.getQueueInfo(this.scheduledTasksQueue),
      this.getQueueInfo(this.smsQueue),
      this.getQueueInfo(this.whatsappQueue),
    ]);

    return {
      booking,
      notification,
      email,
      driverSearch,
      payment,
      scheduledTasks,
      sms,
      whatsapp,
    };
  }

  /**
   * Vider toutes les queues (utiliser avec précaution)
   */
  async clearAllQueues() {
    await Promise.all([
      this.bookingQueue.empty(),
      this.notificationQueue.empty(),
      this.emailQueue.empty(),
      this.driverSearchQueue.empty(),
      this.paymentQueue.empty(),
      this.smsQueue.empty(),
      this.whatsappQueue.empty(),
    ]);
    this.logger.warn('Toutes les queues ont été vidées');
  }

  /**
   * Mettre en pause une queue
   */
  async pauseQueue(queueName: string) {
    const queue = this.getQueueByName(queueName);
    if (queue) {
      await queue.pause();
      this.logger.log(`Queue ${queueName} mise en pause`);
    }
  }

  /**
   * Reprendre une queue
   */
  async resumeQueue(queueName: string) {
    const queue = this.getQueueByName(queueName);
    if (queue) {
      await queue.resume();
      this.logger.log(`Queue ${queueName} reprise`);
    }
  }

  private getQueueByName(name: string): Queue | null {
    switch (name) {
      case QUEUE_NAMES.BOOKING:
        return this.bookingQueue;
      case QUEUE_NAMES.NOTIFICATION:
        return this.notificationQueue;
      case QUEUE_NAMES.EMAIL:
        return this.emailQueue;
      case QUEUE_NAMES.DRIVER_SEARCH:
        return this.driverSearchQueue;
      case QUEUE_NAMES.PAYMENT:
        return this.paymentQueue;
      case QUEUE_NAMES.SCHEDULED_TASKS:
        return this.scheduledTasksQueue;
      case QUEUE_NAMES.SMS:
        return this.smsQueue;
      case QUEUE_NAMES.WHATSAPP:
        return this.whatsappQueue;
      default:
        return null;
    }
  }

  private async getQueueInfo(queue: Queue) {
    const [waiting, active, completed, failed, delayed, paused] = await Promise.all([
      queue.getWaitingCount(),
      queue.getActiveCount(),
      queue.getCompletedCount(),
      queue.getFailedCount(),
      queue.getDelayedCount(),
      queue.isPaused(),
    ]);

    return { waiting, active, completed, failed, delayed, paused };
  }
}
