import { Module } from '@nestjs/common';
import { BullModule } from '@nestjs/bull';
import { ConfigModule, ConfigService } from '@nestjs/config';

import { BookingQueueProcessor } from './processors/booking.processor';
import { NotificationQueueProcessor } from './processors/notification.processor';
import { EmailQueueProcessor } from './processors/email.processor';
import { DriverSearchProcessor } from './processors/driver-search.processor';
import { PaymentProcessor } from './processors/payment.processor';
import { ScheduledTasksProcessor } from './processors/scheduled-tasks.processor';
import { QueueService } from './queue.service';
import { PrismaModule } from '../../shared/database/prisma.module';
import { NotificationModule } from '../notification/notification.module';
import { WebsocketModule } from '../websocket/websocket.module';

// Queue names
export const QUEUE_NAMES = {
  BOOKING: 'booking',
  NOTIFICATION: 'notification',
  EMAIL: 'email',
  DRIVER_SEARCH: 'driver-search',
  PAYMENT: 'payment',
  SCHEDULED_TASKS: 'scheduled-tasks',
  SMS: 'sms',
  WHATSAPP: 'whatsapp',
} as const;

@Module({
  imports: [
    ConfigModule,
    PrismaModule,
    NotificationModule,
    WebsocketModule,

    // Configure Bull with Redis
    BullModule.forRootAsync({
      imports: [ConfigModule],
      inject: [ConfigService],
      useFactory: async (configService: ConfigService) => ({
        redis: {
          host: configService.get<string>('REDIS_HOST', 'localhost'),
          port: configService.get<number>('REDIS_PORT', 6379),
          password: configService.get<string>('REDIS_PASSWORD'),
          db: configService.get<number>('REDIS_QUEUE_DB', 1),
        },
        defaultJobOptions: {
          removeOnComplete: 100,
          removeOnFail: 500,
          attempts: 3,
          backoff: {
            type: 'exponential',
            delay: 1000,
          },
        },
      }),
    }),

    // Register queues
    BullModule.registerQueue(
      { name: QUEUE_NAMES.BOOKING },
      { name: QUEUE_NAMES.NOTIFICATION },
      { name: QUEUE_NAMES.EMAIL },
      { name: QUEUE_NAMES.DRIVER_SEARCH },
      { name: QUEUE_NAMES.PAYMENT },
      { name: QUEUE_NAMES.SCHEDULED_TASKS },
      { name: QUEUE_NAMES.SMS },
      { name: QUEUE_NAMES.WHATSAPP },
    ),
  ],
  providers: [
    QueueService,
    BookingQueueProcessor,
    NotificationQueueProcessor,
    EmailQueueProcessor,
    DriverSearchProcessor,
    PaymentProcessor,
    ScheduledTasksProcessor,
  ],
  exports: [QueueService, BullModule],
})
export class QueueModule {}
