import { Process, Processor, OnQueueFailed } from '@nestjs/bull';
import { Logger } from '@nestjs/common';
import { Job } from 'bull';
import { PrismaService } from '../../../shared/database/prisma.service';
import { NotificationService } from '../../notification/notification.service';
import { QUEUE_NAMES } from '../queue.module';
import { BOOKING_STATUS } from '../../booking/booking.constants';
import { InjectQueue } from '@nestjs/bull';
import { Queue } from 'bull';

@Processor(QUEUE_NAMES.SCHEDULED_TASKS)
export class ScheduledTasksProcessor {
  private readonly logger = new Logger(ScheduledTasksProcessor.name);

  constructor(
    private prisma: PrismaService,
    private notificationService: NotificationService,
    @InjectQueue(QUEUE_NAMES.DRIVER_SEARCH) private driverSearchQueue: Queue,
    @InjectQueue(QUEUE_NAMES.EMAIL) private emailQueue: Queue,
  ) {}

  @OnQueueFailed()
  onFailed(job: Job, error: Error) {
    this.logger.error(
      `Job planifié ${job.name} (${job.id}) échoué: ${error.message}`,
      error.stack,
    );
  }

  /**
   * Lancer les réservations programmées
   */
  @Process('process-scheduled-bookings')
  async handleScheduledBookings(job: Job) {
    this.logger.log('Traitement des réservations programmées...');

    try {
      // Trouver les réservations programmées qui doivent démarrer dans 15 minutes
      const fifteenMinutesFromNow = new Date(Date.now() + 15 * 60 * 1000);
      const fiveMinutesFromNow = new Date(Date.now() + 5 * 60 * 1000);

      const scheduledBookings = await this.prisma.booking.findMany({
        where: {
          booking_status: BOOKING_STATUS.SCHEDULED,
          schedule_time: {
            gte: fiveMinutesFromNow,
            lte: fifteenMinutesFromNow,
          },
        },
        include: {
          user: true,
        },
      });

      this.logger.log(`${scheduledBookings.length} réservations programmées à traiter`);

      for (const booking of scheduledBookings) {
        try {
          // Lancer la recherche de chauffeur
          await this.prisma.booking.update({
            where: { id: booking.id },
            data: { booking_status: BOOKING_STATUS.SEARCHING },
          });

          await this.driverSearchQueue.add('search-drivers', {
            bookingId: booking.id,
            merchantId: booking.merchant_id,
            vehicleTypeId: booking.vehicle_type_id,
            pickupLatitude: booking.pickup_latitude,
            pickupLongitude: booking.pickup_longitude,
            currentRadius: 3,
            maxRadius: 15,
            attempt: 1,
            excludedDriverIds: [],
            userId: booking.user_id,
          });

          // Notifier l'utilisateur
          await this.notificationService.sendToUser(
            booking.user_id,
            {
              title: 'Recherche de chauffeur',
              body: 'Nous cherchons un chauffeur pour votre course programmée',
              data: { type: 'scheduled_search', booking_id: String(booking.id) },
            },
            { push: true },
          );

          this.logger.log(`Recherche lancée pour réservation programmée #${booking.id}`);
        } catch (error) {
          this.logger.error(
            `Erreur traitement réservation programmée #${booking.id}: ${error.message}`,
          );
        }
      }

      return { processed: scheduledBookings.length };
    } catch (error) {
      this.logger.error(`Erreur traitement réservations programmées: ${error.message}`);
      throw error;
    }
  }

  /**
   * Envoyer les rappels de réservation
   */
  @Process('send-booking-reminders')
  async handleBookingReminders(job: Job) {
    this.logger.log('Envoi des rappels de réservation...');

    try {
      // Réservations programmées dans 1 heure
      const oneHourFromNow = new Date(Date.now() + 60 * 60 * 1000);
      const fiftyMinutesFromNow = new Date(Date.now() + 50 * 60 * 1000);

      const upcomingBookings = await this.prisma.booking.findMany({
        where: {
          booking_status: BOOKING_STATUS.SCHEDULED,
          schedule_time: {
            gte: fiftyMinutesFromNow,
            lte: oneHourFromNow,
          },
          // Éviter d'envoyer plusieurs fois
          reminder_sent: { not: 1 },
        },
        include: { user: true },
      });

      for (const booking of upcomingBookings) {
        await this.notificationService.sendToUser(
          booking.user_id,
          {
            title: 'Rappel: Course dans 1 heure',
            body: `Votre course programmée vers ${booking.drop_address} commence bientôt`,
            data: { type: 'reminder', booking_id: String(booking.id) },
          },
          { push: true },
        );

        await this.prisma.booking.update({
          where: { id: booking.id },
          data: { reminder_sent: 1 },
        });
      }

      return { reminders_sent: upcomingBookings.length };
    } catch (error) {
      this.logger.error(`Erreur envoi rappels: ${error.message}`);
      throw error;
    }
  }

  /**
   * Nettoyer les réservations expirées
   */
  @Process('cleanup-expired-bookings')
  async handleCleanupExpiredBookings(job: Job) {
    this.logger.log('Nettoyage des réservations expirées...');

    try {
      const thirtyMinutesAgo = new Date(Date.now() - 30 * 60 * 1000);

      // Réservations en recherche depuis plus de 30 minutes
      const expiredSearching = await this.prisma.booking.updateMany({
        where: {
          booking_status: BOOKING_STATUS.SEARCHING,
          created_at: { lt: thirtyMinutesAgo },
        },
        data: {
          booking_status: BOOKING_STATUS.NO_DRIVER,
          cancelled_time: new Date(),
        },
      });

      // Réservations programmées dont l'heure est passée
      const expiredScheduled = await this.prisma.booking.updateMany({
        where: {
          booking_status: BOOKING_STATUS.SCHEDULED,
          schedule_time: { lt: new Date() },
        },
        data: {
          booking_status: BOOKING_STATUS.CANCELLED,
          cancelled_time: new Date(),
          cancel_reason: 'Réservation programmée expirée',
        },
      });

      this.logger.log(
        `Nettoyage: ${expiredSearching.count} recherches expirées, ${expiredScheduled.count} programmées expirées`,
      );

      return {
        expiredSearching: expiredSearching.count,
        expiredScheduled: expiredScheduled.count,
      };
    } catch (error) {
      this.logger.error(`Erreur nettoyage: ${error.message}`);
      throw error;
    }
  }

  /**
   * Mettre à jour les statistiques quotidiennes des chauffeurs
   */
  @Process('update-driver-daily-stats')
  async handleUpdateDriverStats(job: Job) {
    this.logger.log('Mise à jour des statistiques chauffeurs...');

    try {
      const today = new Date();
      today.setHours(0, 0, 0, 0);
      const tomorrow = new Date(today);
      tomorrow.setDate(tomorrow.getDate() + 1);

      // Obtenir les statistiques de chaque chauffeur actif aujourd'hui
      const driverStats = await this.prisma.booking.groupBy({
        by: ['driver_id'],
        where: {
          driver_id: { not: null },
          booking_status: BOOKING_STATUS.COMPLETED,
          completed_time: {
            gte: today,
            lt: tomorrow,
          },
        },
        _count: { id: true },
        _sum: {
          final_amount: true,
          driver_commission: true,
          travel_distance: true,
        },
      });

      for (const stat of driverStats) {
        if (!stat.driver_id) continue;

        await this.prisma.driverDailyStat.upsert({
          where: {
            driver_id_date: {
              driver_id: stat.driver_id,
              date: today,
            },
          },
          create: {
            driver_id: stat.driver_id,
            date: today,
            trips_count: stat._count.id,
            total_earnings: stat._sum.driver_commission || 0,
            total_revenue: stat._sum.final_amount || 0,
            total_distance: stat._sum.travel_distance || 0,
          },
          update: {
            trips_count: stat._count.id,
            total_earnings: stat._sum.driver_commission || 0,
            total_revenue: stat._sum.final_amount || 0,
            total_distance: stat._sum.travel_distance || 0,
          },
        });
      }

      return { driversUpdated: driverStats.length };
    } catch (error) {
      this.logger.error(`Erreur mise à jour stats: ${error.message}`);
      throw error;
    }
  }

  /**
   * Vérifier les documents chauffeurs qui expirent
   */
  @Process('check-expiring-documents')
  async handleExpiringDocuments(job: Job) {
    this.logger.log('Vérification des documents expirés...');

    try {
      const thirtyDaysFromNow = new Date(Date.now() + 30 * 24 * 60 * 60 * 1000);

      // Documents qui expirent dans 30 jours
      const expiringDocuments = await this.prisma.driverDocument.findMany({
        where: {
          expiry_date: {
            gte: new Date(),
            lte: thirtyDaysFromNow,
          },
          status: 1, // Approuvé
          expiry_notified: { not: 1 },
        },
        include: { driver: true },
      });

      for (const doc of expiringDocuments) {
        const daysUntilExpiry = Math.ceil(
          (doc.expiry_date.getTime() - Date.now()) / (1000 * 60 * 60 * 24),
        );

        await this.notificationService.sendToDriver(
          doc.driver_id,
          {
            title: 'Document bientôt expiré',
            body: `Votre ${doc.document_type} expire dans ${daysUntilExpiry} jours. Veuillez le renouveler.`,
            data: {
              type: 'document_expiring',
              document_id: String(doc.id),
              days_remaining: String(daysUntilExpiry),
            },
          },
          { push: true, email: true },
        );

        await this.prisma.driverDocument.update({
          where: { id: doc.id },
          data: { expiry_notified: 1 },
        });
      }

      // Documents déjà expirés
      const expiredDocuments = await this.prisma.driverDocument.findMany({
        where: {
          expiry_date: { lt: new Date() },
          status: 1,
        },
        include: { driver: true },
      });

      for (const doc of expiredDocuments) {
        // Marquer le document comme expiré
        await this.prisma.driverDocument.update({
          where: { id: doc.id },
          data: { status: 3 }, // Expiré
        });

        // Vérifier si tous les documents requis sont valides
        // Sinon, suspendre le chauffeur
        const validDocs = await this.prisma.driverDocument.count({
          where: {
            driver_id: doc.driver_id,
            status: 1,
            is_required: 1,
          },
        });

        const requiredDocs = await this.prisma.documentType.count({
          where: { is_required: 1 },
        });

        if (validDocs < requiredDocs) {
          await this.prisma.driver.update({
            where: { id: doc.driver_id },
            data: { driver_status: 2 }, // Suspendu
          });

          await this.notificationService.sendToDriver(
            doc.driver_id,
            {
              title: 'Compte suspendu',
              body: 'Votre compte est suspendu car certains documents sont expirés.',
              data: { type: 'account_suspended' },
            },
            { push: true, email: true },
          );
        }
      }

      return {
        expiringNotified: expiringDocuments.length,
        expired: expiredDocuments.length,
      };
    } catch (error) {
      this.logger.error(`Erreur vérification documents: ${error.message}`);
      throw error;
    }
  }

  /**
   * Envoyer le résumé quotidien aux chauffeurs
   */
  @Process('send-daily-summary')
  async handleDailySummary(job: Job) {
    this.logger.log('Envoi des résumés quotidiens...');

    try {
      const yesterday = new Date();
      yesterday.setDate(yesterday.getDate() - 1);
      yesterday.setHours(0, 0, 0, 0);

      const today = new Date();
      today.setHours(0, 0, 0, 0);

      // Chauffeurs qui ont travaillé hier
      const activeDrivers = await this.prisma.driverDailyStat.findMany({
        where: {
          date: yesterday,
          trips_count: { gt: 0 },
        },
        include: {
          driver: {
            select: {
              id: true,
              first_name: true,
              email: true,
            },
          },
        },
      });

      for (const stat of activeDrivers) {
        // Email de résumé
        await this.emailQueue.add('send-email', {
          to: stat.driver.email,
          subject: `Résumé du ${yesterday.toLocaleDateString('fr-FR')}`,
          template: 'daily-summary',
          context: {
            name: stat.driver.first_name,
            date: yesterday.toLocaleDateString('fr-FR'),
            trips: stat.trips_count,
            earnings: stat.total_earnings?.toFixed(0) || '0',
            distance: stat.total_distance?.toFixed(1) || '0',
          },
        });

        // Push notification
        await this.notificationService.sendToDriver(
          stat.driver.id,
          {
            title: 'Résumé de la journée',
            body: `${stat.trips_count} courses, ${stat.total_earnings?.toFixed(0) || 0} XOF gagnés`,
            data: { type: 'daily_summary' },
          },
          { push: true },
        );
      }

      return { summariesSent: activeDrivers.length };
    } catch (error) {
      this.logger.error(`Erreur envoi résumés: ${error.message}`);
      throw error;
    }
  }

  /**
   * Traiter les codes promo expirés
   */
  @Process('expire-promo-codes')
  async handleExpirePromoCodes(job: Job) {
    this.logger.log('Désactivation des codes promo expirés...');

    try {
      const result = await this.prisma.promoCode.updateMany({
        where: {
          end_date: { lt: new Date() },
          is_active: 1,
        },
        data: { is_active: 0 },
      });

      return { expiredCodes: result.count };
    } catch (error) {
      this.logger.error(`Erreur expiration promos: ${error.message}`);
      throw error;
    }
  }

  /**
   * Générer le rapport hebdomadaire
   */
  @Process('generate-weekly-report')
  async handleWeeklyReport(job: Job<{ merchantId: number }>) {
    const { merchantId } = job.data;

    this.logger.log(`Génération rapport hebdomadaire pour merchant #${merchantId}`);

    try {
      const oneWeekAgo = new Date();
      oneWeekAgo.setDate(oneWeekAgo.getDate() - 7);

      // Statistiques de la semaine
      const weeklyStats = await this.prisma.booking.aggregate({
        where: {
          merchant_id: merchantId,
          booking_status: BOOKING_STATUS.COMPLETED,
          completed_time: { gte: oneWeekAgo },
        },
        _count: { id: true },
        _sum: {
          final_amount: true,
          admin_commission: true,
        },
        _avg: {
          travel_distance: true,
          travel_time: true,
        },
      });

      // Nouveaux utilisateurs
      const newUsers = await this.prisma.user.count({
        where: {
          merchant_id: merchantId,
          created_at: { gte: oneWeekAgo },
        },
      });

      // Nouveaux chauffeurs
      const newDrivers = await this.prisma.driver.count({
        where: {
          merchant_id: merchantId,
          created_at: { gte: oneWeekAgo },
        },
      });

      // Envoyer par email aux admins du merchant
      const merchant = await this.prisma.merchant.findUnique({
        where: { id: merchantId },
        select: { admin_email: true, company_name: true },
      });

      if (merchant?.admin_email) {
        await this.emailQueue.add('send-email', {
          to: merchant.admin_email,
          subject: `Rapport hebdomadaire - ${merchant.company_name}`,
          template: 'weekly-report',
          context: {
            companyName: merchant.company_name,
            period: `${oneWeekAgo.toLocaleDateString('fr-FR')} - ${new Date().toLocaleDateString('fr-FR')}`,
            totalTrips: weeklyStats._count.id,
            totalRevenue: weeklyStats._sum.final_amount?.toFixed(0) || '0',
            commission: weeklyStats._sum.admin_commission?.toFixed(0) || '0',
            avgDistance: weeklyStats._avg.travel_distance?.toFixed(1) || '0',
            newUsers,
            newDrivers,
          },
        });
      }

      return {
        merchantId,
        stats: weeklyStats,
        newUsers,
        newDrivers,
      };
    } catch (error) {
      this.logger.error(`Erreur génération rapport: ${error.message}`);
      throw error;
    }
  }
}
