import { Process, Processor, OnQueueFailed } from '@nestjs/bull';
import { Logger } from '@nestjs/common';
import { Job } from 'bull';
import { PrismaService } from '../../../shared/database/prisma.service';
import { FirebaseService } from '../../notification/firebase.service';
import { OneSignalService } from '../../notification/onesignal.service';
import { SmsService } from '../../notification/sms.service';
import { QUEUE_NAMES } from '../queue.module';

@Processor(QUEUE_NAMES.NOTIFICATION)
export class NotificationQueueProcessor {
  private readonly logger = new Logger(NotificationQueueProcessor.name);

  constructor(
    private prisma: PrismaService,
    private firebaseService: FirebaseService,
    private oneSignalService: OneSignalService,
    private smsService: SmsService,
  ) {}

  @OnQueueFailed()
  onFailed(job: Job, error: Error) {
    this.logger.error(
      `Job ${job.name} (${job.id}) failed: ${error.message}`,
      error.stack,
    );
  }

  /**
   * Send push notification to a single user/driver
   */
  @Process('send-push')
  async handleSendPush(
    job: Job<{
      userId: number;
      userType: 'user' | 'driver';
      title: string;
      body: string;
      data?: Record<string, string>;
    }>,
  ) {
    const { userId, userType, title, body, data } = job.data;
    this.logger.log(`Sending push to ${userType} #${userId}: ${title}`);

    try {
      // Get device token
      let device: any;

      if (userType === 'user') {
        device = await this.prisma.userDevice.findFirst({
          where: { user_id: userId, is_active: 1 },
          orderBy: { created_at: 'desc' },
        });
      } else {
        device = await this.prisma.driverDevice.findFirst({
          where: { driver_id: userId, is_active: 1 },
          orderBy: { created_at: 'desc' },
        });
      }

      if (!device?.player_id) {
        this.logger.warn(`No device token for ${userType} #${userId}`);
        return { success: false, reason: 'No device token' };
      }

      // Send via Firebase
      await this.firebaseService.sendPushNotification(
        device.player_id,
        title,
        body,
        data,
      );

      this.logger.log(`Push sent successfully to ${userType} #${userId}`);
      return { success: true };
    } catch (error) {
      this.logger.error(`Push failed: ${error.message}`);
      throw error;
    }
  }

  /**
   * Send push notification to multiple users/drivers
   */
  @Process('send-bulk-push')
  async handleSendBulkPush(
    job: Job<{
      userIds: number[];
      userType: 'user' | 'driver';
      title: string;
      body: string;
      data?: Record<string, string>;
    }>,
  ) {
    const { userIds, userType, title, body, data } = job.data;
    this.logger.log(`Sending bulk push to ${userIds.length} ${userType}s`);

    try {
      // Get all device tokens
      let devices: any[];

      if (userType === 'user') {
        devices = await this.prisma.userDevice.findMany({
          where: {
            user_id: { in: userIds },
            is_active: 1,
          },
          select: { player_id: true },
        });
      } else {
        devices = await this.prisma.driverDevice.findMany({
          where: {
            driver_id: { in: userIds },
            is_active: 1,
          },
          select: { player_id: true },
        });
      }

      const tokens = devices
        .map((d) => d.player_id)
        .filter((t) => t);

      if (tokens.length === 0) {
        this.logger.warn('No device tokens found for bulk push');
        return { success: false, reason: 'No device tokens' };
      }

      // Send via Firebase multicast
      const result = await this.firebaseService.sendMulticast(
        tokens,
        title,
        body,
        data,
      );

      this.logger.log(
        `Bulk push completed: ${result?.successCount || 0} success, ${result?.failureCount || 0} failed`,
      );

      return {
        success: true,
        sent: result?.successCount || 0,
        failed: result?.failureCount || 0,
      };
    } catch (error) {
      this.logger.error(`Bulk push failed: ${error.message}`);
      throw error;
    }
  }

  /**
   * Envoyer SMS notification
   */
  @Process('send-sms')
  async handleSendSms(job: Job<{ phone: string; message: string }>) {
    const { phone, message } = job.data;
    this.logger.log(`Envoi SMS à ${phone}`);

    try {
      const result = await this.smsService.sendSms(phone, message);

      if (result.success) {
        this.logger.log(`SMS envoyé à ${phone}: ${result.messageId}`);
        return { success: true, messageId: result.messageId };
      } else {
        this.logger.error(`SMS échoué: ${result.error}`);
        throw new Error(result.error);
      }
    } catch (error) {
      this.logger.error(`SMS échoué: ${error.message}`);
      throw error;
    }
  }

  /**
   * Envoyer OTP via SMS
   */
  @Process('send-otp-sms')
  async handleSendOtpSms(
    job: Job<{ phone: string; otp: string; merchantName?: string }>,
  ) {
    const { phone, otp, merchantName } = job.data;
    this.logger.log(`Envoi OTP à ${phone}`);

    try {
      const result = await this.smsService.sendOtp(phone, otp, merchantName);

      if (result.success) {
        return { success: true, messageId: result.messageId };
      } else {
        throw new Error(result.error);
      }
    } catch (error) {
      this.logger.error(`OTP SMS échoué: ${error.message}`);
      throw error;
    }
  }

  /**
   * Envoyer push via OneSignal
   */
  @Process('send-onesignal')
  async handleSendOneSignal(
    job: Job<{
      playerId: string;
      title: string;
      message: string;
      data?: Record<string, any>;
      appType: 'user' | 'driver';
    }>,
  ) {
    const { playerId, title, message, data, appType } = job.data;
    this.logger.log(`Envoi OneSignal à ${playerId}`);

    try {
      const result = await this.oneSignalService.sendToPlayers(
        [playerId],
        { title, message, data },
        appType,
      );

      if (result.success) {
        return { success: true, notificationId: result.notificationId };
      } else {
        throw new Error(result.error);
      }
    } catch (error) {
      this.logger.error(`OneSignal échoué: ${error.message}`);
      throw error;
    }
  }

  /**
   * Envoyer push bulk via OneSignal
   */
  @Process('send-onesignal-bulk')
  async handleSendOneSignalBulk(
    job: Job<{
      externalUserIds: string[];
      title: string;
      message: string;
      data?: Record<string, any>;
      appType: 'user' | 'driver';
    }>,
  ) {
    const { externalUserIds, title, message, data, appType } = job.data;
    this.logger.log(`Envoi OneSignal bulk à ${externalUserIds.length} utilisateurs`);

    let sent = 0;
    let failed = 0;

    for (const externalUserId of externalUserIds) {
      try {
        const result = await this.oneSignalService.sendToExternalUserId(
          externalUserId,
          { title, message, data },
          appType,
        );

        if (result.success) {
          sent++;
        } else {
          failed++;
        }
      } catch {
        failed++;
      }
    }

    return { success: true, sent, failed };
  }
}
