import { Process, Processor, OnQueueFailed } from '@nestjs/bull';
import { Logger } from '@nestjs/common';
import { Job } from 'bull';
import { PrismaService } from '../../../shared/database/prisma.service';
import { QUEUE_NAMES } from '../queue.module';
import { BOOKING_STATUS } from '../../booking/booking.constants';

@Processor(QUEUE_NAMES.BOOKING)
export class BookingQueueProcessor {
  private readonly logger = new Logger(BookingQueueProcessor.name);

  constructor(private prisma: PrismaService) {}

  @OnQueueFailed()
  onFailed(job: Job, error: Error) {
    this.logger.error(
      `Job ${job.name} (${job.id}) failed: ${error.message}`,
      error.stack,
    );
  }

  /**
   * Find and assign a driver to a booking
   */
  @Process('assign-driver')
  async handleDriverAssignment(job: Job<{ bookingId: number; merchantId: number }>) {
    const { bookingId, merchantId } = job.data;
    this.logger.log(`Processing driver assignment for booking #${bookingId}`);

    try {
      // Get booking
      const booking = await this.prisma.booking.findUnique({
        where: { id: bookingId },
      });

      if (!booking || booking.booking_status !== BOOKING_STATUS.SEARCHING) {
        this.logger.warn(`Réservation #${bookingId} non disponible pour assignation`);
        return { success: false, reason: 'Réservation non disponible' };
      }

      // Find nearby available drivers
      // TODO: Implement proper geospatial query
      const availableDrivers = await this.prisma.driver.findMany({
        where: {
          merchant_id: merchantId,
          driver_status: 1, // Approved
          is_online: 1, // Online
          free_busy: 2, // Free
        },
        take: 10,
      });

      if (availableDrivers.length === 0) {
        this.logger.warn(`No available drivers for booking #${bookingId}`);
        // TODO: Notify user that no drivers are available
        return { success: false, reason: 'No drivers available' };
      }

      // For now, just pick the first available driver
      // TODO: Implement proper driver selection algorithm based on:
      // - Distance to pickup
      // - Driver rating
      // - Vehicle type match
      // - Driver preferences

      const selectedDriver = availableDrivers[0];

      // TODO: Send booking request to driver via WebSocket
      // TODO: Wait for driver response with timeout
      // For now, auto-assign

      this.logger.log(
        `Assigned driver #${selectedDriver.id} to booking #${bookingId}`,
      );

      return {
        success: true,
        driverId: selectedDriver.id,
        bookingId,
      };
    } catch (error) {
      this.logger.error(`Driver assignment failed: ${error.message}`);
      throw error;
    }
  }

  /**
   * Check if booking has timed out
   */
  @Process('check-timeout')
  async handleTimeoutCheck(job: Job<{ bookingId: number }>) {
    const { bookingId } = job.data;
    this.logger.log(`Checking timeout for booking #${bookingId}`);

    const booking = await this.prisma.booking.findUnique({
      where: { id: bookingId },
    });

    if (!booking) {
      return { success: false, reason: 'Booking not found' };
    }

    // Si toujours en recherche, annuler la réservation
    if (
      booking.booking_status === BOOKING_STATUS.SEARCHING ||
      booking.booking_status === BOOKING_STATUS.PENDING
    ) {
      await this.prisma.booking.update({
        where: { id: bookingId },
        data: {
          booking_status: BOOKING_STATUS.NO_DRIVER,
          cancelled_time: new Date(),
        },
      });

      // TODO: Notify user via WebSocket and push notification

      this.logger.log(`Booking #${bookingId} cancelled due to timeout`);
      return { success: true, action: 'cancelled' };
    }

    return { success: true, action: 'none', currentStatus: booking.booking_status };
  }

  /**
   * Process booking completion tasks
   */
  @Process('complete-booking')
  async handleBookingCompletion(job: Job<{ bookingId: number }>) {
    const { bookingId } = job.data;
    this.logger.log(`Processing completion for booking #${bookingId}`);

    const booking = await this.prisma.booking.findUnique({
      where: { id: bookingId },
      include: {
        user: true,
        driver: true,
      },
    });

    if (!booking) {
      return { success: false, reason: 'Booking not found' };
    }

    // Tasks to perform after ride completion:
    // 1. Calculate final fare (if not already)
    // 2. Process payment (if card/wallet)
    // 3. Update driver earnings
    // 4. Send invoice email
    // 5. Credit reward points
    // 6. Update driver/user statistics

    // TODO: Implement each task

    this.logger.log(`Completed processing for booking #${bookingId}`);
    return { success: true };
  }

  /**
   * Process driver location update
   */
  @Process('update-driver-location')
  async handleDriverLocationUpdate(
    job: Job<{
      bookingId: number;
      driverId: number;
      location: { latitude: number; longitude: number };
    }>,
  ) {
    const { bookingId, driverId, location } = job.data;

    // Update driver location in database
    await this.prisma.driver.update({
      where: { id: driverId },
      data: {
        latitude: location.latitude,
        longitude: location.longitude,
      },
    });

    // TODO: Calculate ETA and broadcast via WebSocket

    return { success: true };
  }
}
