import { Injectable, Logger, OnModuleInit } from '@nestjs/common';
import { Cron, CronExpression } from '@nestjs/schedule';
import { PrismaService } from '../../shared/database/prisma.service';

interface SurgeZone {
  latitude: number;
  longitude: number;
  radiusKm: number;
  multiplier: number;
  reason: string;
}

interface SurgeResult {
  multiplier: number;
  reason: string;
  isActive: boolean;
  expiresAt?: Date;
}

@Injectable()
export class SurgePricingService implements OnModuleInit {
  private readonly logger = new Logger(SurgePricingService.name);

  // Configuration surge
  private readonly MIN_MULTIPLIER = 1.0;
  private readonly MAX_MULTIPLIER = 3.0;
  private readonly SURGE_THRESHOLD_RATIO = 0.5; // Ratio demande/offre pour déclencher le surge
  private readonly ANALYSIS_RADIUS_KM = 3; // Rayon d'analyse
  private readonly CACHE_TTL_SECONDS = 60; // TTL du cache

  // Seuils de demande pour différents niveaux de surge
  private readonly DEMAND_THRESHOLDS = {
    low: 10,      // > 10 demandes = surge léger
    medium: 25,   // > 25 demandes = surge moyen
    high: 50,     // > 50 demandes = surge élevé
    critical: 100 // > 100 demandes = surge maximum
  };

  // Cache en mémoire pour les zones de surge
  private surgeCache: Map<string, { data: SurgeResult; expiresAt: number }> = new Map();

  // Historique des surges pour analyse
  private surgeHistory: Array<{
    zoneKey: string;
    multiplier: number;
    timestamp: Date;
    reason: string;
  }> = [];

  constructor(private prisma: PrismaService) {}

  async onModuleInit() {
    this.logger.log('Service de tarification dynamique initialisé');
    // Charger les zones de surge actives au démarrage
    await this.preloadActiveSurgeZones();
  }

  /**
   * Précharger les zones de surge actives
   */
  private async preloadActiveSurgeZones() {
    try {
      const activeZones = await this.prisma.surgeZone.findMany({
        where: {
          is_active: true,
          end_time: { gte: new Date() },
        },
      });
      this.logger.log(`${activeZones.length} zones de surge actives préchargées`);
    } catch (error) {
      // Table peut ne pas exister encore
      this.logger.warn(`Impossible de précharger les zones: ${error.message}`);
    }
  }

  /**
   * Calculate surge multiplier for a given location
   */
  async calculateSurgeMultiplier(
    merchantId: number,
    latitude: number,
    longitude: number,
    vehicleTypeId?: number,
  ): Promise<SurgeResult> {
    const cacheKey = this.getCacheKey(merchantId, latitude, longitude, vehicleTypeId);

    // Check cache first
    const cached = this.surgeCache.get(cacheKey);
    if (cached && cached.expiresAt > Date.now()) {
      return cached.data;
    }

    // Get merchant surge settings
    const merchant = await this.prisma.merchant.findUnique({
      where: { id: merchantId },
      select: {
        surge_pricing_enabled: true,
        surge_max_multiplier: true,
        surge_trigger_threshold: true,
      },
    });

    // If surge pricing is disabled for merchant
    if (!merchant?.surge_pricing_enabled) {
      return { multiplier: 1.0, reason: 'Surge disabled', isActive: false };
    }

    // Calculate demand/supply ratio
    const { demandCount, supplyCount } = await this.getDemandSupplyRatio(
      merchantId,
      latitude,
      longitude,
      vehicleTypeId,
    );

    // Check for time-based surge (peak hours, events)
    const timeSurge = this.getTimeBasedSurge();

    // Check for weather-based surge
    const weatherSurge = await this.getWeatherBasedSurge(latitude, longitude);

    // Check for event-based surge
    const eventSurge = await this.getEventBasedSurge(merchantId, latitude, longitude);

    // Calculate base surge from demand/supply
    let multiplier = 1.0;
    let reason = 'Normal pricing';

    const threshold = merchant.surge_trigger_threshold || this.SURGE_THRESHOLD_RATIO;
    const maxMultiplier = merchant.surge_max_multiplier || this.MAX_MULTIPLIER;

    if (supplyCount === 0) {
      // No drivers available - maximum surge
      multiplier = maxMultiplier;
      reason = 'Very high demand';
    } else {
      const ratio = demandCount / supplyCount;

      if (ratio > threshold) {
        // Linear scaling based on demand/supply ratio
        multiplier = Math.min(
          maxMultiplier,
          1 + (ratio - threshold) * 0.5,
        );
        reason = 'High demand in your area';
      }
    }

    // Apply additional surges (take maximum)
    if (timeSurge.multiplier > multiplier) {
      multiplier = timeSurge.multiplier;
      reason = timeSurge.reason;
    }

    if (weatherSurge.multiplier > multiplier) {
      multiplier = Math.min(maxMultiplier, weatherSurge.multiplier);
      reason = weatherSurge.reason;
    }

    if (eventSurge.multiplier > multiplier) {
      multiplier = Math.min(maxMultiplier, eventSurge.multiplier);
      reason = eventSurge.reason;
    }

    // Round to 1 decimal place
    multiplier = Math.round(multiplier * 10) / 10;

    const result: SurgeResult = {
      multiplier,
      reason,
      isActive: multiplier > 1.0,
      expiresAt: new Date(Date.now() + this.CACHE_TTL_SECONDS * 1000),
    };

    // Cache result
    this.surgeCache.set(cacheKey, {
      data: result,
      expiresAt: Date.now() + this.CACHE_TTL_SECONDS * 1000,
    });

    this.logger.log(
      `Surge calculation: ${latitude},${longitude} -> ${multiplier}x (${reason})`,
    );

    return result;
  }

  /**
   * Get demand/supply ratio for an area
   */
  private async getDemandSupplyRatio(
    merchantId: number,
    latitude: number,
    longitude: number,
    vehicleTypeId?: number,
  ): Promise<{ demandCount: number; supplyCount: number }> {
    const radiusDegrees = this.ANALYSIS_RADIUS_KM / 111; // Approximate conversion

    // Count pending bookings (demand) in the last 5 minutes
    const fiveMinutesAgo = new Date(Date.now() - 5 * 60 * 1000);

    const demandCount = await this.prisma.booking.count({
      where: {
        merchant_id: merchantId,
        booking_status: 'pending',
        created_at: { gte: fiveMinutesAgo },
        pickup_latitude: {
          gte: latitude - radiusDegrees,
          lte: latitude + radiusDegrees,
        },
        pickup_longitude: {
          gte: longitude - radiusDegrees,
          lte: longitude + radiusDegrees,
        },
        ...(vehicleTypeId && { vehicle_type_id: vehicleTypeId }),
      },
    });

    // Count available drivers (supply)
    const supplyCount = await this.prisma.driver.count({
      where: {
        merchant_id: merchantId,
        driver_status: 1,
        is_online: 1,
        free_busy: 2,
        latitude: {
          gte: latitude - radiusDegrees,
          lte: latitude + radiusDegrees,
        },
        longitude: {
          gte: longitude - radiusDegrees,
          lte: longitude + radiusDegrees,
        },
        ...(vehicleTypeId && { vehicle_type_id: vehicleTypeId }),
      },
    });

    return { demandCount, supplyCount };
  }

  /**
   * Get time-based surge (peak hours)
   */
  private getTimeBasedSurge(): { multiplier: number; reason: string } {
    const now = new Date();
    const hour = now.getHours();
    const dayOfWeek = now.getDay(); // 0 = Sunday

    // Morning rush (7-9 AM on weekdays)
    if (dayOfWeek >= 1 && dayOfWeek <= 5 && hour >= 7 && hour <= 9) {
      return { multiplier: 1.3, reason: 'Morning rush hour' };
    }

    // Evening rush (5-8 PM on weekdays)
    if (dayOfWeek >= 1 && dayOfWeek <= 5 && hour >= 17 && hour <= 20) {
      return { multiplier: 1.4, reason: 'Evening rush hour' };
    }

    // Weekend nights (10 PM - 2 AM)
    if ((dayOfWeek === 5 || dayOfWeek === 6) && (hour >= 22 || hour <= 2)) {
      return { multiplier: 1.5, reason: 'Weekend night' };
    }

    // New Year's Eve / Major holidays
    const month = now.getMonth();
    const date = now.getDate();
    if ((month === 11 && date === 31) || (month === 0 && date === 1)) {
      return { multiplier: 2.0, reason: 'Holiday surge' };
    }

    return { multiplier: 1.0, reason: '' };
  }

  /**
   * Get weather-based surge
   */
  private async getWeatherBasedSurge(
    latitude: number,
    longitude: number,
  ): Promise<{ multiplier: number; reason: string }> {
    // TODO: Integrate with weather API (OpenWeatherMap, etc.)
    // For now, return default

    // Example implementation:
    // const weather = await this.weatherService.getCurrentWeather(latitude, longitude);
    // if (weather.condition === 'rain') return { multiplier: 1.3, reason: 'Rainy weather' };
    // if (weather.condition === 'storm') return { multiplier: 1.5, reason: 'Storm conditions' };

    return { multiplier: 1.0, reason: '' };
  }

  /**
   * Get event-based surge (concerts, sports, etc.)
   */
  private async getEventBasedSurge(
    merchantId: number,
    latitude: number,
    longitude: number,
  ): Promise<{ multiplier: number; reason: string }> {
    // Check for active surge zones defined by admin
    const radiusDegrees = 5 / 111; // 5km radius

    const surgeZone = await this.prisma.surgeZone.findFirst({
      where: {
        merchant_id: merchantId,
        is_active: true,
        start_time: { lte: new Date() },
        end_time: { gte: new Date() },
        latitude: {
          gte: latitude - radiusDegrees,
          lte: latitude + radiusDegrees,
        },
        longitude: {
          gte: longitude - radiusDegrees,
          lte: longitude + radiusDegrees,
        },
      },
    });

    if (surgeZone) {
      return {
        multiplier: Number(surgeZone.multiplier) || 1.5,
        reason: surgeZone.reason || 'Event in area',
      };
    }

    return { multiplier: 1.0, reason: '' };
  }

  /**
   * Generate cache key for surge calculation
   */
  private getCacheKey(
    merchantId: number,
    latitude: number,
    longitude: number,
    vehicleTypeId?: number,
  ): string {
    // Round coordinates to reduce cache fragmentation
    const latRounded = Math.round(latitude * 100) / 100;
    const lngRounded = Math.round(longitude * 100) / 100;
    return `surge:${merchantId}:${latRounded}:${lngRounded}:${vehicleTypeId || 'all'}`;
  }

  /**
   * Admin: Create a manual surge zone
   */
  async createSurgeZone(
    merchantId: number,
    data: {
      latitude: number;
      longitude: number;
      radiusKm: number;
      multiplier: number;
      reason: string;
      startTime: Date;
      endTime: Date;
    },
  ): Promise<any> {
    return this.prisma.surgeZone.create({
      data: {
        merchant_id: merchantId,
        latitude: data.latitude,
        longitude: data.longitude,
        radius_km: data.radiusKm,
        multiplier: data.multiplier,
        reason: data.reason,
        start_time: data.startTime,
        end_time: data.endTime,
        is_active: true,
      },
    });
  }

  /**
   * Admin: Deactivate a surge zone
   */
  async deactivateSurgeZone(surgeZoneId: number): Promise<void> {
    await this.prisma.surgeZone.update({
      where: { id: surgeZoneId },
      data: { is_active: false },
    });
  }

  /**
   * Get all active surge zones for a merchant
   */
  async getActiveSurgeZones(merchantId: number): Promise<any[]> {
    return this.prisma.surgeZone.findMany({
      where: {
        merchant_id: merchantId,
        is_active: true,
        end_time: { gte: new Date() },
      },
    });
  }

  /**
   * Apply surge to a fare amount
   */
  applyMultiplier(baseFare: number, multiplier: number): number {
    return Math.round(baseFare * multiplier);
  }

  /**
   * Clear cache (for testing or admin use)
   */
  clearCache(): void {
    this.surgeCache.clear();
    this.logger.log('Surge cache cleared');
  }

  // =============================================================================
  // TÂCHES CRON
  // =============================================================================

  /**
   * Nettoyer le cache des surges expirés toutes les 5 minutes
   */
  @Cron(CronExpression.EVERY_5_MINUTES)
  async cleanupExpiredCache() {
    const now = Date.now();
    let cleaned = 0;

    for (const [key, value] of this.surgeCache.entries()) {
      if (value.expiresAt < now) {
        this.surgeCache.delete(key);
        cleaned++;
      }
    }

    // Nettoyer l'historique des surges > 24h
    const oneDayAgo = new Date(Date.now() - 24 * 60 * 60 * 1000);
    this.surgeHistory = this.surgeHistory.filter((h) => h.timestamp > oneDayAgo);

    if (cleaned > 0) {
      this.logger.debug(`${cleaned} entrées de cache surge nettoyées`);
    }
  }

  /**
   * Désactiver les zones de surge expirées toutes les heures
   */
  @Cron(CronExpression.EVERY_HOUR)
  async deactivateExpiredSurgeZones() {
    try {
      const result = await this.prisma.surgeZone.updateMany({
        where: {
          is_active: true,
          end_time: { lt: new Date() },
        },
        data: { is_active: false },
      });

      if (result.count > 0) {
        this.logger.log(`${result.count} zones de surge expirées désactivées`);
      }
    } catch (error) {
      this.logger.error(`Erreur désactivation zones expirées: ${error.message}`);
    }
  }

  // =============================================================================
  // MÉTHODES D'ANALYSE
  // =============================================================================

  /**
   * Obtenir les statistiques de surge
   */
  async getSurgeStats(merchantId: number): Promise<{
    activeZones: number;
    averageMultiplier: number;
    topSurgeAreas: Array<{ zone: string; multiplier: number; reason: string }>;
    history: Array<{ multiplier: number; timestamp: Date; reason: string }>;
  }> {
    // Zones actives
    let activeZones = 0;
    try {
      activeZones = await this.prisma.surgeZone.count({
        where: {
          merchant_id: merchantId,
          is_active: true,
          end_time: { gte: new Date() },
        },
      });
    } catch {
      // Table peut ne pas exister
    }

    // Calculer la moyenne des multiplicateurs actuels
    const surgeValues = Array.from(this.surgeCache.values())
      .filter((v) => v.expiresAt > Date.now())
      .map((v) => v.data.multiplier);

    const averageMultiplier =
      surgeValues.length > 0
        ? surgeValues.reduce((a, b) => a + b, 0) / surgeValues.length
        : 1.0;

    // Top zones de surge
    const topSurgeAreas = Array.from(this.surgeCache.entries())
      .filter(([, v]) => v.expiresAt > Date.now() && v.data.multiplier > 1.0)
      .map(([zone, v]) => ({
        zone,
        multiplier: v.data.multiplier,
        reason: v.data.reason,
      }))
      .sort((a, b) => b.multiplier - a.multiplier)
      .slice(0, 10);

    // Historique récent
    const history = this.surgeHistory
      .filter((h) => h.multiplier > 1.0)
      .slice(-50)
      .reverse();

    return {
      activeZones,
      averageMultiplier: Math.round(averageMultiplier * 100) / 100,
      topSurgeAreas,
      history,
    };
  }

  /**
   * Obtenir une carte de chaleur des surges
   */
  async getSurgeHeatmap(
    merchantId: number,
    bounds: {
      minLat: number;
      maxLat: number;
      minLng: number;
      maxLng: number;
    },
    gridSize = 0.01, // ~1km
  ): Promise<
    Array<{
      latitude: number;
      longitude: number;
      multiplier: number;
      intensity: 'low' | 'medium' | 'high' | 'critical';
    }>
  > {
    const heatmapPoints: Array<{
      latitude: number;
      longitude: number;
      multiplier: number;
      intensity: 'low' | 'medium' | 'high' | 'critical';
    }> = [];

    // Parcourir la grille
    for (let lat = bounds.minLat; lat <= bounds.maxLat; lat += gridSize) {
      for (let lng = bounds.minLng; lng <= bounds.maxLng; lng += gridSize) {
        const cacheKey = this.getCacheKey(merchantId, lat, lng);
        const cached = this.surgeCache.get(cacheKey);

        if (cached && cached.expiresAt > Date.now() && cached.data.multiplier > 1.0) {
          heatmapPoints.push({
            latitude: lat,
            longitude: lng,
            multiplier: cached.data.multiplier,
            intensity: this.getIntensity(cached.data.multiplier),
          });
        }
      }
    }

    return heatmapPoints;
  }

  private getIntensity(multiplier: number): 'low' | 'medium' | 'high' | 'critical' {
    if (multiplier >= 2.5) return 'critical';
    if (multiplier >= 2.0) return 'high';
    if (multiplier >= 1.5) return 'medium';
    return 'low';
  }

  /**
   * Prédire le surge pour une heure future
   */
  async predictSurge(
    merchantId: number,
    latitude: number,
    longitude: number,
    hoursAhead: number,
  ): Promise<{
    predictedMultiplier: number;
    confidence: number;
    basedOn: string;
  }> {
    const targetTime = new Date(Date.now() + hoursAhead * 60 * 60 * 1000);
    const hour = targetTime.getHours();
    const dayOfWeek = targetTime.getDay();

    // Prédiction basique basée sur les patterns historiques
    let predictedMultiplier = 1.0;
    let basedOn = 'Tarif normal prévu';

    // Heures de pointe
    if ((hour >= 7 && hour <= 9) || (hour >= 17 && hour <= 20)) {
      if (dayOfWeek >= 1 && dayOfWeek <= 5) {
        predictedMultiplier = 1.3;
        basedOn = 'Heure de pointe prévue';
      }
    }

    // Weekend soir
    if ((dayOfWeek === 5 || dayOfWeek === 6) && (hour >= 21 || hour <= 2)) {
      predictedMultiplier = 1.5;
      basedOn = 'Sortie weekend prévue';
    }

    // Vérifier les événements planifiés
    try {
      const plannedEvent = await this.prisma.surgeZone.findFirst({
        where: {
          merchant_id: merchantId,
          is_active: true,
          start_time: { lte: targetTime },
          end_time: { gte: targetTime },
          latitude: {
            gte: latitude - 0.05,
            lte: latitude + 0.05,
          },
          longitude: {
            gte: longitude - 0.05,
            lte: longitude + 0.05,
          },
        },
      });

      if (plannedEvent) {
        predictedMultiplier = Math.max(predictedMultiplier, Number(plannedEvent.multiplier) || 1.5);
        basedOn = plannedEvent.reason || 'Événement planifié';
      }
    } catch {
      // Table peut ne pas exister
    }

    return {
      predictedMultiplier: Math.round(predictedMultiplier * 10) / 10,
      confidence: predictedMultiplier > 1.0 ? 0.7 : 0.9,
      basedOn,
    };
  }

  /**
   * Enregistrer un surge dans l'historique
   */
  private recordSurgeHistory(zoneKey: string, multiplier: number, reason: string) {
    if (multiplier > 1.0) {
      this.surgeHistory.push({
        zoneKey,
        multiplier,
        timestamp: new Date(),
        reason,
      });

      // Garder seulement les 1000 dernières entrées
      if (this.surgeHistory.length > 1000) {
        this.surgeHistory = this.surgeHistory.slice(-1000);
      }
    }
  }
}
