import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../shared/database/prisma.service';
import { SurgePricingService } from './surge-pricing.service';

interface FareEstimate {
  baseFare: number;
  distanceFare: number;
  timeFare: number;
  waitingFare: number;
  surgeMultiplier: number;
  surgeReason: string;
  subtotal: number;
  discount: number;
  promoCode?: string;
  tax: number;
  serviceFee: number;
  totalFare: number;
  currency: string;
  distanceKm: number;
  durationMinutes: number;
  breakdown: FareBreakdownItem[];
}

interface FareBreakdownItem {
  label: string;
  amount: number;
  type: 'add' | 'subtract' | 'multiply';
}

interface PricingConfig {
  baseFare: number;
  perKmRate: number;
  perMinuteRate: number;
  minimumFare: number;
  waitingChargePerMinute: number;
  freeWaitingMinutes: number;
  bookingFee: number;
  taxPercentage: number;
  serviceFeePercentage: number;
  currency: string;
}

@Injectable()
export class PricingService {
  private readonly logger = new Logger(PricingService.name);

  // Earth radius in km
  private readonly EARTH_RADIUS = 6371;

  constructor(
    private prisma: PrismaService,
    private surgePricingService: SurgePricingService,
  ) {}

  /**
   * Calculate estimated fare for a trip
   */
  async calculateEstimate(
    merchantId: number,
    vehicleTypeId: number,
    pickupLat: number,
    pickupLng: number,
    dropLat: number,
    dropLng: number,
    promoCode?: string,
  ): Promise<FareEstimate> {
    // Get pricing configuration for vehicle type
    const pricing = await this.getPricingConfig(merchantId, vehicleTypeId);

    // Calculate distance using Haversine formula
    const distanceKm = this.calculateDistance(
      pickupLat,
      pickupLng,
      dropLat,
      dropLng,
    );

    // Estimate duration (average speed assumption)
    const averageSpeedKmh = 30; // Adjust based on city/traffic
    const durationMinutes = Math.ceil((distanceKm / averageSpeedKmh) * 60);

    // Calculate base fare components
    const baseFare = pricing.baseFare;
    const distanceFare = distanceKm * pricing.perKmRate;
    const timeFare = durationMinutes * pricing.perMinuteRate;
    const waitingFare = 0; // Calculated at end of trip

    // Get surge multiplier
    const surge = await this.surgePricingService.calculateSurgeMultiplier(
      merchantId,
      pickupLat,
      pickupLng,
      vehicleTypeId,
    );

    // Calculate subtotal before surge
    let subtotal = baseFare + distanceFare + timeFare + pricing.bookingFee;

    // Apply surge
    if (surge.isActive) {
      subtotal = this.surgePricingService.applyMultiplier(subtotal, surge.multiplier);
    }

    // Apply minimum fare if needed
    subtotal = Math.max(subtotal, pricing.minimumFare);

    // Calculate promo discount
    let discount = 0;
    let appliedPromoCode: string | undefined;

    if (promoCode) {
      const promoResult = await this.applyPromoCode(
        promoCode,
        merchantId,
        subtotal,
      );
      if (promoResult.valid) {
        discount = promoResult.discount;
        appliedPromoCode = promoCode;
      }
    }

    // Calculate tax
    const taxableAmount = subtotal - discount;
    const tax = Math.round(taxableAmount * (pricing.taxPercentage / 100));

    // Calculate service fee
    const serviceFee = Math.round(subtotal * (pricing.serviceFeePercentage / 100));

    // Calculate total
    const totalFare = subtotal - discount + tax + serviceFee;

    // Build breakdown
    const breakdown: FareBreakdownItem[] = [
      { label: 'Base fare', amount: baseFare, type: 'add' },
      { label: `Distance (${distanceKm.toFixed(1)} km)`, amount: Math.round(distanceFare), type: 'add' },
      { label: `Time (${durationMinutes} min)`, amount: Math.round(timeFare), type: 'add' },
    ];

    if (pricing.bookingFee > 0) {
      breakdown.push({ label: 'Booking fee', amount: pricing.bookingFee, type: 'add' });
    }

    if (surge.isActive) {
      breakdown.push({
        label: `Surge (${surge.multiplier}x - ${surge.reason})`,
        amount: surge.multiplier,
        type: 'multiply',
      });
    }

    if (discount > 0) {
      breakdown.push({ label: `Promo (${appliedPromoCode})`, amount: discount, type: 'subtract' });
    }

    if (tax > 0) {
      breakdown.push({ label: `Tax (${pricing.taxPercentage}%)`, amount: tax, type: 'add' });
    }

    if (serviceFee > 0) {
      breakdown.push({ label: 'Service fee', amount: serviceFee, type: 'add' });
    }

    return {
      baseFare,
      distanceFare: Math.round(distanceFare),
      timeFare: Math.round(timeFare),
      waitingFare,
      surgeMultiplier: surge.multiplier,
      surgeReason: surge.reason,
      subtotal,
      discount,
      promoCode: appliedPromoCode,
      tax,
      serviceFee,
      totalFare: Math.round(totalFare),
      currency: pricing.currency,
      distanceKm: Math.round(distanceKm * 10) / 10,
      durationMinutes,
      breakdown,
    };
  }

  /**
   * Calculate final fare at end of trip
   */
  async calculateFinalFare(
    bookingId: number,
    actualDistanceKm: number,
    actualDurationMinutes: number,
    waitingMinutes: number = 0,
  ): Promise<FareEstimate> {
    const booking = await this.prisma.booking.findUnique({
      where: { id: bookingId },
    });

    if (!booking) {
      throw new Error('Booking not found');
    }

    // Get pricing configuration
    const pricing = await this.getPricingConfig(
      booking.merchant_id,
      booking.vehicle_type_id,
    );

    // Calculate fare components
    const baseFare = pricing.baseFare;
    const distanceFare = actualDistanceKm * pricing.perKmRate;
    const timeFare = actualDurationMinutes * pricing.perMinuteRate;

    // Calculate waiting charges (after free waiting period)
    const chargeableWaitingMinutes = Math.max(
      0,
      waitingMinutes - pricing.freeWaitingMinutes,
    );
    const waitingFare = chargeableWaitingMinutes * pricing.waitingChargePerMinute;

    // Use stored surge multiplier from booking creation
    const surgeMultiplier = Number(booking.surge_multiplier) || 1.0;

    // Calculate subtotal
    let subtotal = baseFare + distanceFare + timeFare + waitingFare + pricing.bookingFee;

    // Apply surge
    if (surgeMultiplier > 1.0) {
      subtotal = this.surgePricingService.applyMultiplier(subtotal, surgeMultiplier);
    }

    // Apply minimum fare
    subtotal = Math.max(subtotal, pricing.minimumFare);

    // Apply stored discount
    const discount = Number(booking.discount_amount) || 0;

    // Calculate tax
    const taxableAmount = subtotal - discount;
    const tax = Math.round(taxableAmount * (pricing.taxPercentage / 100));

    // Calculate service fee
    const serviceFee = Math.round(subtotal * (pricing.serviceFeePercentage / 100));

    // Calculate total
    const totalFare = subtotal - discount + tax + serviceFee;

    const breakdown: FareBreakdownItem[] = [
      { label: 'Base fare', amount: baseFare, type: 'add' },
      { label: `Distance (${actualDistanceKm.toFixed(1)} km)`, amount: Math.round(distanceFare), type: 'add' },
      { label: `Time (${actualDurationMinutes} min)`, amount: Math.round(timeFare), type: 'add' },
    ];

    if (waitingFare > 0) {
      breakdown.push({
        label: `Waiting (${chargeableWaitingMinutes} min)`,
        amount: Math.round(waitingFare),
        type: 'add',
      });
    }

    if (surgeMultiplier > 1.0) {
      breakdown.push({
        label: `Surge (${surgeMultiplier}x)`,
        amount: surgeMultiplier,
        type: 'multiply',
      });
    }

    if (discount > 0) {
      breakdown.push({ label: 'Discount', amount: discount, type: 'subtract' });
    }

    if (tax > 0) {
      breakdown.push({ label: `Tax (${pricing.taxPercentage}%)`, amount: tax, type: 'add' });
    }

    if (serviceFee > 0) {
      breakdown.push({ label: 'Service fee', amount: serviceFee, type: 'add' });
    }

    return {
      baseFare,
      distanceFare: Math.round(distanceFare),
      timeFare: Math.round(timeFare),
      waitingFare: Math.round(waitingFare),
      surgeMultiplier,
      surgeReason: '',
      subtotal,
      discount,
      tax,
      serviceFee,
      totalFare: Math.round(totalFare),
      currency: pricing.currency,
      distanceKm: actualDistanceKm,
      durationMinutes: actualDurationMinutes,
      breakdown,
    };
  }

  /**
   * Get pricing configuration for a vehicle type
   */
  async getPricingConfig(
    merchantId: number,
    vehicleTypeId: number,
  ): Promise<PricingConfig> {
    // Try to get vehicle-specific pricing
    const vehiclePricing = await this.prisma.vehicleTypePricing.findFirst({
      where: {
        merchant_id: merchantId,
        vehicle_type_id: vehicleTypeId,
        is_active: 1,
      },
    });

    if (vehiclePricing) {
      return {
        baseFare: Number(vehiclePricing.base_fare) || 500,
        perKmRate: Number(vehiclePricing.per_km_rate) || 100,
        perMinuteRate: Number(vehiclePricing.per_minute_rate) || 20,
        minimumFare: Number(vehiclePricing.minimum_fare) || 1000,
        waitingChargePerMinute: Number(vehiclePricing.waiting_charge_per_minute) || 50,
        freeWaitingMinutes: Number(vehiclePricing.free_waiting_minutes) || 3,
        bookingFee: Number(vehiclePricing.booking_fee) || 0,
        taxPercentage: Number(vehiclePricing.tax_percentage) || 0,
        serviceFeePercentage: Number(vehiclePricing.service_fee_percentage) || 0,
        currency: vehiclePricing.currency || 'XOF',
      };
    }

    // Fallback to merchant default pricing
    const merchant = await this.prisma.merchant.findUnique({
      where: { id: merchantId },
    });

    return {
      baseFare: Number(merchant?.base_fare) || 500,
      perKmRate: Number(merchant?.per_km_rate) || 100,
      perMinuteRate: Number(merchant?.per_minute_rate) || 20,
      minimumFare: Number(merchant?.minimum_fare) || 1000,
      waitingChargePerMinute: Number(merchant?.waiting_charge_per_minute) || 50,
      freeWaitingMinutes: Number(merchant?.free_waiting_minutes) || 3,
      bookingFee: Number(merchant?.booking_fee) || 0,
      taxPercentage: Number(merchant?.tax_percentage) || 0,
      serviceFeePercentage: Number(merchant?.service_fee_percentage) || 0,
      currency: merchant?.currency || 'XOF',
    };
  }

  /**
   * Apply promo code and calculate discount
   */
  private async applyPromoCode(
    code: string,
    merchantId: number,
    subtotal: number,
  ): Promise<{ valid: boolean; discount: number; message?: string }> {
    // This will be replaced by PromoCodeService
    // Placeholder implementation
    const promo = await this.prisma.promoCode.findFirst({
      where: {
        code: code.toUpperCase(),
        merchant_id: merchantId,
        is_active: 1,
        start_date: { lte: new Date() },
        end_date: { gte: new Date() },
      },
    });

    if (!promo) {
      return { valid: false, discount: 0, message: 'Invalid promo code' };
    }

    // Check usage limit
    if (promo.max_uses && promo.used_count >= promo.max_uses) {
      return { valid: false, discount: 0, message: 'Promo code expired' };
    }

    // Check minimum order value
    if (promo.min_order_value && subtotal < Number(promo.min_order_value)) {
      return {
        valid: false,
        discount: 0,
        message: `Minimum order value: ${promo.min_order_value}`,
      };
    }

    // Calculate discount
    let discount = 0;

    if (promo.discount_type === 'percentage') {
      discount = Math.round(subtotal * (Number(promo.discount_value) / 100));
      // Apply max discount cap
      if (promo.max_discount && discount > Number(promo.max_discount)) {
        discount = Number(promo.max_discount);
      }
    } else {
      // Fixed amount
      discount = Number(promo.discount_value);
    }

    // Ensure discount doesn't exceed subtotal
    discount = Math.min(discount, subtotal);

    return { valid: true, discount };
  }

  /**
   * Calculate distance between two points using Haversine formula
   */
  calculateDistance(
    lat1: number,
    lng1: number,
    lat2: number,
    lng2: number,
  ): number {
    const dLat = this.toRad(lat2 - lat1);
    const dLng = this.toRad(lng2 - lng1);

    const a =
      Math.sin(dLat / 2) * Math.sin(dLat / 2) +
      Math.cos(this.toRad(lat1)) *
        Math.cos(this.toRad(lat2)) *
        Math.sin(dLng / 2) *
        Math.sin(dLng / 2);

    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
    return this.EARTH_RADIUS * c;
  }

  private toRad(deg: number): number {
    return deg * (Math.PI / 180);
  }

  /**
   * Get estimated arrival time based on distance and traffic
   */
  async getEstimatedArrivalTime(
    driverLat: number,
    driverLng: number,
    destinationLat: number,
    destinationLng: number,
  ): Promise<{ distanceKm: number; durationMinutes: number; arrivalTime: Date }> {
    const distanceKm = this.calculateDistance(
      driverLat,
      driverLng,
      destinationLat,
      destinationLng,
    );

    // Estimate based on average speed (can be enhanced with traffic API)
    const averageSpeedKmh = 25; // Conservative estimate for city traffic
    const durationMinutes = Math.ceil((distanceKm / averageSpeedKmh) * 60);

    const arrivalTime = new Date(Date.now() + durationMinutes * 60 * 1000);

    return {
      distanceKm: Math.round(distanceKm * 10) / 10,
      durationMinutes,
      arrivalTime,
    };
  }
}
