// =============================================================================
// PricingService Unit Tests
// =============================================================================

import { Test, TestingModule } from '@nestjs/testing';
import { PricingService } from './pricing.service';
import { SurgePricingService } from './surge-pricing.service';
import { PrismaService } from '../../shared/database/prisma.service';
import {
  mockPrismaService,
  mockSurgePricingService,
  createMockBooking,
  createMockVehicleType,
  createMockPromoCode,
  createMockMerchant,
  resetAllMocks,
} from '../../../test/setup';

describe('PricingService', () => {
  let service: PricingService;
  let prisma: typeof mockPrismaService;
  let surgePricing: typeof mockSurgePricingService;

  beforeEach(async () => {
    resetAllMocks();

    const module: TestingModule = await Test.createTestingModule({
      providers: [
        PricingService,
        { provide: PrismaService, useValue: mockPrismaService },
        { provide: SurgePricingService, useValue: mockSurgePricingService },
      ],
    }).compile();

    service = module.get<PricingService>(PricingService);
    prisma = mockPrismaService;
    surgePricing = mockSurgePricingService;
  });

  afterEach(() => {
    jest.clearAllMocks();
  });

  // ===========================================================================
  // CALCULATE ESTIMATE
  // ===========================================================================

  describe('calculateEstimate', () => {
    it('devrait calculer une estimation de tarif correcte', async () => {
      const mockVehiclePricing = {
        base_fare: 500,
        per_km_rate: 200,
        per_minute_rate: 25,
        minimum_fare: 1000,
        waiting_charge_per_minute: 20,
        free_waiting_minutes: 3,
        booking_fee: 100,
        tax_percentage: 0,
        service_fee_percentage: 5,
        currency: 'XOF',
      };

      prisma.vehicleTypePricing = {
        findFirst: jest.fn().mockResolvedValue(mockVehiclePricing),
      };
      surgePricing.calculateSurgeMultiplier.mockResolvedValue({
        multiplier: 1.0,
        reason: 'Normal pricing',
        isActive: false,
      });

      const result = await service.calculateEstimate(
        1, // merchantId
        1, // vehicleTypeId
        6.1319, // pickupLat (Lomé Centre)
        1.2228, // pickupLng
        6.1654, // dropLat (Aéroport)
        1.2544, // dropLng
      );

      expect(result).toHaveProperty('baseFare');
      expect(result).toHaveProperty('distanceFare');
      expect(result).toHaveProperty('timeFare');
      expect(result).toHaveProperty('totalFare');
      expect(result).toHaveProperty('currency', 'XOF');
      expect(result.totalFare).toBeGreaterThan(0);
    });

    it('devrait appliquer le surge pricing quand actif', async () => {
      const mockVehiclePricing = {
        base_fare: 500,
        per_km_rate: 200,
        per_minute_rate: 25,
        minimum_fare: 1000,
        currency: 'XOF',
      };

      prisma.vehicleTypePricing = {
        findFirst: jest.fn().mockResolvedValue(mockVehiclePricing),
      };
      surgePricing.calculateSurgeMultiplier.mockResolvedValue({
        multiplier: 1.5,
        reason: 'High demand',
        isActive: true,
      });
      surgePricing.applyMultiplier.mockImplementation((fare, mult) => Math.round(fare * mult));

      const result = await service.calculateEstimate(
        1, 1, 6.1319, 1.2228, 6.1654, 1.2544,
      );

      expect(result.surgeMultiplier).toBe(1.5);
      expect(result.surgeReason).toBe('High demand');
    });

    it('devrait appliquer un code promo pourcentage', async () => {
      const mockVehiclePricing = {
        base_fare: 500,
        per_km_rate: 200,
        per_minute_rate: 25,
        minimum_fare: 1000,
        currency: 'XOF',
      };
      const mockPromo = createMockPromoCode({
        discount_type: 'percentage',
        discount_value: 20,
        max_discount: 2000,
      });

      prisma.vehicleTypePricing = {
        findFirst: jest.fn().mockResolvedValue(mockVehiclePricing),
      };
      prisma.promoCode.findFirst.mockResolvedValue(mockPromo);
      surgePricing.calculateSurgeMultiplier.mockResolvedValue({
        multiplier: 1.0,
        reason: '',
        isActive: false,
      });

      const result = await service.calculateEstimate(
        1, 1, 6.1319, 1.2228, 6.1654, 1.2544, 'TEST20',
      );

      expect(result.discount).toBeGreaterThan(0);
      expect(result.promoCode).toBe('TEST20');
    });

    it('devrait appliquer le tarif minimum', async () => {
      const mockVehiclePricing = {
        base_fare: 100,
        per_km_rate: 50,
        per_minute_rate: 5,
        minimum_fare: 1000,
        currency: 'XOF',
      };

      prisma.vehicleTypePricing = {
        findFirst: jest.fn().mockResolvedValue(mockVehiclePricing),
      };
      surgePricing.calculateSurgeMultiplier.mockResolvedValue({
        multiplier: 1.0,
        reason: '',
        isActive: false,
      });

      // Distance très courte = tarif minimum
      const result = await service.calculateEstimate(
        1, 1, 6.1319, 1.2228, 6.1320, 1.2229,
      );

      expect(result.subtotal).toBeGreaterThanOrEqual(1000);
    });
  });

  // ===========================================================================
  // CALCULATE FINAL FARE
  // ===========================================================================

  describe('calculateFinalFare', () => {
    it('devrait calculer le tarif final avec frais d\'attente', async () => {
      const mockBooking = createMockBooking({
        surge_multiplier: 1.0,
        discount_amount: 0,
      });
      const mockVehiclePricing = {
        base_fare: 500,
        per_km_rate: 200,
        per_minute_rate: 25,
        minimum_fare: 1000,
        waiting_charge_per_minute: 50,
        free_waiting_minutes: 3,
        currency: 'XOF',
      };

      prisma.booking.findUnique.mockResolvedValue(mockBooking);
      prisma.vehicleTypePricing = {
        findFirst: jest.fn().mockResolvedValue(mockVehiclePricing),
      };

      const result = await service.calculateFinalFare(
        1,    // bookingId
        10,   // actualDistanceKm
        25,   // actualDurationMinutes
        8,    // waitingMinutes (5 chargeable after 3 free)
      );

      expect(result.waitingFare).toBeGreaterThan(0);
      expect(result.totalFare).toBeGreaterThan(0);
    });

    it('devrait utiliser le surge stocké de la réservation', async () => {
      const mockBooking = createMockBooking({
        surge_multiplier: 1.8,
        discount_amount: 500,
      });

      prisma.booking.findUnique.mockResolvedValue(mockBooking);
      prisma.vehicleTypePricing = {
        findFirst: jest.fn().mockResolvedValue({
          base_fare: 500,
          per_km_rate: 200,
          per_minute_rate: 25,
          minimum_fare: 1000,
          currency: 'XOF',
        }),
      };

      const result = await service.calculateFinalFare(1, 10, 25, 0);

      expect(result.surgeMultiplier).toBe(1.8);
      expect(result.discount).toBe(500);
    });

    it('devrait lever une erreur si réservation non trouvée', async () => {
      prisma.booking.findUnique.mockResolvedValue(null);

      await expect(
        service.calculateFinalFare(999, 10, 25, 0),
      ).rejects.toThrow('Booking not found');
    });
  });

  // ===========================================================================
  // GET PRICING CONFIG
  // ===========================================================================

  describe('getPricingConfig', () => {
    it('devrait retourner la config du type de véhicule', async () => {
      const mockVehiclePricing = {
        base_fare: 600,
        per_km_rate: 250,
        per_minute_rate: 30,
        minimum_fare: 1200,
        currency: 'XOF',
      };

      prisma.vehicleTypePricing = {
        findFirst: jest.fn().mockResolvedValue(mockVehiclePricing),
      };

      const result = await service.getPricingConfig(1, 1);

      expect(result.baseFare).toBe(600);
      expect(result.perKmRate).toBe(250);
    });

    it('devrait utiliser la config merchant par défaut si pas de config véhicule', async () => {
      const mockMerchant = createMockMerchant({
        base_fare: 400,
        per_km_rate: 150,
        per_minute_rate: 20,
        minimum_fare: 800,
        currency: 'XOF',
      });

      prisma.vehicleTypePricing = {
        findFirst: jest.fn().mockResolvedValue(null),
      };
      prisma.merchant.findUnique.mockResolvedValue(mockMerchant);

      const result = await service.getPricingConfig(1, 1);

      expect(result.baseFare).toBe(400);
    });
  });

  // ===========================================================================
  // CALCULATE DISTANCE
  // ===========================================================================

  describe('calculateDistance', () => {
    it('devrait calculer la distance entre Lomé Centre et Aéroport (~5km)', () => {
      const distance = service.calculateDistance(
        6.1319, 1.2228, // Lomé Centre
        6.1654, 1.2544, // Aéroport
      );

      expect(distance).toBeGreaterThan(4);
      expect(distance).toBeLessThan(7);
    });

    it('devrait retourner 0 pour le même point', () => {
      const distance = service.calculateDistance(
        6.1319, 1.2228,
        6.1319, 1.2228,
      );

      expect(distance).toBe(0);
    });

    it('devrait calculer des longues distances correctement', () => {
      // Lomé à Kara (~400km)
      const distance = service.calculateDistance(
        6.1319, 1.2228, // Lomé
        9.5511, 1.1865, // Kara
      );

      expect(distance).toBeGreaterThan(350);
      expect(distance).toBeLessThan(450);
    });
  });

  // ===========================================================================
  // GET ESTIMATED ARRIVAL TIME
  // ===========================================================================

  describe('getEstimatedArrivalTime', () => {
    it('devrait estimer le temps d\'arrivée', async () => {
      const result = await service.getEstimatedArrivalTime(
        6.1319, 1.2228, // Driver position
        6.1654, 1.2544, // Destination
      );

      expect(result).toHaveProperty('distanceKm');
      expect(result).toHaveProperty('durationMinutes');
      expect(result).toHaveProperty('arrivalTime');
      expect(result.durationMinutes).toBeGreaterThan(0);
      expect(result.arrivalTime).toBeInstanceOf(Date);
      expect(result.arrivalTime.getTime()).toBeGreaterThan(Date.now());
    });
  });
});
