import {
  Controller,
  Get,
  Post,
  Body,
  Param,
  UseGuards,
  ParseIntPipe,
} from '@nestjs/common';
import { ApiTags, ApiOperation, ApiBearerAuth } from '@nestjs/swagger';

import { PaymentService } from './payment.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantGuard } from '../../common/guards/merchant.guard';
import { CurrentUser, MerchantId } from '../../common/decorators/merchant.decorator';
import { Public } from '../../common/decorators/public.decorator';

import { AddCardDto } from './dto/add-card.dto';
import { ProcessPaymentDto } from './dto/process-payment.dto';
import { AddWalletFundsDto } from './dto/add-wallet-funds.dto';

@ApiTags('payments')
@Controller('payment')
@UseGuards(JwtAuthGuard, MerchantGuard)
@ApiBearerAuth()
export class PaymentController {
  constructor(private readonly paymentService: PaymentService) {}

  @Get('methods')
  @ApiOperation({ summary: 'Get available payment methods' })
  async getMethods(@MerchantId() merchantId: number) {
    return this.paymentService.getMethods(merchantId);
  }

  @Get('cards')
  @ApiOperation({ summary: 'Get user saved cards' })
  async getCards(@CurrentUser() user: any) {
    return this.paymentService.getCards(user.id, user.type);
  }

  @Post('cards')
  @ApiOperation({ summary: 'Add a new card' })
  async addCard(
    @CurrentUser() user: any,
    @Body() dto: AddCardDto,
  ) {
    return this.paymentService.addCard(user.id, user.type, dto);
  }

  @Post('cards/:id/delete')
  @ApiOperation({ summary: 'Delete a saved card' })
  async deleteCard(
    @CurrentUser() user: any,
    @Param('id', ParseIntPipe) cardId: number,
  ) {
    return this.paymentService.deleteCard(user.id, user.type, cardId);
  }

  @Post('process')
  @ApiOperation({ summary: 'Process a payment' })
  async process(
    @CurrentUser() user: any,
    @Body() dto: ProcessPaymentDto,
  ) {
    return this.paymentService.processPayment(user.id, user.type, dto);
  }

  @Post('wallet/add')
  @ApiOperation({ summary: 'Add funds to wallet' })
  async addWalletFunds(
    @CurrentUser() user: any,
    @Body() dto: AddWalletFundsDto,
  ) {
    return this.paymentService.addWalletFunds(user.id, user.type, dto);
  }

  // Webhook endpoints (public)
  @Public()
  @Post('webhook/stripe')
  @ApiOperation({ summary: 'Stripe webhook' })
  async stripeWebhook(@Body() body: any) {
    return this.paymentService.handleStripeWebhook(body);
  }
}
