import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import axios from 'axios';
import * as crypto from 'crypto';
import {
  PaymentGateway,
  InitializePaymentParams,
  InitializePaymentResult,
  VerifyPaymentResult,
  RefundResult,
  WebhookResult,
} from './gateway.interface';

@Injectable()
export class WaveGateway implements PaymentGateway {
  readonly name = 'wave';
  readonly displayName = 'Wave';
  readonly supportedCurrencies = ['XOF', 'XAF'];
  readonly supportedCountries = ['SN', 'CI', 'ML', 'BF', 'GM', 'UG'];

  private readonly logger = new Logger(WaveGateway.name);
  private readonly baseUrl = 'https://api.wave.com/v1';
  private apiKey: string;
  private webhookSecret: string;

  constructor(private configService: ConfigService) {
    this.apiKey = this.configService.get<string>('WAVE_API_KEY');
    this.webhookSecret = this.configService.get<string>('WAVE_WEBHOOK_SECRET');
  }

  isSupported(currency: string, country?: string): boolean {
    const currencySupported = this.supportedCurrencies.includes(currency.toUpperCase());
    if (!country) return currencySupported;
    return currencySupported && this.supportedCountries.includes(country.toUpperCase());
  }

  async initializePayment(params: InitializePaymentParams): Promise<InitializePaymentResult> {
    try {
      const response = await axios.post(
        `${this.baseUrl}/checkout/sessions`,
        {
          amount: Math.round(params.amount).toString(),
          currency: params.currency,
          error_url: `${params.callbackUrl}?status=error&reference=${params.reference}`,
          success_url: `${params.callbackUrl}?status=success&reference=${params.reference}`,
          client_reference: params.reference,
        },
        {
          headers: {
            Authorization: `Bearer ${this.apiKey}`,
            'Content-Type': 'application/json',
          },
        },
      );

      if (response.data.id) {
        return {
          success: true,
          paymentUrl: response.data.wave_launch_url,
          reference: params.reference,
          transactionId: response.data.id,
          rawResponse: response.data,
        };
      }

      return {
        success: false,
        message: 'Failed to create Wave checkout session',
        rawResponse: response.data,
      };
    } catch (error) {
      this.logger.error(`Wave initializePayment error: ${error.message}`);
      return {
        success: false,
        message: error.response?.data?.message || error.message,
      };
    }
  }

  async verifyPayment(reference: string): Promise<VerifyPaymentResult> {
    try {
      // Wave uses session ID for verification
      const response = await axios.get(
        `${this.baseUrl}/checkout/sessions/${reference}`,
        {
          headers: {
            Authorization: `Bearer ${this.apiKey}`,
          },
        },
      );

      const data = response.data;

      if (data.payment_status === 'succeeded') {
        return {
          success: true,
          status: 'success',
          amount: parseFloat(data.amount),
          currency: data.currency,
          reference: data.client_reference,
          transactionId: data.id,
          paidAt: new Date(data.when_completed),
          rawResponse: response.data,
        };
      }

      const statusMap: Record<string, VerifyPaymentResult['status']> = {
        succeeded: 'success',
        failed: 'failed',
        cancelled: 'cancelled',
        pending: 'pending',
        processing: 'pending',
      };

      return {
        success: false,
        status: statusMap[data.payment_status] || 'pending',
        message: `Payment status: ${data.payment_status}`,
        rawResponse: response.data,
      };
    } catch (error) {
      this.logger.error(`Wave verifyPayment error: ${error.message}`);
      return {
        success: false,
        status: 'failed',
        message: error.response?.data?.message || error.message,
      };
    }
  }

  async refundPayment(reference: string, amount?: number): Promise<RefundResult> {
    // Wave refunds are handled through their dashboard or support
    return {
      success: false,
      message: 'Les remboursements Wave doivent etre effectues via le support Wave',
    };
  }

  async handleWebhook(payload: any, signature?: string): Promise<WebhookResult> {
    try {
      // Verify webhook signature
      if (this.webhookSecret && signature) {
        const expectedSignature = crypto
          .createHmac('sha256', this.webhookSecret)
          .update(JSON.stringify(payload))
          .digest('hex');

        if (expectedSignature !== signature) {
          return { success: false, event: 'error', data: { error: 'Invalid signature' } };
        }
      }

      const eventType = payload.type;
      const data = payload.data;

      switch (eventType) {
        case 'checkout.session.completed':
          return {
            success: true,
            event: eventType,
            reference: data.client_reference,
            status: 'success',
            amount: parseFloat(data.amount),
            data,
          };

        case 'checkout.session.failed':
          return {
            success: true,
            event: eventType,
            reference: data.client_reference,
            status: 'failed',
            data,
          };

        case 'checkout.session.cancelled':
          return {
            success: true,
            event: eventType,
            reference: data.client_reference,
            status: 'cancelled',
            data,
          };

        default:
          return {
            success: true,
            event: eventType,
            data,
          };
      }
    } catch (error) {
      this.logger.error(`Wave webhook error: ${error.message}`);
      return { success: false, event: 'error', data: { error: error.message } };
    }
  }

  // ============================================================================
  // WAVE PAYOUT (B2C)
  // ============================================================================

  async sendMoney(
    phoneNumber: string,
    amount: number,
    currency: string,
    reference: string,
  ): Promise<{ success: boolean; transactionId?: string; message?: string }> {
    try {
      const response = await axios.post(
        `${this.baseUrl}/payout`,
        {
          receive_amount: Math.round(amount).toString(),
          currency,
          mobile: phoneNumber,
          client_reference: reference,
          name: 'Payout',
        },
        {
          headers: {
            Authorization: `Bearer ${this.apiKey}`,
            'Content-Type': 'application/json',
          },
        },
      );

      if (response.data.id) {
        return {
          success: true,
          transactionId: response.data.id,
        };
      }

      return { success: false, message: 'Payout failed' };
    } catch (error) {
      this.logger.error(`Wave sendMoney error: ${error.message}`);
      return { success: false, message: error.message };
    }
  }

  async checkPayoutStatus(
    payoutId: string,
  ): Promise<{ success: boolean; status?: string; message?: string }> {
    try {
      const response = await axios.get(
        `${this.baseUrl}/payout/${payoutId}`,
        {
          headers: {
            Authorization: `Bearer ${this.apiKey}`,
          },
        },
      );

      return {
        success: true,
        status: response.data.status,
      };
    } catch (error) {
      return { success: false, message: error.message };
    }
  }
}
