import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import Stripe from 'stripe';
import {
  PaymentGateway,
  InitializePaymentParams,
  InitializePaymentResult,
  VerifyPaymentResult,
  RefundResult,
  WebhookResult,
} from './gateway.interface';

@Injectable()
export class StripeGateway implements PaymentGateway {
  readonly name = 'stripe';
  readonly displayName = 'Stripe';
  readonly supportedCurrencies = ['USD', 'EUR', 'GBP', 'XOF', 'XAF', 'NGN', 'KES', 'GHS'];
  readonly supportedCountries = ['US', 'GB', 'FR', 'DE', 'SN', 'CI', 'TG', 'BJ', 'NG', 'GH', 'KE'];

  private readonly logger = new Logger(StripeGateway.name);
  private stripe: Stripe;

  constructor(private configService: ConfigService) {
    const secretKey = this.configService.get<string>('STRIPE_SECRET_KEY');
    if (secretKey) {
      this.stripe = new Stripe(secretKey, { apiVersion: '2023-10-16' });
    }
  }

  isSupported(currency: string, country?: string): boolean {
    const currencySupported = this.supportedCurrencies.includes(currency.toUpperCase());
    if (!country) return currencySupported;
    return currencySupported && this.supportedCountries.includes(country.toUpperCase());
  }

  async initializePayment(params: InitializePaymentParams): Promise<InitializePaymentResult> {
    try {
      // Create checkout session
      const session = await this.stripe.checkout.sessions.create({
        payment_method_types: ['card'],
        line_items: [
          {
            price_data: {
              currency: params.currency.toLowerCase(),
              product_data: {
                name: 'Paiement course',
                description: `Reference: ${params.reference}`,
              },
              unit_amount: Math.round(params.amount * 100), // Stripe uses cents
            },
            quantity: 1,
          },
        ],
        mode: 'payment',
        success_url: `${params.callbackUrl}?reference=${params.reference}&status=success`,
        cancel_url: `${params.callbackUrl}?reference=${params.reference}&status=cancelled`,
        customer_email: params.email,
        metadata: {
          reference: params.reference,
          ...params.metadata,
        },
      });

      return {
        success: true,
        paymentUrl: session.url,
        reference: params.reference,
        transactionId: session.id,
        rawResponse: session,
      };
    } catch (error) {
      this.logger.error(`Stripe initializePayment error: ${error.message}`);
      return {
        success: false,
        message: error.message,
      };
    }
  }

  async verifyPayment(reference: string): Promise<VerifyPaymentResult> {
    try {
      // Find session by reference in metadata
      const sessions = await this.stripe.checkout.sessions.list({
        limit: 1,
      });

      // In production, you'd search by metadata
      const session = sessions.data.find(
        (s) => s.metadata?.reference === reference,
      );

      if (!session) {
        return {
          success: false,
          status: 'pending',
          message: 'Session not found',
        };
      }

      const statusMap: Record<string, VerifyPaymentResult['status']> = {
        complete: 'success',
        expired: 'failed',
        open: 'pending',
      };

      return {
        success: session.payment_status === 'paid',
        status: statusMap[session.status] || 'pending',
        amount: session.amount_total ? session.amount_total / 100 : 0,
        currency: session.currency?.toUpperCase(),
        reference,
        transactionId: session.id,
        rawResponse: session,
      };
    } catch (error) {
      this.logger.error(`Stripe verifyPayment error: ${error.message}`);
      return {
        success: false,
        status: 'failed',
        message: error.message,
      };
    }
  }

  async refundPayment(reference: string, amount?: number): Promise<RefundResult> {
    try {
      // Get payment intent from session
      const sessions = await this.stripe.checkout.sessions.list({ limit: 100 });
      const session = sessions.data.find((s) => s.metadata?.reference === reference);

      if (!session || !session.payment_intent) {
        return { success: false, message: 'Payment not found' };
      }

      const refundParams: Stripe.RefundCreateParams = {
        payment_intent: session.payment_intent as string,
      };

      if (amount) {
        refundParams.amount = Math.round(amount * 100);
      }

      const refund = await this.stripe.refunds.create(refundParams);

      return {
        success: true,
        refundId: refund.id,
        amount: refund.amount / 100,
        status: refund.status,
      };
    } catch (error) {
      this.logger.error(`Stripe refundPayment error: ${error.message}`);
      return { success: false, message: error.message };
    }
  }

  async handleWebhook(payload: any, signature?: string): Promise<WebhookResult> {
    try {
      const webhookSecret = this.configService.get<string>('STRIPE_WEBHOOK_SECRET');

      let event: Stripe.Event;

      if (signature && webhookSecret) {
        event = this.stripe.webhooks.constructEvent(
          JSON.stringify(payload),
          signature,
          webhookSecret,
        );
      } else {
        event = payload as Stripe.Event;
      }

      switch (event.type) {
        case 'checkout.session.completed': {
          const session = event.data.object as Stripe.Checkout.Session;
          return {
            success: true,
            event: event.type,
            reference: session.metadata?.reference,
            status: session.payment_status === 'paid' ? 'success' : 'pending',
            amount: session.amount_total ? session.amount_total / 100 : 0,
            data: session,
          };
        }

        case 'payment_intent.payment_failed': {
          const paymentIntent = event.data.object as Stripe.PaymentIntent;
          return {
            success: true,
            event: event.type,
            reference: paymentIntent.metadata?.reference,
            status: 'failed',
            data: paymentIntent,
          };
        }

        default:
          return {
            success: true,
            event: event.type,
            data: event.data.object,
          };
      }
    } catch (error) {
      this.logger.error(`Stripe webhook error: ${error.message}`);
      return { success: false, event: 'error', data: { error: error.message } };
    }
  }
}
