import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import axios from 'axios';
import * as crypto from 'crypto';
import {
  PaymentGateway,
  InitializePaymentParams,
  InitializePaymentResult,
  VerifyPaymentResult,
  RefundResult,
  WebhookResult,
} from './gateway.interface';

@Injectable()
export class PaystackGateway implements PaymentGateway {
  readonly name = 'paystack';
  readonly displayName = 'Paystack';
  readonly supportedCurrencies = ['NGN', 'GHS', 'ZAR', 'KES'];
  readonly supportedCountries = ['NG', 'GH', 'ZA', 'KE'];

  private readonly logger = new Logger(PaystackGateway.name);
  private readonly baseUrl = 'https://api.paystack.co';
  private secretKey: string;
  private publicKey: string;

  constructor(private configService: ConfigService) {
    this.secretKey = this.configService.get<string>('PAYSTACK_SECRET_KEY');
    this.publicKey = this.configService.get<string>('PAYSTACK_PUBLIC_KEY');
  }

  isSupported(currency: string, country?: string): boolean {
    const currencySupported = this.supportedCurrencies.includes(currency.toUpperCase());
    if (!country) return currencySupported;
    return currencySupported && this.supportedCountries.includes(country.toUpperCase());
  }

  async initializePayment(params: InitializePaymentParams): Promise<InitializePaymentResult> {
    try {
      const response = await axios.post(
        `${this.baseUrl}/transaction/initialize`,
        {
          email: params.email,
          amount: Math.round(params.amount * 100), // Paystack uses kobo/pesewas
          currency: params.currency,
          reference: params.reference,
          callback_url: params.callbackUrl,
          metadata: {
            custom_fields: [
              {
                display_name: 'Reference',
                variable_name: 'reference',
                value: params.reference,
              },
            ],
            ...params.metadata,
          },
        },
        {
          headers: {
            Authorization: `Bearer ${this.secretKey}`,
            'Content-Type': 'application/json',
          },
        },
      );

      if (response.data.status) {
        return {
          success: true,
          paymentUrl: response.data.data.authorization_url,
          reference: response.data.data.reference,
          transactionId: response.data.data.access_code,
          rawResponse: response.data,
        };
      }

      return {
        success: false,
        message: response.data.message,
        rawResponse: response.data,
      };
    } catch (error) {
      this.logger.error(`Paystack initializePayment error: ${error.message}`);
      return {
        success: false,
        message: error.response?.data?.message || error.message,
      };
    }
  }

  async verifyPayment(reference: string): Promise<VerifyPaymentResult> {
    try {
      const response = await axios.get(
        `${this.baseUrl}/transaction/verify/${reference}`,
        {
          headers: {
            Authorization: `Bearer ${this.secretKey}`,
          },
        },
      );

      const data = response.data.data;

      if (response.data.status && data.status === 'success') {
        return {
          success: true,
          status: 'success',
          amount: data.amount / 100,
          currency: data.currency,
          reference: data.reference,
          transactionId: String(data.id),
          paidAt: new Date(data.paid_at),
          rawResponse: response.data,
        };
      }

      const statusMap: Record<string, VerifyPaymentResult['status']> = {
        success: 'success',
        failed: 'failed',
        abandoned: 'cancelled',
        pending: 'pending',
      };

      return {
        success: false,
        status: statusMap[data?.status] || 'pending',
        message: response.data.message,
        rawResponse: response.data,
      };
    } catch (error) {
      this.logger.error(`Paystack verifyPayment error: ${error.message}`);
      return {
        success: false,
        status: 'failed',
        message: error.response?.data?.message || error.message,
      };
    }
  }

  async refundPayment(reference: string, amount?: number): Promise<RefundResult> {
    try {
      // First get transaction ID
      const verifyResult = await this.verifyPayment(reference);
      if (!verifyResult.transactionId) {
        return { success: false, message: 'Transaction not found' };
      }

      const refundData: any = {
        transaction: verifyResult.transactionId,
      };

      if (amount) {
        refundData.amount = Math.round(amount * 100);
      }

      const response = await axios.post(
        `${this.baseUrl}/refund`,
        refundData,
        {
          headers: {
            Authorization: `Bearer ${this.secretKey}`,
            'Content-Type': 'application/json',
          },
        },
      );

      if (response.data.status) {
        return {
          success: true,
          refundId: String(response.data.data.id),
          amount: response.data.data.amount / 100,
          status: response.data.data.status,
        };
      }

      return {
        success: false,
        message: response.data.message,
      };
    } catch (error) {
      this.logger.error(`Paystack refundPayment error: ${error.message}`);
      return { success: false, message: error.message };
    }
  }

  async handleWebhook(payload: any, signature?: string): Promise<WebhookResult> {
    try {
      // Verify webhook signature
      if (signature) {
        const hash = crypto
          .createHmac('sha512', this.secretKey)
          .update(JSON.stringify(payload))
          .digest('hex');

        if (hash !== signature) {
          return { success: false, event: 'error', data: { error: 'Invalid signature' } };
        }
      }

      const event = payload.event;
      const data = payload.data;

      switch (event) {
        case 'charge.success':
          return {
            success: true,
            event,
            reference: data.reference,
            status: 'success',
            amount: data.amount / 100,
            data,
          };

        case 'charge.failed':
          return {
            success: true,
            event,
            reference: data.reference,
            status: 'failed',
            data,
          };

        case 'transfer.success':
          return {
            success: true,
            event,
            reference: data.reference,
            status: 'success',
            amount: data.amount / 100,
            data,
          };

        default:
          return {
            success: true,
            event,
            data,
          };
      }
    } catch (error) {
      this.logger.error(`Paystack webhook error: ${error.message}`);
      return { success: false, event: 'error', data: { error: error.message } };
    }
  }

  // ============================================================================
  // BANK TRANSFER
  // ============================================================================

  async createTransferRecipient(
    name: string,
    accountNumber: string,
    bankCode: string,
    currency: string = 'NGN',
  ): Promise<{ success: boolean; recipientCode?: string; message?: string }> {
    try {
      const response = await axios.post(
        `${this.baseUrl}/transferrecipient`,
        {
          type: 'nuban',
          name,
          account_number: accountNumber,
          bank_code: bankCode,
          currency,
        },
        {
          headers: {
            Authorization: `Bearer ${this.secretKey}`,
            'Content-Type': 'application/json',
          },
        },
      );

      if (response.data.status) {
        return {
          success: true,
          recipientCode: response.data.data.recipient_code,
        };
      }

      return { success: false, message: response.data.message };
    } catch (error) {
      return { success: false, message: error.message };
    }
  }

  async initiateTransfer(
    amount: number,
    recipientCode: string,
    reason: string,
    reference: string,
  ): Promise<{ success: boolean; transferCode?: string; message?: string }> {
    try {
      const response = await axios.post(
        `${this.baseUrl}/transfer`,
        {
          source: 'balance',
          amount: Math.round(amount * 100),
          recipient: recipientCode,
          reason,
          reference,
        },
        {
          headers: {
            Authorization: `Bearer ${this.secretKey}`,
            'Content-Type': 'application/json',
          },
        },
      );

      if (response.data.status) {
        return {
          success: true,
          transferCode: response.data.data.transfer_code,
        };
      }

      return { success: false, message: response.data.message };
    } catch (error) {
      return { success: false, message: error.message };
    }
  }

  async getBanks(country: string = 'nigeria'): Promise<any[]> {
    try {
      const response = await axios.get(
        `${this.baseUrl}/bank?country=${country}`,
        {
          headers: {
            Authorization: `Bearer ${this.secretKey}`,
          },
        },
      );

      return response.data.data || [];
    } catch (error) {
      this.logger.error(`Paystack getBanks error: ${error.message}`);
      return [];
    }
  }
}
