/**
 * Payment Gateway Interface
 * All payment gateways must implement this interface
 */
export interface PaymentGateway {
  /**
   * Gateway unique identifier
   */
  readonly name: string;

  /**
   * Gateway display name
   */
  readonly displayName: string;

  /**
   * Supported currencies
   */
  readonly supportedCurrencies: string[];

  /**
   * Supported countries (ISO codes)
   */
  readonly supportedCountries: string[];

  /**
   * Initialize payment
   */
  initializePayment(params: InitializePaymentParams): Promise<InitializePaymentResult>;

  /**
   * Verify payment status
   */
  verifyPayment(reference: string): Promise<VerifyPaymentResult>;

  /**
   * Process refund
   */
  refundPayment?(reference: string, amount?: number): Promise<RefundResult>;

  /**
   * Handle webhook
   */
  handleWebhook(payload: any, signature?: string): Promise<WebhookResult>;

  /**
   * Check if gateway supports a currency/country
   */
  isSupported(currency: string, country?: string): boolean;
}

export interface InitializePaymentParams {
  amount: number;
  currency: string;
  reference: string;
  email: string;
  phone?: string;
  name?: string;
  callbackUrl?: string;
  metadata?: Record<string, any>;
}

export interface InitializePaymentResult {
  success: boolean;
  paymentUrl?: string;
  reference?: string;
  transactionId?: string;
  message?: string;
  rawResponse?: any;
}

export interface VerifyPaymentResult {
  success: boolean;
  status: 'pending' | 'success' | 'failed' | 'cancelled';
  amount?: number;
  currency?: string;
  reference?: string;
  transactionId?: string;
  paidAt?: Date;
  message?: string;
  rawResponse?: any;
}

export interface RefundResult {
  success: boolean;
  refundId?: string;
  amount?: number;
  status?: string;
  message?: string;
}

export interface WebhookResult {
  success: boolean;
  event: string;
  reference?: string;
  status?: 'success' | 'failed' | 'pending';
  amount?: number;
  data?: any;
}

export enum PaymentGatewayType {
  // Card payments
  STRIPE = 'stripe',
  PAYPAL = 'paypal',
  FLUTTERWAVE = 'flutterwave',
  PAYSTACK = 'paystack',

  // Mobile Money - West Africa
  ORANGE_MONEY = 'orange_money',
  MTN_MOMO = 'mtn_momo',
  MOOV_MONEY = 'moov_money',
  WAVE = 'wave',
  FREE_MONEY = 'free_money',

  // Mobile Money - East Africa
  MPESA = 'mpesa',
  AIRTEL_MONEY = 'airtel_money',
  TIGO_PESA = 'tigo_pesa',

  // Aggregators
  CINETPAY = 'cinetpay',
  PAYDUNYA = 'paydunya',
  FEDAPAY = 'fedapay',
  HUBTEL = 'hubtel',
  MNOTIFY = 'mnotify',
  CAMPAY = 'campay',

  // Other
  CASH = 'cash',
  WALLET = 'wallet',
}
