import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import axios from 'axios';
import * as crypto from 'crypto';
import {
  PaymentGateway,
  InitializePaymentParams,
  InitializePaymentResult,
  VerifyPaymentResult,
  RefundResult,
  WebhookResult,
} from './gateway.interface';

@Injectable()
export class FlutterwaveGateway implements PaymentGateway {
  readonly name = 'flutterwave';
  readonly displayName = 'Flutterwave';
  readonly supportedCurrencies = ['NGN', 'GHS', 'KES', 'UGX', 'TZS', 'ZAR', 'XOF', 'XAF', 'USD', 'EUR', 'GBP'];
  readonly supportedCountries = ['NG', 'GH', 'KE', 'UG', 'TZ', 'ZA', 'SN', 'CI', 'CM', 'RW'];

  private readonly logger = new Logger(FlutterwaveGateway.name);
  private readonly baseUrl = 'https://api.flutterwave.com/v3';
  private secretKey: string;
  private publicKey: string;

  constructor(private configService: ConfigService) {
    this.secretKey = this.configService.get<string>('FLUTTERWAVE_SECRET_KEY');
    this.publicKey = this.configService.get<string>('FLUTTERWAVE_PUBLIC_KEY');
  }

  isSupported(currency: string, country?: string): boolean {
    const currencySupported = this.supportedCurrencies.includes(currency.toUpperCase());
    if (!country) return currencySupported;
    return currencySupported && this.supportedCountries.includes(country.toUpperCase());
  }

  async initializePayment(params: InitializePaymentParams): Promise<InitializePaymentResult> {
    try {
      const response = await axios.post(
        `${this.baseUrl}/payments`,
        {
          tx_ref: params.reference,
          amount: params.amount,
          currency: params.currency,
          redirect_url: params.callbackUrl,
          customer: {
            email: params.email,
            phonenumber: params.phone,
            name: params.name,
          },
          meta: params.metadata,
          customizations: {
            title: 'Paiement Course',
            description: `Reference: ${params.reference}`,
          },
        },
        {
          headers: {
            Authorization: `Bearer ${this.secretKey}`,
            'Content-Type': 'application/json',
          },
        },
      );

      if (response.data.status === 'success') {
        return {
          success: true,
          paymentUrl: response.data.data.link,
          reference: params.reference,
          rawResponse: response.data,
        };
      }

      return {
        success: false,
        message: response.data.message,
        rawResponse: response.data,
      };
    } catch (error) {
      this.logger.error(`Flutterwave initializePayment error: ${error.message}`);
      return {
        success: false,
        message: error.response?.data?.message || error.message,
      };
    }
  }

  async verifyPayment(reference: string): Promise<VerifyPaymentResult> {
    try {
      // First, get transaction by tx_ref
      const response = await axios.get(
        `${this.baseUrl}/transactions/verify_by_reference?tx_ref=${reference}`,
        {
          headers: {
            Authorization: `Bearer ${this.secretKey}`,
          },
        },
      );

      const data = response.data.data;

      if (response.data.status === 'success' && data.status === 'successful') {
        return {
          success: true,
          status: 'success',
          amount: data.amount,
          currency: data.currency,
          reference: data.tx_ref,
          transactionId: String(data.id),
          paidAt: new Date(data.created_at),
          rawResponse: response.data,
        };
      }

      const statusMap: Record<string, VerifyPaymentResult['status']> = {
        successful: 'success',
        failed: 'failed',
        pending: 'pending',
      };

      return {
        success: false,
        status: statusMap[data?.status] || 'pending',
        message: response.data.message,
        rawResponse: response.data,
      };
    } catch (error) {
      this.logger.error(`Flutterwave verifyPayment error: ${error.message}`);
      return {
        success: false,
        status: 'failed',
        message: error.response?.data?.message || error.message,
      };
    }
  }

  async refundPayment(reference: string, amount?: number): Promise<RefundResult> {
    try {
      // First verify to get transaction ID
      const verifyResult = await this.verifyPayment(reference);
      if (!verifyResult.transactionId) {
        return { success: false, message: 'Transaction not found' };
      }

      const refundData: any = {};
      if (amount) {
        refundData.amount = amount;
      }

      const response = await axios.post(
        `${this.baseUrl}/transactions/${verifyResult.transactionId}/refund`,
        refundData,
        {
          headers: {
            Authorization: `Bearer ${this.secretKey}`,
            'Content-Type': 'application/json',
          },
        },
      );

      if (response.data.status === 'success') {
        return {
          success: true,
          refundId: String(response.data.data.id),
          amount: response.data.data.amount_refunded,
          status: response.data.data.status,
        };
      }

      return {
        success: false,
        message: response.data.message,
      };
    } catch (error) {
      this.logger.error(`Flutterwave refundPayment error: ${error.message}`);
      return { success: false, message: error.message };
    }
  }

  async handleWebhook(payload: any, signature?: string): Promise<WebhookResult> {
    try {
      // Verify webhook signature
      const webhookSecret = this.configService.get<string>('FLUTTERWAVE_WEBHOOK_SECRET');

      if (webhookSecret && signature) {
        const hash = crypto
          .createHmac('sha256', webhookSecret)
          .update(JSON.stringify(payload))
          .digest('hex');

        if (hash !== signature) {
          return { success: false, event: 'error', data: { error: 'Invalid signature' } };
        }
      }

      const event = payload.event;
      const data = payload.data;

      switch (event) {
        case 'charge.completed':
          return {
            success: true,
            event,
            reference: data.tx_ref,
            status: data.status === 'successful' ? 'success' : 'failed',
            amount: data.amount,
            data,
          };

        case 'transfer.completed':
          return {
            success: true,
            event,
            reference: data.reference,
            status: data.status === 'SUCCESSFUL' ? 'success' : 'failed',
            amount: data.amount,
            data,
          };

        default:
          return {
            success: true,
            event,
            data,
          };
      }
    } catch (error) {
      this.logger.error(`Flutterwave webhook error: ${error.message}`);
      return { success: false, event: 'error', data: { error: error.message } };
    }
  }

  // ============================================================================
  // MOBILE MONEY SPECIFIC
  // ============================================================================

  async initializeMobileMoney(
    params: InitializePaymentParams & { provider: string; phoneNumber: string },
  ): Promise<InitializePaymentResult> {
    try {
      const response = await axios.post(
        `${this.baseUrl}/charges?type=mobile_money_${params.provider}`,
        {
          tx_ref: params.reference,
          amount: params.amount,
          currency: params.currency,
          phone_number: params.phoneNumber,
          email: params.email,
          fullname: params.name,
          meta: params.metadata,
        },
        {
          headers: {
            Authorization: `Bearer ${this.secretKey}`,
            'Content-Type': 'application/json',
          },
        },
      );

      if (response.data.status === 'success') {
        return {
          success: true,
          reference: params.reference,
          transactionId: String(response.data.data.id),
          message: response.data.data.processor_response || 'En attente de validation',
          rawResponse: response.data,
        };
      }

      return {
        success: false,
        message: response.data.message,
        rawResponse: response.data,
      };
    } catch (error) {
      this.logger.error(`Flutterwave MoMo error: ${error.message}`);
      return {
        success: false,
        message: error.response?.data?.message || error.message,
      };
    }
  }
}
