import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import axios from 'axios';
import * as crypto from 'crypto';
import {
  PaymentGateway,
  InitializePaymentParams,
  InitializePaymentResult,
  VerifyPaymentResult,
  RefundResult,
  WebhookResult,
} from './gateway.interface';

@Injectable()
export class CinetPayGateway implements PaymentGateway {
  readonly name = 'cinetpay';
  readonly displayName = 'CinetPay';
  readonly supportedCurrencies = ['XOF', 'XAF', 'GNF', 'CDF'];
  readonly supportedCountries = ['SN', 'CI', 'TG', 'BJ', 'BF', 'ML', 'NE', 'CM', 'GA', 'CG', 'GN', 'CD'];

  private readonly logger = new Logger(CinetPayGateway.name);
  private readonly baseUrl = 'https://api-checkout.cinetpay.com/v2';
  private apiKey: string;
  private siteId: string;
  private secretKey: string;

  constructor(private configService: ConfigService) {
    this.apiKey = this.configService.get<string>('CINETPAY_API_KEY');
    this.siteId = this.configService.get<string>('CINETPAY_SITE_ID');
    this.secretKey = this.configService.get<string>('CINETPAY_SECRET_KEY');
  }

  isSupported(currency: string, country?: string): boolean {
    const currencySupported = this.supportedCurrencies.includes(currency.toUpperCase());
    if (!country) return currencySupported;
    return currencySupported && this.supportedCountries.includes(country.toUpperCase());
  }

  async initializePayment(params: InitializePaymentParams): Promise<InitializePaymentResult> {
    try {
      const response = await axios.post(
        `${this.baseUrl}/payment`,
        {
          apikey: this.apiKey,
          site_id: this.siteId,
          transaction_id: params.reference,
          amount: Math.round(params.amount),
          currency: params.currency,
          description: `Paiement - Ref: ${params.reference}`,
          return_url: params.callbackUrl,
          notify_url: `${this.configService.get('APP_URL')}/api/payment/webhook/cinetpay`,
          customer_name: params.name || 'Client',
          customer_email: params.email,
          customer_phone_number: params.phone,
          customer_address: '',
          customer_city: '',
          customer_country: '',
          customer_state: '',
          customer_zip_code: '',
          metadata: JSON.stringify(params.metadata || {}),
          channels: 'ALL',
        },
        {
          headers: {
            'Content-Type': 'application/json',
          },
        },
      );

      if (response.data.code === '201') {
        return {
          success: true,
          paymentUrl: response.data.data.payment_url,
          reference: params.reference,
          transactionId: response.data.data.payment_token,
          rawResponse: response.data,
        };
      }

      return {
        success: false,
        message: response.data.message,
        rawResponse: response.data,
      };
    } catch (error) {
      this.logger.error(`CinetPay initializePayment error: ${error.message}`);
      return {
        success: false,
        message: error.response?.data?.message || error.message,
      };
    }
  }

  async verifyPayment(reference: string): Promise<VerifyPaymentResult> {
    try {
      const response = await axios.post(
        `${this.baseUrl}/payment/check`,
        {
          apikey: this.apiKey,
          site_id: this.siteId,
          transaction_id: reference,
        },
        {
          headers: {
            'Content-Type': 'application/json',
          },
        },
      );

      const data = response.data.data;

      if (response.data.code === '00' && data.status === 'ACCEPTED') {
        return {
          success: true,
          status: 'success',
          amount: Number(data.amount),
          currency: data.currency,
          reference: data.cpm_trans_id,
          transactionId: data.payment_token,
          paidAt: new Date(data.payment_date),
          rawResponse: response.data,
        };
      }

      const statusMap: Record<string, VerifyPaymentResult['status']> = {
        ACCEPTED: 'success',
        REFUSED: 'failed',
        CANCELLED: 'cancelled',
        PENDING: 'pending',
      };

      return {
        success: false,
        status: statusMap[data?.status] || 'pending',
        message: response.data.message,
        rawResponse: response.data,
      };
    } catch (error) {
      this.logger.error(`CinetPay verifyPayment error: ${error.message}`);
      return {
        success: false,
        status: 'failed',
        message: error.response?.data?.message || error.message,
      };
    }
  }

  async refundPayment(reference: string, amount?: number): Promise<RefundResult> {
    // CinetPay doesn't have a direct refund API
    // Refunds are handled manually through their dashboard
    return {
      success: false,
      message: 'Les remboursements CinetPay doivent etre effectues via le tableau de bord',
    };
  }

  async handleWebhook(payload: any, signature?: string): Promise<WebhookResult> {
    try {
      // Verify the payment
      const transactionId = payload.cpm_trans_id || payload.transaction_id;

      if (!transactionId) {
        return { success: false, event: 'error', data: { error: 'Missing transaction ID' } };
      }

      const verifyResult = await this.verifyPayment(transactionId);

      return {
        success: true,
        event: 'payment_notification',
        reference: transactionId,
        status: verifyResult.status,
        amount: verifyResult.amount,
        data: payload,
      };
    } catch (error) {
      this.logger.error(`CinetPay webhook error: ${error.message}`);
      return { success: false, event: 'error', data: { error: error.message } };
    }
  }
}
