import { Injectable, BadRequestException, NotFoundException } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';

// ============================================================================
// TYPES
// ============================================================================

export type PartnerType = 'hotel' | 'restaurant' | 'airport' | 'mall' | 'hospital' | 'corporate' | 'event_venue' | 'other';
export type PartnerStatus = 'pending' | 'active' | 'inactive' | 'suspended';

export interface Partner {
  id: number;
  name: string;
  type: PartnerType;
  status: PartnerStatus;
  description?: string;
  logo?: string;
  contactName: string;
  contactEmail: string;
  contactPhone: string;
  address: string;
  latitude: number;
  longitude: number;
  commissionRate: number; // Commission reversée au partenaire
  discountRate: number; // Réduction pour les clients venant du partenaire
  settings: PartnerSettings;
  stats: PartnerStats;
  apiKey?: string;
  webhookUrl?: string;
  merchantId: number;
  createdAt: Date;
  updatedAt: Date;
}

export interface PartnerSettings {
  autoCreateBookings: boolean;
  requireApproval: boolean;
  allowScheduled: boolean;
  maxBookingsPerDay: number;
  operatingHours?: { start: string; end: string };
  allowedVehicleTypes: number[];
  preferredDrivers: number[];
  customPricing?: {
    baseFareMultiplier: number;
    perKmMultiplier: number;
  };
}

export interface PartnerStats {
  totalBookings: number;
  completedBookings: number;
  totalRevenue: number;
  totalCommission: number;
  avgRating: number;
}

export interface PartnerLocation {
  id: number;
  partnerId: number;
  name: string;
  address: string;
  latitude: number;
  longitude: number;
  isDefault: boolean;
  active: boolean;
}

export interface PartnerBooking {
  id: number;
  partnerId: number;
  partnerReference?: string;
  bookingId: number;
  guestName: string;
  guestPhone: string;
  guestEmail?: string;
  pickupLocationId?: number;
  notes?: string;
  commission: number;
  status: 'pending' | 'confirmed' | 'completed' | 'cancelled';
  createdAt: Date;
}

export interface CreatePartnerDto {
  name: string;
  type: PartnerType;
  description?: string;
  logo?: string;
  contactName: string;
  contactEmail: string;
  contactPhone: string;
  address: string;
  latitude: number;
  longitude: number;
  commissionRate?: number;
  discountRate?: number;
  settings?: Partial<PartnerSettings>;
}

// ============================================================================
// SERVICE
// ============================================================================

@Injectable()
export class PartnerService {
  constructor(private readonly prisma: PrismaService) {}

  // ==========================================================================
  // PARTNER CRUD
  // ==========================================================================

  /**
   * Créer un partenaire
   */
  async createPartner(merchantId: number, data: CreatePartnerDto): Promise<Partner> {
    const defaultSettings: PartnerSettings = {
      autoCreateBookings: false,
      requireApproval: true,
      allowScheduled: true,
      maxBookingsPerDay: 100,
      allowedVehicleTypes: [],
      preferredDrivers: [],
    };

    // Générer une clé API
    const apiKey = this.generateApiKey();

    const partner = await this.prisma.partner.create({
      data: {
        merchantId,
        name: data.name,
        type: data.type,
        status: 'pending',
        description: data.description,
        logo: data.logo,
        contactName: data.contactName,
        contactEmail: data.contactEmail,
        contactPhone: data.contactPhone,
        address: data.address,
        latitude: data.latitude,
        longitude: data.longitude,
        commissionRate: data.commissionRate || 5,
        discountRate: data.discountRate || 0,
        settings: JSON.stringify({ ...defaultSettings, ...data.settings }),
        apiKey,
      },
    });

    // Créer la localisation par défaut
    await this.prisma.partnerLocation.create({
      data: {
        partnerId: partner.id,
        name: 'Principal',
        address: data.address,
        latitude: data.latitude,
        longitude: data.longitude,
        isDefault: true,
        active: true,
      },
    });

    return this.getPartner(partner.id, merchantId) as Promise<Partner>;
  }

  /**
   * Obtenir un partenaire
   */
  async getPartner(id: number, merchantId: number): Promise<Partner | null> {
    const partner = await this.prisma.partner.findFirst({
      where: { id, merchantId },
    });

    if (!partner) return null;

    const stats = await this.getPartnerStats(id);
    return this.mapPartner(partner, stats);
  }

  /**
   * Lister les partenaires
   */
  async listPartners(
    merchantId: number,
    options: {
      page?: number;
      limit?: number;
      type?: PartnerType;
      status?: PartnerStatus;
      search?: string;
    } = {},
  ): Promise<{ partners: Partner[]; total: number; page: number; totalPages: number }> {
    const page = options.page || 1;
    const limit = options.limit || 20;
    const skip = (page - 1) * limit;

    const where: any = { merchantId };
    if (options.type) where.type = options.type;
    if (options.status) where.status = options.status;
    if (options.search) {
      where.OR = [
        { name: { contains: options.search } },
        { contactName: { contains: options.search } },
        { contactEmail: { contains: options.search } },
      ];
    }

    const [partners, total] = await Promise.all([
      this.prisma.partner.findMany({
        where,
        skip,
        take: limit,
        orderBy: { name: 'asc' },
      }),
      this.prisma.partner.count({ where }),
    ]);

    const partnersWithStats = await Promise.all(
      partners.map(async (p) => {
        const stats = await this.getPartnerStats(p.id);
        return this.mapPartner(p, stats);
      }),
    );

    return {
      partners: partnersWithStats,
      total,
      page,
      totalPages: Math.ceil(total / limit),
    };
  }

  /**
   * Mettre à jour un partenaire
   */
  async updatePartner(
    id: number,
    merchantId: number,
    data: Partial<CreatePartnerDto & { status: PartnerStatus }>,
  ): Promise<Partner | null> {
    const partner = await this.prisma.partner.findFirst({
      where: { id, merchantId },
    });

    if (!partner) return null;

    const currentSettings = JSON.parse(partner.settings || '{}');
    const newSettings = data.settings
      ? { ...currentSettings, ...data.settings }
      : currentSettings;

    const updated = await this.prisma.partner.update({
      where: { id },
      data: {
        name: data.name,
        type: data.type,
        status: data.status,
        description: data.description,
        logo: data.logo,
        contactName: data.contactName,
        contactEmail: data.contactEmail,
        contactPhone: data.contactPhone,
        address: data.address,
        latitude: data.latitude,
        longitude: data.longitude,
        commissionRate: data.commissionRate,
        discountRate: data.discountRate,
        settings: JSON.stringify(newSettings),
      },
    });

    return this.getPartner(id, merchantId);
  }

  /**
   * Activer un partenaire
   */
  async activatePartner(id: number, merchantId: number): Promise<Partner | null> {
    return this.updatePartner(id, merchantId, { status: 'active' });
  }

  /**
   * Suspendre un partenaire
   */
  async suspendPartner(id: number, merchantId: number): Promise<Partner | null> {
    return this.updatePartner(id, merchantId, { status: 'suspended' });
  }

  /**
   * Régénérer la clé API
   */
  async regenerateApiKey(id: number, merchantId: number): Promise<{ apiKey: string } | null> {
    const partner = await this.prisma.partner.findFirst({
      where: { id, merchantId },
    });

    if (!partner) return null;

    const apiKey = this.generateApiKey();

    await this.prisma.partner.update({
      where: { id },
      data: { apiKey },
    });

    return { apiKey };
  }

  // ==========================================================================
  // PARTNER LOCATIONS
  // ==========================================================================

  /**
   * Ajouter une localisation
   */
  async addLocation(
    partnerId: number,
    merchantId: number,
    data: {
      name: string;
      address: string;
      latitude: number;
      longitude: number;
    },
  ): Promise<PartnerLocation> {
    const partner = await this.prisma.partner.findFirst({
      where: { id: partnerId, merchantId },
    });

    if (!partner) {
      throw new NotFoundException('Partenaire non trouvé');
    }

    const location = await this.prisma.partnerLocation.create({
      data: {
        partnerId,
        name: data.name,
        address: data.address,
        latitude: data.latitude,
        longitude: data.longitude,
        isDefault: false,
        active: true,
      },
    });

    return this.mapLocation(location);
  }

  /**
   * Lister les localisations
   */
  async getLocations(partnerId: number): Promise<PartnerLocation[]> {
    const locations = await this.prisma.partnerLocation.findMany({
      where: { partnerId, active: true },
      orderBy: [{ isDefault: 'desc' }, { name: 'asc' }],
    });

    return locations.map(this.mapLocation);
  }

  /**
   * Mettre à jour une localisation
   */
  async updateLocation(
    locationId: number,
    partnerId: number,
    data: Partial<{ name: string; address: string; latitude: number; longitude: number; active: boolean }>,
  ): Promise<PartnerLocation | null> {
    const location = await this.prisma.partnerLocation.findFirst({
      where: { id: locationId, partnerId },
    });

    if (!location) return null;

    const updated = await this.prisma.partnerLocation.update({
      where: { id: locationId },
      data,
    });

    return this.mapLocation(updated);
  }

  /**
   * Définir la localisation par défaut
   */
  async setDefaultLocation(locationId: number, partnerId: number): Promise<boolean> {
    const location = await this.prisma.partnerLocation.findFirst({
      where: { id: locationId, partnerId },
    });

    if (!location) return false;

    await this.prisma.$transaction([
      this.prisma.partnerLocation.updateMany({
        where: { partnerId, isDefault: true },
        data: { isDefault: false },
      }),
      this.prisma.partnerLocation.update({
        where: { id: locationId },
        data: { isDefault: true },
      }),
    ]);

    return true;
  }

  // ==========================================================================
  // PARTNER BOOKINGS
  // ==========================================================================

  /**
   * Créer une réservation partenaire
   */
  async createPartnerBooking(
    partnerId: number,
    merchantId: number,
    data: {
      guestName: string;
      guestPhone: string;
      guestEmail?: string;
      pickupLocationId?: number;
      dropoffAddress: string;
      dropoffLat: number;
      dropoffLng: number;
      vehicleTypeId: number;
      scheduledAt?: Date;
      notes?: string;
      partnerReference?: string;
    },
  ): Promise<PartnerBooking> {
    const partner = await this.prisma.partner.findFirst({
      where: { id: partnerId, merchantId, status: 'active' },
    });

    if (!partner) {
      throw new NotFoundException('Partenaire non trouvé ou inactif');
    }

    const settings = JSON.parse(partner.settings || '{}');

    // Vérifier la limite quotidienne
    const today = new Date();
    today.setHours(0, 0, 0, 0);
    const todayBookings = await this.prisma.partnerBooking.count({
      where: {
        partnerId,
        createdAt: { gte: today },
      },
    });

    if (todayBookings >= settings.maxBookingsPerDay) {
      throw new BadRequestException('Limite de réservations quotidiennes atteinte');
    }

    // Obtenir l'adresse de pickup
    let pickupAddress: string;
    let pickupLat: number;
    let pickupLng: number;

    if (data.pickupLocationId) {
      const location = await this.prisma.partnerLocation.findFirst({
        where: { id: data.pickupLocationId, partnerId },
      });
      if (!location) {
        throw new BadRequestException('Localisation non trouvée');
      }
      pickupAddress = location.address;
      pickupLat = location.latitude;
      pickupLng = location.longitude;
    } else {
      // Utiliser la localisation par défaut
      const defaultLocation = await this.prisma.partnerLocation.findFirst({
        where: { partnerId, isDefault: true },
      });
      if (!defaultLocation) {
        throw new BadRequestException('Aucune localisation par défaut');
      }
      pickupAddress = defaultLocation.address;
      pickupLat = defaultLocation.latitude;
      pickupLng = defaultLocation.longitude;
    }

    // Créer la réservation (via BookingService en production)
    const booking = await this.prisma.booking.create({
      data: {
        merchantId,
        // userId: null, // Pas d'utilisateur lié
        vehicleTypeId: data.vehicleTypeId,
        pickupAddress,
        pickupLat,
        pickupLng,
        dropoffAddress: data.dropoffAddress,
        dropoffLat: data.dropoffLat,
        dropoffLng: data.dropoffLng,
        scheduledAt: data.scheduledAt,
        status: settings.requireApproval ? 'pending' : 'searching',
        source: 'partner',
        sourceId: partnerId,
        notes: data.notes,
      },
    });

    // Calculer la commission
    const commission = 0; // Calculer après le tarif

    // Créer l'enregistrement partenaire
    const partnerBooking = await this.prisma.partnerBooking.create({
      data: {
        partnerId,
        bookingId: booking.id,
        guestName: data.guestName,
        guestPhone: data.guestPhone,
        guestEmail: data.guestEmail,
        pickupLocationId: data.pickupLocationId,
        partnerReference: data.partnerReference,
        notes: data.notes,
        commission: 0,
        status: 'pending',
      },
    });

    return this.mapPartnerBooking(partnerBooking);
  }

  /**
   * Lister les réservations partenaire
   */
  async getPartnerBookings(
    partnerId: number,
    merchantId: number,
    options: {
      page?: number;
      limit?: number;
      status?: string;
      dateFrom?: Date;
      dateTo?: Date;
    } = {},
  ): Promise<{ bookings: PartnerBooking[]; total: number }> {
    const partner = await this.prisma.partner.findFirst({
      where: { id: partnerId, merchantId },
    });

    if (!partner) return { bookings: [], total: 0 };

    const page = options.page || 1;
    const limit = options.limit || 20;
    const skip = (page - 1) * limit;

    const where: any = { partnerId };
    if (options.status) where.status = options.status;
    if (options.dateFrom || options.dateTo) {
      where.createdAt = {};
      if (options.dateFrom) where.createdAt.gte = options.dateFrom;
      if (options.dateTo) where.createdAt.lte = options.dateTo;
    }

    const [bookings, total] = await Promise.all([
      this.prisma.partnerBooking.findMany({
        where,
        include: {
          booking: {
            select: {
              bookingNumber: true,
              status: true,
              totalFare: true,
              pickupAddress: true,
              dropoffAddress: true,
            },
          },
        },
        skip,
        take: limit,
        orderBy: { createdAt: 'desc' },
      }),
      this.prisma.partnerBooking.count({ where }),
    ]);

    return {
      bookings: bookings.map(this.mapPartnerBooking),
      total,
    };
  }

  // ==========================================================================
  // PARTNER STATISTICS
  // ==========================================================================

  /**
   * Obtenir les statistiques d'un partenaire
   */
  async getPartnerStats(partnerId: number): Promise<PartnerStats> {
    const [bookingStats, ratings] = await Promise.all([
      this.prisma.partnerBooking.aggregate({
        where: { partnerId },
        _count: { id: true },
        _sum: { commission: true },
      }),
      this.prisma.$queryRaw<any[]>`
        SELECT
          COUNT(CASE WHEN b.status = 'completed' THEN 1 END) as completed,
          SUM(CASE WHEN b.status = 'completed' THEN b.total_fare ELSE 0 END) as revenue,
          AVG(b.rating) as avgRating
        FROM partner_bookings pb
        INNER JOIN bookings b ON b.id = pb.booking_id
        WHERE pb.partner_id = ${partnerId}
      `,
    ]);

    const stats = ratings[0] || {};

    return {
      totalBookings: bookingStats._count.id || 0,
      completedBookings: parseInt(stats.completed || '0'),
      totalRevenue: parseFloat(stats.revenue || '0'),
      totalCommission: parseFloat(bookingStats._sum.commission || '0'),
      avgRating: parseFloat(stats.avgRating || '0'),
    };
  }

  /**
   * Rapport partenaire
   */
  async getPartnerReport(
    partnerId: number,
    merchantId: number,
    startDate: Date,
    endDate: Date,
  ): Promise<{
    daily: any[];
    summary: Record<string, any>;
  }> {
    const partner = await this.prisma.partner.findFirst({
      where: { id: partnerId, merchantId },
    });

    if (!partner) {
      throw new NotFoundException('Partenaire non trouvé');
    }

    const daily = await this.prisma.$queryRaw<any[]>`
      SELECT
        DATE(pb.created_at) as date,
        COUNT(*) as bookings,
        SUM(CASE WHEN b.status = 'completed' THEN 1 ELSE 0 END) as completed,
        SUM(CASE WHEN b.status = 'completed' THEN b.total_fare ELSE 0 END) as revenue,
        SUM(pb.commission) as commission
      FROM partner_bookings pb
      INNER JOIN bookings b ON b.id = pb.booking_id
      WHERE pb.partner_id = ${partnerId}
        AND pb.created_at >= ${startDate}
        AND pb.created_at <= ${endDate}
      GROUP BY DATE(pb.created_at)
      ORDER BY date ASC
    `;

    const totals = await this.prisma.$queryRaw<any[]>`
      SELECT
        COUNT(*) as totalBookings,
        SUM(CASE WHEN b.status = 'completed' THEN 1 ELSE 0 END) as completedBookings,
        SUM(CASE WHEN b.status = 'completed' THEN b.total_fare ELSE 0 END) as totalRevenue,
        SUM(pb.commission) as totalCommission
      FROM partner_bookings pb
      INNER JOIN bookings b ON b.id = pb.booking_id
      WHERE pb.partner_id = ${partnerId}
        AND pb.created_at >= ${startDate}
        AND pb.created_at <= ${endDate}
    `;

    const summary = totals[0] || {};

    return {
      daily,
      summary: {
        totalBookings: parseInt(summary.totalBookings || '0'),
        completedBookings: parseInt(summary.completedBookings || '0'),
        totalRevenue: parseFloat(summary.totalRevenue || '0'),
        totalCommission: parseFloat(summary.totalCommission || '0'),
        completionRate: summary.totalBookings > 0
          ? (summary.completedBookings / summary.totalBookings) * 100
          : 0,
      },
    };
  }

  // ==========================================================================
  // HELPERS
  // ==========================================================================

  /**
   * Générer une clé API
   */
  private generateApiKey(): string {
    const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
    let key = 'pk_';
    for (let i = 0; i < 32; i++) {
      key += chars.charAt(Math.floor(Math.random() * chars.length));
    }
    return key;
  }

  private mapPartner(partner: any, stats?: PartnerStats): Partner {
    return {
      id: partner.id,
      name: partner.name,
      type: partner.type,
      status: partner.status,
      description: partner.description,
      logo: partner.logo,
      contactName: partner.contactName,
      contactEmail: partner.contactEmail,
      contactPhone: partner.contactPhone,
      address: partner.address,
      latitude: partner.latitude,
      longitude: partner.longitude,
      commissionRate: parseFloat(partner.commissionRate || '0'),
      discountRate: parseFloat(partner.discountRate || '0'),
      settings: JSON.parse(partner.settings || '{}'),
      stats: stats || {
        totalBookings: 0,
        completedBookings: 0,
        totalRevenue: 0,
        totalCommission: 0,
        avgRating: 0,
      },
      apiKey: partner.apiKey,
      webhookUrl: partner.webhookUrl,
      merchantId: partner.merchantId,
      createdAt: partner.createdAt,
      updatedAt: partner.updatedAt,
    };
  }

  private mapLocation(location: any): PartnerLocation {
    return {
      id: location.id,
      partnerId: location.partnerId,
      name: location.name,
      address: location.address,
      latitude: location.latitude,
      longitude: location.longitude,
      isDefault: location.isDefault,
      active: location.active,
    };
  }

  private mapPartnerBooking(pb: any): PartnerBooking {
    return {
      id: pb.id,
      partnerId: pb.partnerId,
      partnerReference: pb.partnerReference,
      bookingId: pb.bookingId,
      guestName: pb.guestName,
      guestPhone: pb.guestPhone,
      guestEmail: pb.guestEmail,
      pickupLocationId: pb.pickupLocationId,
      notes: pb.notes,
      commission: parseFloat(pb.commission || '0'),
      status: pb.status,
      createdAt: pb.createdAt,
      ...(pb.booking && { booking: pb.booking }),
    };
  }
}
