import { Injectable, Logger, OnModuleInit } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';

interface SmsResult {
  success: boolean;
  messageId?: string;
  error?: string;
}

@Injectable()
export class SmsService implements OnModuleInit {
  private readonly logger = new Logger(SmsService.name);
  private twilioClient: any = null;
  private twilioPhoneNumber: string;

  constructor(private configService: ConfigService) {}

  async onModuleInit() {
    await this.initializeTwilio();
  }

  private async initializeTwilio() {
    try {
      const accountSid = this.configService.get<string>('TWILIO_ACCOUNT_SID');
      const authToken = this.configService.get<string>('TWILIO_AUTH_TOKEN');
      this.twilioPhoneNumber = this.configService.get<string>('TWILIO_PHONE_NUMBER') || '';

      if (!accountSid || !authToken) {
        this.logger.warn('Twilio credentials not configured');
        return;
      }

      // Dynamic import to avoid issues if twilio is not installed
      const twilio = await import('twilio');
      this.twilioClient = twilio.default(accountSid, authToken);

      this.logger.log('Twilio SMS service initialized');
    } catch (error) {
      this.logger.error(`Failed to initialize Twilio: ${error.message}`);
    }
  }

  /**
   * Send SMS via Twilio
   */
  async sendSms(to: string, message: string): Promise<SmsResult> {
    if (!this.twilioClient) {
      this.logger.warn('Twilio not initialized, skipping SMS');
      return { success: false, error: 'SMS service not configured' };
    }

    try {
      // Format phone number (ensure it starts with +)
      const formattedPhone = this.formatPhoneNumber(to);

      const result = await this.twilioClient.messages.create({
        body: message,
        from: this.twilioPhoneNumber,
        to: formattedPhone,
      });

      this.logger.log(`SMS sent to ${formattedPhone}: ${result.sid}`);
      return { success: true, messageId: result.sid };
    } catch (error) {
      this.logger.error(`Failed to send SMS to ${to}: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  /**
   * Send OTP via SMS
   */
  async sendOtp(to: string, otp: string, merchantName = 'MonkAPI'): Promise<SmsResult> {
    const message = `[${merchantName}] Votre code de verification est: ${otp}. Valide pendant 5 minutes.`;
    return this.sendSms(to, message);
  }

  /**
   * Send booking confirmation SMS
   */
  async sendBookingConfirmation(
    to: string,
    bookingId: string,
    driverName: string,
    vehiclePlate: string,
  ): Promise<SmsResult> {
    const message = `Votre course ${bookingId} est confirmee. Chauffeur: ${driverName}, Vehicule: ${vehiclePlate}`;
    return this.sendSms(to, message);
  }

  /**
   * Send driver arrival SMS
   */
  async sendDriverArrivalSms(to: string, driverName: string): Promise<SmsResult> {
    const message = `Votre chauffeur ${driverName} est arrive au point de prise en charge.`;
    return this.sendSms(to, message);
  }

  /**
   * Send ride completion SMS with invoice
   */
  async sendRideCompletionSms(
    to: string,
    bookingId: string,
    amount: number,
    currency = 'XOF',
  ): Promise<SmsResult> {
    const message = `Course ${bookingId} terminee. Montant: ${amount} ${currency}. Merci d'avoir voyage avec nous!`;
    return this.sendSms(to, message);
  }

  /**
   * Send promotional SMS
   */
  async sendPromoSms(to: string, promoMessage: string): Promise<SmsResult> {
    return this.sendSms(to, promoMessage);
  }

  /**
   * Format phone number to E.164 format
   */
  private formatPhoneNumber(phone: string): string {
    // Remove spaces and dashes
    let cleaned = phone.replace(/[\s-]/g, '');

    // If it doesn't start with +, assume it's a local number
    if (!cleaned.startsWith('+')) {
      // Default to Togo country code if no country code provided
      if (cleaned.startsWith('00')) {
        cleaned = '+' + cleaned.substring(2);
      } else if (cleaned.startsWith('0')) {
        cleaned = '+228' + cleaned.substring(1); // Togo
      } else {
        cleaned = '+' + cleaned;
      }
    }

    return cleaned;
  }
}
