import { Injectable, Logger, OnModuleInit } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import axios, { AxiosInstance } from 'axios';

interface OneSignalNotification {
  title: string;
  message: string;
  data?: Record<string, any>;
  buttons?: Array<{ id: string; text: string; icon?: string }>;
  bigPicture?: string;
  sound?: string;
  ttl?: number; // Time to live in seconds
}

interface OneSignalResult {
  success: boolean;
  notificationId?: string;
  recipients?: number;
  error?: string;
}

@Injectable()
export class OneSignalService implements OnModuleInit {
  private readonly logger = new Logger(OneSignalService.name);
  private client: AxiosInstance | null = null;
  private appId: string;
  private userAppId: string;
  private driverAppId: string;

  constructor(private configService: ConfigService) {}

  async onModuleInit() {
    this.initializeOneSignal();
  }

  private initializeOneSignal() {
    try {
      const restApiKey = this.configService.get<string>('ONESIGNAL_REST_API_KEY');
      this.appId = this.configService.get<string>('ONESIGNAL_APP_ID') || '';
      this.userAppId = this.configService.get<string>('ONESIGNAL_USER_APP_ID') || this.appId;
      this.driverAppId = this.configService.get<string>('ONESIGNAL_DRIVER_APP_ID') || this.appId;

      if (!restApiKey || !this.appId) {
        this.logger.warn('OneSignal credentials not configured');
        return;
      }

      this.client = axios.create({
        baseURL: 'https://onesignal.com/api/v1',
        headers: {
          Authorization: `Basic ${restApiKey}`,
          'Content-Type': 'application/json',
        },
      });

      this.logger.log('OneSignal service initialized');
    } catch (error) {
      this.logger.error(`Failed to initialize OneSignal: ${error.message}`);
    }
  }

  /**
   * Send notification to specific player IDs
   */
  async sendToPlayers(
    playerIds: string[],
    notification: OneSignalNotification,
    appType: 'user' | 'driver' = 'user',
  ): Promise<OneSignalResult> {
    if (!this.client) {
      return { success: false, error: 'OneSignal not configured' };
    }

    try {
      const appId = appType === 'driver' ? this.driverAppId : this.userAppId;

      const payload: any = {
        app_id: appId,
        include_player_ids: playerIds,
        headings: { en: notification.title, fr: notification.title },
        contents: { en: notification.message, fr: notification.message },
        data: notification.data || {},
        ios_badgeType: 'Increase',
        ios_badgeCount: 1,
        android_channel_id: 'monkapi_channel',
      };

      // Add optional fields
      if (notification.buttons) {
        payload.buttons = notification.buttons;
      }
      if (notification.bigPicture) {
        payload.big_picture = notification.bigPicture;
        payload.ios_attachments = { picture: notification.bigPicture };
      }
      if (notification.sound) {
        payload.ios_sound = notification.sound;
        payload.android_sound = notification.sound;
      }
      if (notification.ttl) {
        payload.ttl = notification.ttl;
      }

      const response = await this.client.post('/notifications', payload);

      this.logger.log(`OneSignal notification sent: ${response.data.id}`);
      return {
        success: true,
        notificationId: response.data.id,
        recipients: response.data.recipients,
      };
    } catch (error) {
      this.logger.error(`Failed to send OneSignal notification: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  /**
   * Send notification to external user ID (user_id or driver_id)
   */
  async sendToExternalUserId(
    externalUserId: string,
    notification: OneSignalNotification,
    appType: 'user' | 'driver' = 'user',
  ): Promise<OneSignalResult> {
    if (!this.client) {
      return { success: false, error: 'OneSignal not configured' };
    }

    try {
      const appId = appType === 'driver' ? this.driverAppId : this.userAppId;

      const payload = {
        app_id: appId,
        include_external_user_ids: [externalUserId],
        channel_for_external_user_ids: 'push',
        headings: { en: notification.title, fr: notification.title },
        contents: { en: notification.message, fr: notification.message },
        data: notification.data || {},
      };

      const response = await this.client.post('/notifications', payload);

      return {
        success: true,
        notificationId: response.data.id,
        recipients: response.data.recipients,
      };
    } catch (error) {
      this.logger.error(`Failed to send to external user: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  /**
   * Send notification to segment (all users, drivers, etc.)
   */
  async sendToSegment(
    segment: string,
    notification: OneSignalNotification,
    appType: 'user' | 'driver' = 'user',
  ): Promise<OneSignalResult> {
    if (!this.client) {
      return { success: false, error: 'OneSignal not configured' };
    }

    try {
      const appId = appType === 'driver' ? this.driverAppId : this.userAppId;

      const payload = {
        app_id: appId,
        included_segments: [segment],
        headings: { en: notification.title, fr: notification.title },
        contents: { en: notification.message, fr: notification.message },
        data: notification.data || {},
      };

      const response = await this.client.post('/notifications', payload);

      return {
        success: true,
        notificationId: response.data.id,
        recipients: response.data.recipients,
      };
    } catch (error) {
      this.logger.error(`Failed to send to segment: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  /**
   * Send notification with filters
   */
  async sendWithFilters(
    filters: Array<{ field: string; key?: string; value: string; relation?: string }>,
    notification: OneSignalNotification,
    appType: 'user' | 'driver' = 'user',
  ): Promise<OneSignalResult> {
    if (!this.client) {
      return { success: false, error: 'OneSignal not configured' };
    }

    try {
      const appId = appType === 'driver' ? this.driverAppId : this.userAppId;

      const payload = {
        app_id: appId,
        filters,
        headings: { en: notification.title, fr: notification.title },
        contents: { en: notification.message, fr: notification.message },
        data: notification.data || {},
      };

      const response = await this.client.post('/notifications', payload);

      return {
        success: true,
        notificationId: response.data.id,
        recipients: response.data.recipients,
      };
    } catch (error) {
      this.logger.error(`Failed to send with filters: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  // =============================================================================
  // BOOKING NOTIFICATIONS
  // =============================================================================

  /**
   * Notify driver of new booking request
   */
  async notifyNewBookingRequest(
    driverPlayerId: string,
    booking: any,
  ): Promise<OneSignalResult> {
    return this.sendToPlayers(
      [driverPlayerId],
      {
        title: '🚗 Nouvelle course disponible',
        message: `De: ${booking.pickup_address}\nVers: ${booking.drop_address}`,
        data: {
          type: 'new_booking_request',
          booking_id: String(booking.id),
          pickup_lat: String(booking.pickup_latitude),
          pickup_lng: String(booking.pickup_longitude),
          drop_lat: String(booking.drop_latitude),
          drop_lng: String(booking.drop_longitude),
          amount: String(booking.estimate_amount),
        },
        buttons: [
          { id: 'accept', text: 'Accepter' },
          { id: 'reject', text: 'Refuser' },
        ],
        ttl: 30, // Expire after 30 seconds
        sound: 'new_ride',
      },
      'driver',
    );
  }

  /**
   * Notify user that driver accepted
   */
  async notifyBookingAccepted(
    userPlayerId: string,
    booking: any,
    driver: any,
  ): Promise<OneSignalResult> {
    return this.sendToPlayers(
      [userPlayerId],
      {
        title: '✅ Course confirmée',
        message: `${driver.first_name} arrive dans ${booking.eta || 5} minutes`,
        data: {
          type: 'booking_accepted',
          booking_id: String(booking.id),
          driver_id: String(driver.id),
          driver_name: driver.first_name,
          driver_phone: driver.phoneNumber,
          driver_photo: driver.profile_image,
          driver_lat: String(driver.latitude),
          driver_lng: String(driver.longitude),
        },
        sound: 'booking_accepted',
      },
      'user',
    );
  }

  /**
   * Notify user that driver arrived
   */
  async notifyDriverArrived(userPlayerId: string, booking: any): Promise<OneSignalResult> {
    return this.sendToPlayers(
      [userPlayerId],
      {
        title: '📍 Chauffeur arrivé',
        message: 'Votre chauffeur vous attend au point de prise en charge',
        data: {
          type: 'driver_arrived',
          booking_id: String(booking.id),
        },
        sound: 'driver_arrived',
      },
      'user',
    );
  }

  /**
   * Notify user that ride started
   */
  async notifyRideStarted(userPlayerId: string, booking: any): Promise<OneSignalResult> {
    return this.sendToPlayers(
      [userPlayerId],
      {
        title: '🚀 Course démarrée',
        message: `En route vers ${booking.drop_address}`,
        data: {
          type: 'ride_started',
          booking_id: String(booking.id),
        },
      },
      'user',
    );
  }

  /**
   * Notify user that ride completed
   */
  async notifyRideCompleted(
    userPlayerId: string,
    booking: any,
  ): Promise<OneSignalResult> {
    return this.sendToPlayers(
      [userPlayerId],
      {
        title: '🏁 Course terminée',
        message: `Montant: ${booking.final_amount} XOF`,
        data: {
          type: 'ride_completed',
          booking_id: String(booking.id),
          amount: String(booking.final_amount),
        },
        buttons: [{ id: 'rate', text: 'Noter le chauffeur' }],
      },
      'user',
    );
  }

  /**
   * Notify about booking cancellation
   */
  async notifyBookingCancelled(
    playerId: string,
    appType: 'user' | 'driver',
    booking: any,
    cancelledBy: string,
  ): Promise<OneSignalResult> {
    return this.sendToPlayers(
      [playerId],
      {
        title: '❌ Course annulée',
        message: `La course a été annulée par ${cancelledBy}`,
        data: {
          type: 'booking_cancelled',
          booking_id: String(booking.id),
          cancelled_by: cancelledBy,
        },
      },
      appType,
    );
  }

  /**
   * Notify driver of new earnings
   */
  async notifyDriverEarnings(
    driverPlayerId: string,
    booking: any,
    earnings: number,
  ): Promise<OneSignalResult> {
    return this.sendToPlayers(
      [driverPlayerId],
      {
        title: '💰 Nouveau gain',
        message: `+${earnings} XOF pour la course #${booking.merchant_booking_id}`,
        data: {
          type: 'new_earnings',
          booking_id: String(booking.id),
          amount: String(earnings),
        },
      },
      'driver',
    );
  }

  // =============================================================================
  // DEVICE MANAGEMENT
  // =============================================================================

  /**
   * Set external user ID for a device
   */
  async setExternalUserId(
    playerId: string,
    externalUserId: string,
    appType: 'user' | 'driver' = 'user',
  ): Promise<boolean> {
    if (!this.client) return false;

    try {
      const appId = appType === 'driver' ? this.driverAppId : this.userAppId;

      await this.client.put(`/players/${playerId}`, {
        app_id: appId,
        external_user_id: externalUserId,
      });

      return true;
    } catch (error) {
      this.logger.error(`Failed to set external user ID: ${error.message}`);
      return false;
    }
  }

  /**
   * Add tags to a device
   */
  async setDeviceTags(
    playerId: string,
    tags: Record<string, string | number>,
    appType: 'user' | 'driver' = 'user',
  ): Promise<boolean> {
    if (!this.client) return false;

    try {
      const appId = appType === 'driver' ? this.driverAppId : this.userAppId;

      await this.client.put(`/players/${playerId}`, {
        app_id: appId,
        tags,
      });

      return true;
    } catch (error) {
      this.logger.error(`Failed to set device tags: ${error.message}`);
      return false;
    }
  }

  /**
   * Get device info
   */
  async getDevice(playerId: string, appType: 'user' | 'driver' = 'user'): Promise<any> {
    if (!this.client) return null;

    try {
      const appId = appType === 'driver' ? this.driverAppId : this.userAppId;

      const response = await this.client.get(`/players/${playerId}`, {
        params: { app_id: appId },
      });

      return response.data;
    } catch (error) {
      this.logger.error(`Failed to get device: ${error.message}`);
      return null;
    }
  }

  /**
   * Cancel a scheduled notification
   */
  async cancelNotification(notificationId: string): Promise<boolean> {
    if (!this.client) return false;

    try {
      await this.client.delete(`/notifications/${notificationId}`, {
        params: { app_id: this.appId },
      });
      return true;
    } catch (error) {
      this.logger.error(`Failed to cancel notification: ${error.message}`);
      return false;
    }
  }
}
