import {
  Controller,
  Get,
  Post,
  Put,
  Body,
  Param,
  Query,
  UseGuards,
  ParseIntPipe,
} from '@nestjs/common';
import { ApiTags, ApiOperation, ApiBearerAuth } from '@nestjs/swagger';
import { HandymanService, BookingStatus } from './handyman.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantGuard } from '../../common/guards/merchant.guard';
import { MerchantId, CurrentUser } from '../../common/decorators/merchant.decorator';

@ApiTags('Handyman')
@Controller('api/handyman')
@UseGuards(JwtAuthGuard, MerchantGuard)
@ApiBearerAuth()
export class HandymanController {
  constructor(private handymanService: HandymanService) {}

  // ============================================================================
  // PUBLIC ENDPOINTS (Users)
  // ============================================================================

  @Get('categories')
  @ApiOperation({ summary: 'Get service categories' })
  async getCategories(@MerchantId() merchantId: number) {
    return this.handymanService.getServiceCategories(merchantId);
  }

  @Get('categories/:id/services')
  @ApiOperation({ summary: 'Get services by category' })
  async getServicesByCategory(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) categoryId: number,
  ) {
    return this.handymanService.getServicesByCategory(merchantId, categoryId);
  }

  @Get('services')
  @ApiOperation({ summary: 'Get all services' })
  async getAllServices(
    @MerchantId() merchantId: number,
    @Query('categoryId') categoryId?: string,
    @Query('search') search?: string,
  ) {
    return this.handymanService.getAllServices(merchantId, {
      categoryId: categoryId ? parseInt(categoryId) : undefined,
      search,
    });
  }

  @Get('services/:id')
  @ApiOperation({ summary: 'Get service details' })
  async getServiceDetails(@Param('id', ParseIntPipe) serviceId: number) {
    return this.handymanService.getServiceDetails(serviceId);
  }

  @Post('bookings')
  @ApiOperation({ summary: 'Create booking' })
  async createBooking(
    @CurrentUser() user: any,
    @MerchantId() merchantId: number,
    @Body()
    body: {
      serviceId: number;
      scheduledDate: string;
      scheduledTime: string;
      address: string;
      latitude: number;
      longitude: number;
      description?: string;
      images?: string[];
    },
  ) {
    return this.handymanService.createBooking(user.id, merchantId, {
      serviceId: body.serviceId,
      scheduledDate: new Date(body.scheduledDate),
      scheduledTime: body.scheduledTime,
      address: body.address,
      latitude: body.latitude,
      longitude: body.longitude,
      description: body.description,
      images: body.images,
    });
  }

  @Get('bookings')
  @ApiOperation({ summary: 'Get my bookings' })
  async getMyBookings(
    @CurrentUser() user: any,
    @Query('status') status?: BookingStatus,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
  ) {
    return this.handymanService.getUserBookings(user.id, {
      status,
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 20,
    });
  }

  @Get('bookings/:id')
  @ApiOperation({ summary: 'Get booking details' })
  async getBookingDetails(@Param('id', ParseIntPipe) bookingId: number) {
    return this.handymanService.getBookingDetails(bookingId);
  }

  @Post('bookings/:id/cancel')
  @ApiOperation({ summary: 'Cancel booking' })
  async cancelBooking(
    @Param('id', ParseIntPipe) bookingId: number,
    @CurrentUser() user: any,
    @Body() body?: { reason?: string },
  ) {
    await this.handymanService.cancelBooking(bookingId, user.id, body?.reason);
    return { message: 'Reservation annulee' };
  }

  // ============================================================================
  // PROVIDER ENDPOINTS
  // ============================================================================

  @Get('provider/bookings')
  @ApiOperation({ summary: 'Get provider bookings' })
  async getProviderBookings(
    @CurrentUser() provider: any,
    @Query('status') status?: BookingStatus,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
  ) {
    return this.handymanService.getProviderBookings(provider.id, {
      status,
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 20,
    });
  }

  @Post('provider/bookings/:id/accept')
  @ApiOperation({ summary: 'Accept booking (provider)' })
  async acceptBooking(
    @Param('id', ParseIntPipe) bookingId: number,
    @CurrentUser() provider: any,
  ) {
    return this.handymanService.providerAcceptBooking(bookingId, provider.id);
  }

  @Post('provider/bookings/:id/en-route')
  @ApiOperation({ summary: 'Mark en route (provider)' })
  async markEnRoute(
    @Param('id', ParseIntPipe) bookingId: number,
    @CurrentUser() provider: any,
  ) {
    return this.handymanService.providerEnRoute(bookingId, provider.id);
  }

  @Post('provider/bookings/:id/arrived')
  @ApiOperation({ summary: 'Mark arrived (provider)' })
  async markArrived(
    @Param('id', ParseIntPipe) bookingId: number,
    @CurrentUser() provider: any,
  ) {
    return this.handymanService.providerArrived(bookingId, provider.id);
  }

  @Post('provider/bookings/:id/start')
  @ApiOperation({ summary: 'Start service (provider)' })
  async startService(
    @Param('id', ParseIntPipe) bookingId: number,
    @CurrentUser() provider: any,
  ) {
    return this.handymanService.startService(bookingId, provider.id);
  }

  @Post('provider/bookings/:id/complete')
  @ApiOperation({ summary: 'Complete service (provider)' })
  async completeService(
    @Param('id', ParseIntPipe) bookingId: number,
    @CurrentUser() provider: any,
    @Body()
    body: {
      finalPrice: number;
      workDuration?: number;
      completionNotes?: string;
      images?: string[];
    },
  ) {
    return this.handymanService.completeService(bookingId, provider.id, body);
  }

  @Put('provider/location')
  @ApiOperation({ summary: 'Update location (provider)' })
  async updateLocation(
    @CurrentUser() provider: any,
    @Body() body: { latitude: number; longitude: number },
  ) {
    await this.handymanService.updateProviderLocation(
      provider.id,
      body.latitude,
      body.longitude,
    );
    return { message: 'Location mise a jour' };
  }

  @Put('provider/status')
  @ApiOperation({ summary: 'Toggle online status (provider)' })
  async toggleOnlineStatus(
    @CurrentUser() provider: any,
    @Body() body: { isOnline: boolean },
  ) {
    await this.handymanService.toggleProviderOnlineStatus(provider.id, body.isOnline);
    return { message: body.isOnline ? 'Vous etes maintenant en ligne' : 'Vous etes maintenant hors ligne' };
  }

  // ============================================================================
  // ADMIN ENDPOINTS
  // ============================================================================

  @Post('admin/categories')
  @ApiOperation({ summary: 'Create category (admin)' })
  async createCategory(
    @MerchantId() merchantId: number,
    @Body()
    body: {
      name: string;
      description?: string;
      icon?: string;
      sortOrder?: number;
    },
  ) {
    return this.handymanService.createCategory(merchantId, body);
  }

  @Post('admin/services')
  @ApiOperation({ summary: 'Create service (admin)' })
  async createService(
    @MerchantId() merchantId: number,
    @Body()
    body: {
      categoryId: number;
      name: string;
      description?: string;
      basePrice: number;
      icon?: string;
      estimatedDuration?: number;
    },
  ) {
    return this.handymanService.createService(merchantId, body);
  }

  @Get('admin/providers')
  @ApiOperation({ summary: 'Get all providers (admin)' })
  async getAllProviders(
    @MerchantId() merchantId: number,
    @Query('isOnline') isOnline?: string,
    @Query('categoryId') categoryId?: string,
  ) {
    return this.handymanService.getAllProviders(merchantId, {
      isOnline: isOnline ? isOnline === 'true' : undefined,
      categoryId: categoryId ? parseInt(categoryId) : undefined,
    });
  }

  @Post('admin/bookings/:id/assign')
  @ApiOperation({ summary: 'Assign provider to booking (admin)' })
  async assignProvider(
    @Param('id', ParseIntPipe) bookingId: number,
    @Body() body: { providerId: number },
  ) {
    return this.handymanService.assignProvider(bookingId, body.providerId);
  }
}
