import { Injectable, Logger, OnModuleInit } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import { GeocodingProvider } from './geocoding.interface';
import { GoogleMapsProvider } from './google-maps.provider';
import { OpenStreetMapProvider } from './openstreetmap.provider';

@Injectable()
export class GeocodingProviderFactory implements OnModuleInit {
  private readonly logger = new Logger(GeocodingProviderFactory.name);
  private providers: Map<string, GeocodingProvider> = new Map();
  private defaultProvider: string;

  constructor(
    private configService: ConfigService,
    private googleMapsProvider: GoogleMapsProvider,
    private osmProvider: OpenStreetMapProvider,
  ) {
    this.defaultProvider = this.configService.get<string>('GEOCODING_DEFAULT_PROVIDER') || 'google';
  }

  onModuleInit() {
    // Register providers
    this.registerProvider(this.googleMapsProvider);
    this.registerProvider(this.osmProvider);

    this.logger.log(`Registered ${this.providers.size} geocoding providers. Default: ${this.defaultProvider}`);
  }

  /**
   * Register a provider
   */
  registerProvider(provider: GeocodingProvider): void {
    this.providers.set(provider.name, provider);
    this.logger.debug(`Registered geocoding provider: ${provider.name}`);
  }

  /**
   * Get provider by name
   */
  getProvider(name?: string): GeocodingProvider | undefined {
    return this.providers.get(name || this.defaultProvider);
  }

  /**
   * Get default provider
   */
  getDefaultProvider(): GeocodingProvider | undefined {
    return this.getProvider(this.defaultProvider);
  }

  /**
   * Get all providers
   */
  getAllProviders(): GeocodingProvider[] {
    return Array.from(this.providers.values());
  }

  /**
   * Get provider info
   */
  getProviderInfo(): Array<{ name: string; displayName: string; isDefault: boolean }> {
    return this.getAllProviders().map(provider => ({
      name: provider.name,
      displayName: provider.displayName,
      isDefault: provider.name === this.defaultProvider,
    }));
  }

  /**
   * Get best provider for country
   * Google Maps has better coverage in Africa
   */
  getBestProviderForCountry(countryCode: string): GeocodingProvider {
    // Google Maps is generally better for Africa
    const africaCountries = [
      'SN', 'CI', 'TG', 'BJ', 'BF', 'ML', 'NE', 'GN', 'GH', 'NG',
      'CM', 'GA', 'CG', 'CD', 'KE', 'UG', 'TZ', 'RW', 'ET', 'ZA',
    ];

    if (africaCountries.includes(countryCode.toUpperCase())) {
      const google = this.providers.get('google');
      if (google) return google;
    }

    return this.getDefaultProvider();
  }
}
