import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import axios from 'axios';
import {
  GeocodingProvider,
  GeocodingResult,
  ReverseGeocodingResult,
  AutocompleteResult,
  Coordinates,
} from './geocoding.interface';

@Injectable()
export class OpenStreetMapProvider implements GeocodingProvider {
  readonly name = 'osm';
  readonly displayName = 'OpenStreetMap (Nominatim)';

  private readonly logger = new Logger(OpenStreetMapProvider.name);
  private readonly nominatimUrl = 'https://nominatim.openstreetmap.org';
  private readonly userAgent: string;

  constructor(private configService: ConfigService) {
    this.userAgent = this.configService.get<string>('OSM_USER_AGENT') || 'MonkAPI/1.0';
  }

  async geocode(
    address: string,
    options?: { country?: string },
  ): Promise<GeocodingResult> {
    try {
      const params: any = {
        q: address,
        format: 'json',
        addressdetails: 1,
        limit: 1,
      };

      if (options?.country) {
        params.countrycodes = options.country.toLowerCase();
      }

      const response = await axios.get(`${this.nominatimUrl}/search`, {
        params,
        headers: { 'User-Agent': this.userAgent },
      });

      if (!response.data.length) {
        return { success: false, error: 'No results found' };
      }

      const result = response.data[0];
      const components = this.parseAddressComponents(result.address);

      return {
        success: true,
        address,
        formattedAddress: result.display_name,
        coordinates: {
          lat: parseFloat(result.lat),
          lng: parseFloat(result.lon),
        },
        components,
        placeId: result.place_id?.toString(),
        confidence: this.getConfidence(result.importance),
      };
    } catch (error) {
      this.logger.error(`Geocode error: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  async reverseGeocode(lat: number, lng: number): Promise<ReverseGeocodingResult> {
    try {
      const response = await axios.get(`${this.nominatimUrl}/reverse`, {
        params: {
          lat,
          lon: lng,
          format: 'json',
          addressdetails: 1,
        },
        headers: { 'User-Agent': this.userAgent },
      });

      if (response.data.error) {
        return { success: false, error: response.data.error };
      }

      const result = response.data;
      const components = this.parseAddressComponents(result.address);

      return {
        success: true,
        address: result.display_name,
        formattedAddress: result.display_name,
        coordinates: { lat, lng },
        components,
        placeId: result.place_id?.toString(),
      };
    } catch (error) {
      this.logger.error(`Reverse geocode error: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  async autocomplete(
    input: string,
    options?: {
      location?: Coordinates;
      radius?: number;
      types?: string[];
      country?: string;
    },
  ): Promise<AutocompleteResult> {
    try {
      const params: any = {
        q: input,
        format: 'json',
        addressdetails: 1,
        limit: 10,
      };

      if (options?.country) {
        params.countrycodes = options.country.toLowerCase();
      }

      if (options?.location && options?.radius) {
        // Nominatim uses viewbox for bounded search
        const latDelta = options.radius / 111000; // approx degrees per meter
        const lngDelta = options.radius / (111000 * Math.cos(options.location.lat * Math.PI / 180));

        params.viewbox = [
          options.location.lng - lngDelta,
          options.location.lat + latDelta,
          options.location.lng + lngDelta,
          options.location.lat - latDelta,
        ].join(',');
        params.bounded = 1;
      }

      const response = await axios.get(`${this.nominatimUrl}/search`, {
        params,
        headers: { 'User-Agent': this.userAgent },
      });

      return {
        success: true,
        predictions: response.data.map((result: any) => ({
          placeId: result.place_id?.toString() || '',
          description: result.display_name,
          mainText: this.getMainText(result),
          secondaryText: this.getSecondaryText(result),
          types: result.class ? [result.class] : [],
        })),
      };
    } catch (error) {
      this.logger.error(`Autocomplete error: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  private parseAddressComponents(address: any): GeocodingResult['components'] {
    if (!address) return {};

    return {
      streetNumber: address.house_number,
      street: address.road || address.street,
      neighborhood: address.neighbourhood || address.suburb,
      city: address.city || address.town || address.village,
      state: address.state || address.region,
      country: address.country,
      countryCode: address.country_code?.toUpperCase(),
      postalCode: address.postcode,
    };
  }

  private getConfidence(importance: number): number {
    // Nominatim importance is 0-1, where higher is better
    return importance || 0.5;
  }

  private getMainText(result: any): string {
    if (result.address) {
      if (result.address.house_number && result.address.road) {
        return `${result.address.house_number} ${result.address.road}`;
      }
      if (result.address.road) return result.address.road;
      if (result.address.neighbourhood) return result.address.neighbourhood;
    }
    const parts = result.display_name.split(',');
    return parts[0]?.trim() || result.display_name;
  }

  private getSecondaryText(result: any): string {
    if (result.address) {
      const parts = [];
      if (result.address.neighbourhood) parts.push(result.address.neighbourhood);
      if (result.address.city || result.address.town) {
        parts.push(result.address.city || result.address.town);
      }
      if (result.address.country) parts.push(result.address.country);
      return parts.join(', ');
    }
    const parts = result.display_name.split(',');
    return parts.slice(1).join(',').trim();
  }
}
