import { Injectable, Logger } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';
import axios from 'axios';
import {
  GeocodingProvider,
  GeocodingResult,
  ReverseGeocodingResult,
  AutocompleteResult,
  PlaceDetailsResult,
  DirectionsResult,
  DistanceMatrixResult,
  Coordinates,
} from './geocoding.interface';

@Injectable()
export class GoogleMapsProvider implements GeocodingProvider {
  readonly name = 'google';
  readonly displayName = 'Google Maps';

  private readonly logger = new Logger(GoogleMapsProvider.name);
  private readonly apiKey: string;
  private readonly baseUrl = 'https://maps.googleapis.com/maps/api';

  constructor(private configService: ConfigService) {
    this.apiKey = this.configService.get<string>('GOOGLE_MAPS_API_KEY');
  }

  async geocode(
    address: string,
    options?: { country?: string },
  ): Promise<GeocodingResult> {
    try {
      const params: any = {
        address,
        key: this.apiKey,
      };

      if (options?.country) {
        params.components = `country:${options.country}`;
      }

      const response = await axios.get(`${this.baseUrl}/geocode/json`, { params });

      if (response.data.status !== 'OK') {
        return { success: false, error: response.data.status };
      }

      const result = response.data.results[0];
      const components = this.parseAddressComponents(result.address_components);

      return {
        success: true,
        address,
        formattedAddress: result.formatted_address,
        coordinates: {
          lat: result.geometry.location.lat,
          lng: result.geometry.location.lng,
        },
        components,
        placeId: result.place_id,
        confidence: this.getConfidenceFromType(result.geometry.location_type),
      };
    } catch (error) {
      this.logger.error(`Geocode error: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  async reverseGeocode(lat: number, lng: number): Promise<ReverseGeocodingResult> {
    try {
      const response = await axios.get(`${this.baseUrl}/geocode/json`, {
        params: {
          latlng: `${lat},${lng}`,
          key: this.apiKey,
        },
      });

      if (response.data.status !== 'OK') {
        return { success: false, error: response.data.status };
      }

      const result = response.data.results[0];
      const components = this.parseAddressComponents(result.address_components);

      return {
        success: true,
        address: result.formatted_address,
        formattedAddress: result.formatted_address,
        coordinates: { lat, lng },
        components,
        placeId: result.place_id,
      };
    } catch (error) {
      this.logger.error(`Reverse geocode error: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  async autocomplete(
    input: string,
    options?: {
      location?: Coordinates;
      radius?: number;
      types?: string[];
      country?: string;
    },
  ): Promise<AutocompleteResult> {
    try {
      const params: any = {
        input,
        key: this.apiKey,
      };

      if (options?.location) {
        params.location = `${options.location.lat},${options.location.lng}`;
      }
      if (options?.radius) {
        params.radius = options.radius;
      }
      if (options?.types?.length) {
        params.types = options.types.join('|');
      }
      if (options?.country) {
        params.components = `country:${options.country}`;
      }

      const response = await axios.get(`${this.baseUrl}/place/autocomplete/json`, { params });

      if (response.data.status !== 'OK' && response.data.status !== 'ZERO_RESULTS') {
        return { success: false, error: response.data.status };
      }

      return {
        success: true,
        predictions: (response.data.predictions || []).map((p: any) => ({
          placeId: p.place_id,
          description: p.description,
          mainText: p.structured_formatting?.main_text || p.description,
          secondaryText: p.structured_formatting?.secondary_text || '',
          types: p.types,
        })),
      };
    } catch (error) {
      this.logger.error(`Autocomplete error: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  async getPlaceDetails(placeId: string): Promise<PlaceDetailsResult> {
    try {
      const response = await axios.get(`${this.baseUrl}/place/details/json`, {
        params: {
          place_id: placeId,
          fields: 'name,formatted_address,geometry,type,formatted_phone_number,website,opening_hours,rating,price_level',
          key: this.apiKey,
        },
      });

      if (response.data.status !== 'OK') {
        return { success: false, error: response.data.status };
      }

      const result = response.data.result;

      return {
        success: true,
        place: {
          placeId,
          name: result.name,
          address: result.formatted_address,
          formattedAddress: result.formatted_address,
          coordinates: {
            lat: result.geometry.location.lat,
            lng: result.geometry.location.lng,
          },
          types: result.types,
          phoneNumber: result.formatted_phone_number,
          website: result.website,
          openingHours: result.opening_hours?.weekday_text,
          rating: result.rating,
          priceLevel: result.price_level,
        },
      };
    } catch (error) {
      this.logger.error(`Place details error: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  async getDirections(
    origin: Coordinates,
    destination: Coordinates,
    options?: {
      waypoints?: Coordinates[];
      mode?: 'driving' | 'walking' | 'bicycling' | 'transit';
      avoidTolls?: boolean;
      avoidHighways?: boolean;
      departureTime?: Date;
    },
  ): Promise<DirectionsResult> {
    try {
      const params: any = {
        origin: `${origin.lat},${origin.lng}`,
        destination: `${destination.lat},${destination.lng}`,
        mode: options?.mode || 'driving',
        key: this.apiKey,
      };

      if (options?.waypoints?.length) {
        params.waypoints = options.waypoints.map(w => `${w.lat},${w.lng}`).join('|');
      }

      const avoid: string[] = [];
      if (options?.avoidTolls) avoid.push('tolls');
      if (options?.avoidHighways) avoid.push('highways');
      if (avoid.length) params.avoid = avoid.join('|');

      if (options?.departureTime) {
        params.departure_time = Math.floor(options.departureTime.getTime() / 1000);
        params.traffic_model = 'best_guess';
      }

      const response = await axios.get(`${this.baseUrl}/directions/json`, { params });

      if (response.data.status !== 'OK') {
        return { success: false, error: response.data.status };
      }

      return {
        success: true,
        routes: response.data.routes.map((route: any) => {
          const leg = route.legs[0];
          return {
            distance: leg.distance.value,
            duration: leg.duration.value,
            durationInTraffic: leg.duration_in_traffic?.value,
            polyline: route.overview_polyline.points,
            steps: leg.steps.map((step: any) => ({
              distance: step.distance.value,
              duration: step.duration.value,
              instruction: step.html_instructions.replace(/<[^>]*>/g, ''),
              maneuver: step.maneuver,
              startLocation: {
                lat: step.start_location.lat,
                lng: step.start_location.lng,
              },
              endLocation: {
                lat: step.end_location.lat,
                lng: step.end_location.lng,
              },
            })),
          };
        }),
      };
    } catch (error) {
      this.logger.error(`Directions error: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  async getDistanceMatrix(
    origins: Coordinates[],
    destinations: Coordinates[],
    options?: {
      mode?: 'driving' | 'walking' | 'bicycling' | 'transit';
      departureTime?: Date;
    },
  ): Promise<DistanceMatrixResult> {
    try {
      const params: any = {
        origins: origins.map(o => `${o.lat},${o.lng}`).join('|'),
        destinations: destinations.map(d => `${d.lat},${d.lng}`).join('|'),
        mode: options?.mode || 'driving',
        key: this.apiKey,
      };

      if (options?.departureTime) {
        params.departure_time = Math.floor(options.departureTime.getTime() / 1000);
        params.traffic_model = 'best_guess';
      }

      const response = await axios.get(`${this.baseUrl}/distancematrix/json`, { params });

      if (response.data.status !== 'OK') {
        return { success: false, error: response.data.status };
      }

      return {
        success: true,
        rows: response.data.rows.map((row: any) => ({
          elements: row.elements.map((el: any) => ({
            status: el.status,
            distance: el.distance?.value,
            duration: el.duration?.value,
            durationInTraffic: el.duration_in_traffic?.value,
          })),
        })),
      };
    } catch (error) {
      this.logger.error(`Distance matrix error: ${error.message}`);
      return { success: false, error: error.message };
    }
  }

  private parseAddressComponents(components: any[]): GeocodingResult['components'] {
    const result: GeocodingResult['components'] = {};

    for (const component of components) {
      const types = component.types;

      if (types.includes('street_number')) {
        result.streetNumber = component.long_name;
      } else if (types.includes('route')) {
        result.street = component.long_name;
      } else if (types.includes('neighborhood') || types.includes('sublocality')) {
        result.neighborhood = component.long_name;
      } else if (types.includes('locality')) {
        result.city = component.long_name;
      } else if (types.includes('administrative_area_level_1')) {
        result.state = component.long_name;
      } else if (types.includes('country')) {
        result.country = component.long_name;
        result.countryCode = component.short_name;
      } else if (types.includes('postal_code')) {
        result.postalCode = component.long_name;
      }
    }

    return result;
  }

  private getConfidenceFromType(locationType: string): number {
    const confidenceMap: Record<string, number> = {
      ROOFTOP: 1.0,
      RANGE_INTERPOLATED: 0.8,
      GEOMETRIC_CENTER: 0.6,
      APPROXIMATE: 0.4,
    };
    return confidenceMap[locationType] || 0.5;
  }
}
