/**
 * Geocoding Provider Interface
 */

export interface Coordinates {
  lat: number;
  lng: number;
}

export interface GeocodingResult {
  success: boolean;
  address?: string;
  formattedAddress?: string;
  coordinates?: Coordinates;
  components?: {
    street?: string;
    streetNumber?: string;
    neighborhood?: string;
    city?: string;
    state?: string;
    country?: string;
    countryCode?: string;
    postalCode?: string;
  };
  placeId?: string;
  confidence?: number;
  error?: string;
}

export interface ReverseGeocodingResult extends GeocodingResult {}

export interface AutocompleteResult {
  success: boolean;
  predictions?: Array<{
    placeId: string;
    description: string;
    mainText: string;
    secondaryText: string;
    types?: string[];
  }>;
  error?: string;
}

export interface PlaceDetailsResult {
  success: boolean;
  place?: {
    placeId: string;
    name: string;
    address: string;
    formattedAddress: string;
    coordinates: Coordinates;
    types?: string[];
    phoneNumber?: string;
    website?: string;
    openingHours?: string[];
    rating?: number;
    priceLevel?: number;
  };
  error?: string;
}

export interface DirectionsResult {
  success: boolean;
  routes?: Array<{
    distance: number; // meters
    duration: number; // seconds
    durationInTraffic?: number;
    polyline: string;
    steps?: Array<{
      distance: number;
      duration: number;
      instruction: string;
      maneuver?: string;
      startLocation: Coordinates;
      endLocation: Coordinates;
    }>;
  }>;
  error?: string;
}

export interface DistanceMatrixResult {
  success: boolean;
  rows?: Array<{
    elements: Array<{
      status: string;
      distance?: number; // meters
      duration?: number; // seconds
      durationInTraffic?: number;
    }>;
  }>;
  error?: string;
}

export interface GeocodingProvider {
  readonly name: string;
  readonly displayName: string;

  /**
   * Geocode address to coordinates
   */
  geocode(address: string, options?: { country?: string }): Promise<GeocodingResult>;

  /**
   * Reverse geocode coordinates to address
   */
  reverseGeocode(lat: number, lng: number): Promise<ReverseGeocodingResult>;

  /**
   * Autocomplete address
   */
  autocomplete?(
    input: string,
    options?: {
      location?: Coordinates;
      radius?: number;
      types?: string[];
      country?: string;
    },
  ): Promise<AutocompleteResult>;

  /**
   * Get place details
   */
  getPlaceDetails?(placeId: string): Promise<PlaceDetailsResult>;

  /**
   * Get directions
   */
  getDirections?(
    origin: Coordinates,
    destination: Coordinates,
    options?: {
      waypoints?: Coordinates[];
      mode?: 'driving' | 'walking' | 'bicycling' | 'transit';
      avoidTolls?: boolean;
      avoidHighways?: boolean;
      departureTime?: Date;
    },
  ): Promise<DirectionsResult>;

  /**
   * Get distance matrix
   */
  getDistanceMatrix?(
    origins: Coordinates[],
    destinations: Coordinates[],
    options?: {
      mode?: 'driving' | 'walking' | 'bicycling' | 'transit';
      departureTime?: Date;
    },
  ): Promise<DistanceMatrixResult>;
}
