import {
  Controller,
  Get,
  Post,
  Delete,
  Body,
  Param,
  Query,
  UseGuards,
} from '@nestjs/common';
import { GeocodingService } from './geocoding.service';
import { DistanceService } from './distance.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { CurrentUser } from '../../common/decorators/merchant.decorator';
import { Public } from '../../common/decorators/public.decorator';

// ============================================================================
// DTOs
// ============================================================================

class GeocodeDto {
  address: string;
  country?: string;
  provider?: string;
}

class ReverseGeocodeDto {
  lat: number;
  lng: number;
  provider?: string;
}

class AutocompleteDto {
  input: string;
  lat?: number;
  lng?: number;
  radius?: number;
  types?: string;
  country?: string;
  provider?: string;
}

class DirectionsDto {
  originLat: number;
  originLng: number;
  destLat: number;
  destLng: number;
  waypoints?: Array<{ lat: number; lng: number }>;
  mode?: 'driving' | 'walking' | 'bicycling' | 'transit';
  avoidTolls?: boolean;
  avoidHighways?: boolean;
  provider?: string;
}

class DistanceDto {
  lat1: number;
  lng1: number;
  lat2: number;
  lng2: number;
  unit?: 'km' | 'm' | 'mi';
}

class SavePlaceDto {
  name: string;
  address: string;
  lat: number;
  lng: number;
  type?: string;
  placeId?: string;
}

// ============================================================================
// CONTROLLER
// ============================================================================

@Controller('geocoding')
export class GeocodingController {
  constructor(
    private readonly geocodingService: GeocodingService,
    private readonly distanceService: DistanceService,
  ) {}

  // ============================================================================
  // GEOCODING
  // ============================================================================

  /**
   * Geocode address to coordinates
   */
  @Post('geocode')
  @Public()
  async geocode(@Body() dto: GeocodeDto) {
    return this.geocodingService.geocode(dto.address, {
      country: dto.country,
      provider: dto.provider,
    });
  }

  /**
   * Reverse geocode coordinates to address
   */
  @Post('reverse')
  @Public()
  async reverseGeocode(@Body() dto: ReverseGeocodeDto) {
    return this.geocodingService.reverseGeocode(dto.lat, dto.lng, {
      provider: dto.provider,
    });
  }

  /**
   * Autocomplete address
   */
  @Get('autocomplete')
  @Public()
  async autocomplete(@Query() dto: AutocompleteDto) {
    return this.geocodingService.autocomplete(dto.input, {
      provider: dto.provider,
      location: dto.lat && dto.lng ? { lat: dto.lat, lng: dto.lng } : undefined,
      radius: dto.radius,
      types: dto.types ? dto.types.split(',') : undefined,
      country: dto.country,
    });
  }

  /**
   * Get place details
   */
  @Get('places/:placeId')
  @Public()
  async getPlaceDetails(
    @Param('placeId') placeId: string,
    @Query('provider') provider?: string,
  ) {
    return this.geocodingService.getPlaceDetails(placeId, provider);
  }

  // ============================================================================
  // DIRECTIONS & DISTANCE
  // ============================================================================

  /**
   * Get directions
   */
  @Post('directions')
  @Public()
  async getDirections(@Body() dto: DirectionsDto) {
    return this.geocodingService.getDirections(
      { lat: dto.originLat, lng: dto.originLng },
      { lat: dto.destLat, lng: dto.destLng },
      {
        provider: dto.provider,
        waypoints: dto.waypoints,
        mode: dto.mode,
        avoidTolls: dto.avoidTolls,
        avoidHighways: dto.avoidHighways,
      },
    );
  }

  /**
   * Get ETA
   */
  @Post('eta')
  @Public()
  async getETA(@Body() dto: DirectionsDto) {
    return this.geocodingService.getETA(
      { lat: dto.originLat, lng: dto.originLng },
      { lat: dto.destLat, lng: dto.destLng },
      { provider: dto.provider, mode: dto.mode },
    );
  }

  /**
   * Calculate distance (Haversine)
   */
  @Post('distance')
  @Public()
  async calculateDistance(@Body() dto: DistanceDto) {
    const distance = this.distanceService.calculateDistance(
      { lat: dto.lat1, lng: dto.lng1 },
      { lat: dto.lat2, lng: dto.lng2 },
      dto.unit,
    );

    const bearing = this.distanceService.calculateBearing(
      { lat: dto.lat1, lng: dto.lng1 },
      { lat: dto.lat2, lng: dto.lng2 },
    );

    const estimatedDuration = this.distanceService.estimateDrivingTime(
      dto.unit === 'm' ? distance / 1000 : dto.unit === 'mi' ? distance * 1.60934 : distance,
    );

    return {
      distance,
      unit: dto.unit || 'km',
      bearing: Math.round(bearing),
      direction: this.distanceService.getBearingDirection(bearing),
      estimatedDuration,
      formattedDistance: this.distanceService.formatDistance(
        dto.unit === 'm' ? distance / 1000 : dto.unit === 'mi' ? distance * 1.60934 : distance,
      ),
      formattedDuration: this.distanceService.formatDuration(estimatedDuration),
    };
  }

  // ============================================================================
  // FAVORITE PLACES (requires auth)
  // ============================================================================

  /**
   * Save favorite place
   */
  @Post('favorites')
  @UseGuards(JwtAuthGuard)
  async saveFavoritePlace(
    @CurrentUser() userId: number,
    @Body() dto: SavePlaceDto,
  ) {
    return this.geocodingService.saveFavoritePlace(userId, dto);
  }

  /**
   * Get favorite places
   */
  @Get('favorites')
  @UseGuards(JwtAuthGuard)
  async getFavoritePlaces(@CurrentUser() userId: number) {
    return this.geocodingService.getFavoritePlaces(userId);
  }

  /**
   * Delete favorite place
   */
  @Delete('favorites/:id')
  @UseGuards(JwtAuthGuard)
  async deleteFavoritePlace(
    @CurrentUser() userId: number,
    @Param('id') id: string,
  ) {
    const deleted = await this.geocodingService.deleteFavoritePlace(userId, parseInt(id));
    return { success: deleted };
  }

  /**
   * Get recent addresses
   */
  @Get('recent')
  @UseGuards(JwtAuthGuard)
  async getRecentAddresses(
    @CurrentUser() userId: number,
    @Query('limit') limit?: string,
  ) {
    return this.geocodingService.getRecentAddresses(userId, limit ? parseInt(limit) : 10);
  }

  // ============================================================================
  // UTILITIES
  // ============================================================================

  /**
   * Get nearby places
   */
  @Get('nearby')
  @Public()
  async getNearbyPlaces(
    @Query('lat') lat: string,
    @Query('lng') lng: string,
    @Query('type') type: string,
    @Query('radius') radius?: string,
    @Query('provider') provider?: string,
  ) {
    return this.geocodingService.getNearbyPlaces(
      { lat: parseFloat(lat), lng: parseFloat(lng) },
      type,
      radius ? parseInt(radius) : 1000,
      provider,
    );
  }

  /**
   * Get available providers
   */
  @Get('providers')
  @Public()
  async getProviders() {
    return this.geocodingService.getAvailableProviders();
  }

  /**
   * Decode polyline
   */
  @Post('decode-polyline')
  @Public()
  async decodePolyline(@Body() body: { polyline: string }) {
    const coordinates = this.distanceService.decodePolyline(body.polyline);
    return { coordinates };
  }

  /**
   * Encode polyline
   */
  @Post('encode-polyline')
  @Public()
  async encodePolyline(@Body() body: { coordinates: Array<{ lat: number; lng: number }> }) {
    const polyline = this.distanceService.encodePolyline(body.coordinates);
    return { polyline };
  }
}
