import { Injectable, Logger } from '@nestjs/common';
import { Coordinates } from './providers/geocoding.interface';

@Injectable()
export class DistanceService {
  private readonly logger = new Logger(DistanceService.name);
  private readonly EARTH_RADIUS_KM = 6371;

  /**
   * Calculate distance using Haversine formula
   */
  calculateDistance(
    point1: Coordinates,
    point2: Coordinates,
    unit: 'km' | 'm' | 'mi' = 'km',
  ): number {
    const lat1Rad = this.toRadians(point1.lat);
    const lat2Rad = this.toRadians(point2.lat);
    const deltaLat = this.toRadians(point2.lat - point1.lat);
    const deltaLng = this.toRadians(point2.lng - point1.lng);

    const a =
      Math.sin(deltaLat / 2) * Math.sin(deltaLat / 2) +
      Math.cos(lat1Rad) * Math.cos(lat2Rad) *
      Math.sin(deltaLng / 2) * Math.sin(deltaLng / 2);

    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
    const distanceKm = this.EARTH_RADIUS_KM * c;

    switch (unit) {
      case 'm':
        return Math.round(distanceKm * 1000);
      case 'mi':
        return Math.round(distanceKm * 0.621371 * 100) / 100;
      default:
        return Math.round(distanceKm * 100) / 100;
    }
  }

  /**
   * Calculate bearing between two points
   */
  calculateBearing(from: Coordinates, to: Coordinates): number {
    const lat1 = this.toRadians(from.lat);
    const lat2 = this.toRadians(to.lat);
    const deltaLng = this.toRadians(to.lng - from.lng);

    const y = Math.sin(deltaLng) * Math.cos(lat2);
    const x =
      Math.cos(lat1) * Math.sin(lat2) -
      Math.sin(lat1) * Math.cos(lat2) * Math.cos(deltaLng);

    const bearing = this.toDegrees(Math.atan2(y, x));
    return (bearing + 360) % 360;
  }

  /**
   * Get compass direction from bearing
   */
  getBearingDirection(bearing: number): string {
    const directions = ['N', 'NE', 'E', 'SE', 'S', 'SW', 'W', 'NW'];
    const index = Math.round(bearing / 45) % 8;
    return directions[index];
  }

  /**
   * Calculate destination point given start point, bearing, and distance
   */
  calculateDestination(
    start: Coordinates,
    bearing: number,
    distanceKm: number,
  ): Coordinates {
    const bearingRad = this.toRadians(bearing);
    const lat1 = this.toRadians(start.lat);
    const lng1 = this.toRadians(start.lng);
    const angularDistance = distanceKm / this.EARTH_RADIUS_KM;

    const lat2 = Math.asin(
      Math.sin(lat1) * Math.cos(angularDistance) +
      Math.cos(lat1) * Math.sin(angularDistance) * Math.cos(bearingRad)
    );

    const lng2 = lng1 + Math.atan2(
      Math.sin(bearingRad) * Math.sin(angularDistance) * Math.cos(lat1),
      Math.cos(angularDistance) - Math.sin(lat1) * Math.sin(lat2)
    );

    return {
      lat: this.toDegrees(lat2),
      lng: this.toDegrees(lng2),
    };
  }

  /**
   * Check if a point is within a radius of another point
   */
  isWithinRadius(
    center: Coordinates,
    point: Coordinates,
    radiusKm: number,
  ): boolean {
    const distance = this.calculateDistance(center, point, 'km');
    return distance <= radiusKm;
  }

  /**
   * Find points within radius
   */
  filterWithinRadius<T extends { lat: number; lng: number }>(
    center: Coordinates,
    points: T[],
    radiusKm: number,
  ): T[] {
    return points.filter(point =>
      this.isWithinRadius(center, { lat: point.lat, lng: point.lng }, radiusKm)
    );
  }

  /**
   * Sort points by distance from a reference point
   */
  sortByDistance<T extends { lat: number; lng: number }>(
    reference: Coordinates,
    points: T[],
    order: 'asc' | 'desc' = 'asc',
  ): Array<T & { distance: number }> {
    const withDistance = points.map(point => ({
      ...point,
      distance: this.calculateDistance(reference, { lat: point.lat, lng: point.lng }, 'km'),
    }));

    return withDistance.sort((a, b) =>
      order === 'asc' ? a.distance - b.distance : b.distance - a.distance
    );
  }

  /**
   * Get nearest point
   */
  findNearest<T extends { lat: number; lng: number }>(
    reference: Coordinates,
    points: T[],
  ): (T & { distance: number }) | null {
    if (!points.length) return null;
    const sorted = this.sortByDistance(reference, points, 'asc');
    return sorted[0];
  }

  /**
   * Calculate bounding box for a center point and radius
   */
  getBoundingBox(
    center: Coordinates,
    radiusKm: number,
  ): { minLat: number; maxLat: number; minLng: number; maxLng: number } {
    const latDelta = radiusKm / 111; // approx km per degree latitude
    const lngDelta = radiusKm / (111 * Math.cos(this.toRadians(center.lat)));

    return {
      minLat: center.lat - latDelta,
      maxLat: center.lat + latDelta,
      minLng: center.lng - lngDelta,
      maxLng: center.lng + lngDelta,
    };
  }

  /**
   * Check if point is inside polygon
   */
  isPointInPolygon(point: Coordinates, polygon: Coordinates[]): boolean {
    let inside = false;
    const n = polygon.length;

    for (let i = 0, j = n - 1; i < n; j = i++) {
      const xi = polygon[i].lng;
      const yi = polygon[i].lat;
      const xj = polygon[j].lng;
      const yj = polygon[j].lat;

      const intersect =
        yi > point.lat !== yj > point.lat &&
        point.lng < ((xj - xi) * (point.lat - yi)) / (yj - yi) + xi;

      if (intersect) inside = !inside;
    }

    return inside;
  }

  /**
   * Calculate center of multiple points
   */
  calculateCenter(points: Coordinates[]): Coordinates {
    if (!points.length) {
      throw new Error('Cannot calculate center of empty array');
    }

    let totalLat = 0;
    let totalLng = 0;

    for (const point of points) {
      totalLat += point.lat;
      totalLng += point.lng;
    }

    return {
      lat: totalLat / points.length,
      lng: totalLng / points.length,
    };
  }

  /**
   * Estimate driving time (rough estimate without traffic)
   * @param distanceKm Distance in kilometers
   * @param speedKmh Average speed (default 30 km/h for urban)
   * @returns Duration in seconds
   */
  estimateDrivingTime(distanceKm: number, speedKmh: number = 30): number {
    return Math.round((distanceKm / speedKmh) * 3600);
  }

  /**
   * Format distance for display
   */
  formatDistance(distanceKm: number): string {
    if (distanceKm < 1) {
      return `${Math.round(distanceKm * 1000)} m`;
    }
    return `${distanceKm.toFixed(1)} km`;
  }

  /**
   * Format duration for display
   */
  formatDuration(seconds: number): string {
    if (seconds < 60) {
      return `${seconds} sec`;
    }
    if (seconds < 3600) {
      return `${Math.round(seconds / 60)} min`;
    }
    const hours = Math.floor(seconds / 3600);
    const minutes = Math.round((seconds % 3600) / 60);
    return `${hours}h ${minutes}min`;
  }

  /**
   * Decode Google polyline
   */
  decodePolyline(encoded: string): Coordinates[] {
    const coordinates: Coordinates[] = [];
    let index = 0;
    let lat = 0;
    let lng = 0;

    while (index < encoded.length) {
      let shift = 0;
      let result = 0;
      let byte: number;

      do {
        byte = encoded.charCodeAt(index++) - 63;
        result |= (byte & 0x1f) << shift;
        shift += 5;
      } while (byte >= 0x20);

      const dlat = result & 1 ? ~(result >> 1) : result >> 1;
      lat += dlat;

      shift = 0;
      result = 0;

      do {
        byte = encoded.charCodeAt(index++) - 63;
        result |= (byte & 0x1f) << shift;
        shift += 5;
      } while (byte >= 0x20);

      const dlng = result & 1 ? ~(result >> 1) : result >> 1;
      lng += dlng;

      coordinates.push({
        lat: lat / 1e5,
        lng: lng / 1e5,
      });
    }

    return coordinates;
  }

  /**
   * Encode coordinates to Google polyline
   */
  encodePolyline(coordinates: Coordinates[]): string {
    let encoded = '';
    let prevLat = 0;
    let prevLng = 0;

    for (const coord of coordinates) {
      const lat = Math.round(coord.lat * 1e5);
      const lng = Math.round(coord.lng * 1e5);

      encoded += this.encodeNumber(lat - prevLat);
      encoded += this.encodeNumber(lng - prevLng);

      prevLat = lat;
      prevLng = lng;
    }

    return encoded;
  }

  private encodeNumber(num: number): string {
    let encoded = '';
    let value = num < 0 ? ~(num << 1) : num << 1;

    while (value >= 0x20) {
      encoded += String.fromCharCode((0x20 | (value & 0x1f)) + 63);
      value >>= 5;
    }

    encoded += String.fromCharCode(value + 63);
    return encoded;
  }

  private toRadians(degrees: number): number {
    return degrees * (Math.PI / 180);
  }

  private toDegrees(radians: number): number {
    return radians * (180 / Math.PI);
  }
}
