import { Injectable, Logger, BadRequestException, NotFoundException } from '@nestjs/common';
import { PrismaService } from '../../common/prisma/prisma.service';
import { NotificationService } from '../notification/notification.service';

export enum FranchiseStatus {
  PENDING = 'pending',
  ACTIVE = 'active',
  SUSPENDED = 'suspended',
  TERMINATED = 'terminated',
}

export enum PaymentCycle {
  WEEKLY = 'weekly',
  BIWEEKLY = 'biweekly',
  MONTHLY = 'monthly',
}

@Injectable()
export class FranchiseService {
  private readonly logger = new Logger(FranchiseService.name);

  constructor(
    private prisma: PrismaService,
    private notificationService: NotificationService,
  ) {}

  // ============================================================================
  // FRANCHISE MANAGEMENT
  // ============================================================================

  /**
   * Create franchise (master merchant creates a sub-franchise)
   */
  async createFranchise(
    masterMerchantId: number,
    data: {
      name: string;
      ownerName: string;
      ownerEmail: string;
      ownerPhone: string;
      address: string;
      city: string;
      country: string;
      latitude?: number;
      longitude?: number;
      commissionRate: number;
      paymentCycle: PaymentCycle;
      contractStartDate: Date;
      contractEndDate?: Date;
      territoryRadius?: number;
    },
  ): Promise<any> {
    // Check if master merchant has franchise rights
    const masterMerchant = await this.prisma.merchant.findUnique({
      where: { id: masterMerchantId },
    });

    if (!masterMerchant || masterMerchant.is_franchise_master !== 1) {
      throw new BadRequestException('Vous n\'avez pas les droits de franchise');
    }

    // Create franchise merchant account
    const franchiseMerchant = await this.prisma.merchant.create({
      data: {
        name: data.name,
        email: data.ownerEmail,
        mobile: data.ownerPhone,
        address: data.address,
        city: data.city,
        country: data.country,
        latitude: data.latitude,
        longitude: data.longitude,
        is_active: 1,
        is_franchise: 1,
        parent_merchant_id: masterMerchantId,
        created_at: new Date(),
      },
    });

    // Create franchise record
    const franchise = await this.prisma.franchise.create({
      data: {
        master_merchant_id: masterMerchantId,
        franchise_merchant_id: franchiseMerchant.id,
        owner_name: data.ownerName,
        owner_email: data.ownerEmail,
        owner_phone: data.ownerPhone,
        commission_rate: data.commissionRate,
        payment_cycle: data.paymentCycle,
        contract_start_date: data.contractStartDate,
        contract_end_date: data.contractEndDate,
        territory_radius: data.territoryRadius,
        status: FranchiseStatus.ACTIVE,
        created_at: new Date(),
      },
    });

    this.logger.log(`Franchise created: ${data.name} (#${franchise.id})`);

    return {
      ...franchise,
      merchant: franchiseMerchant,
    };
  }

  /**
   * Get franchises for master merchant
   */
  async getFranchises(
    masterMerchantId: number,
    options?: { status?: FranchiseStatus; page?: number; limit?: number },
  ): Promise<{ data: any[]; total: number }> {
    const page = options?.page || 1;
    const limit = options?.limit || 20;
    const skip = (page - 1) * limit;

    const where: any = { master_merchant_id: masterMerchantId };
    if (options?.status) {
      where.status = options.status;
    }

    const [franchises, total] = await Promise.all([
      this.prisma.franchise.findMany({
        where,
        include: {
          franchiseMerchant: {
            select: {
              id: true,
              name: true,
              city: true,
              country: true,
              is_active: true,
            },
          },
        },
        skip,
        take: limit,
        orderBy: { created_at: 'desc' },
      }),
      this.prisma.franchise.count({ where }),
    ]);

    return { data: franchises, total };
  }

  /**
   * Get franchise details
   */
  async getFranchiseById(franchiseId: number, masterMerchantId: number): Promise<any> {
    const franchise = await this.prisma.franchise.findFirst({
      where: { id: franchiseId, master_merchant_id: masterMerchantId },
      include: {
        franchiseMerchant: true,
      },
    });

    if (!franchise) {
      throw new NotFoundException('Franchise non trouvee');
    }

    // Get statistics
    const stats = await this.getFranchiseStats(franchise.franchise_merchant_id);

    return {
      ...franchise,
      stats,
    };
  }

  /**
   * Update franchise
   */
  async updateFranchise(
    franchiseId: number,
    masterMerchantId: number,
    data: {
      commissionRate?: number;
      paymentCycle?: PaymentCycle;
      contractEndDate?: Date;
      territoryRadius?: number;
    },
  ): Promise<any> {
    const franchise = await this.prisma.franchise.findFirst({
      where: { id: franchiseId, master_merchant_id: masterMerchantId },
    });

    if (!franchise) {
      throw new NotFoundException('Franchise non trouvee');
    }

    return this.prisma.franchise.update({
      where: { id: franchiseId },
      data: {
        ...(data.commissionRate !== undefined && { commission_rate: data.commissionRate }),
        ...(data.paymentCycle && { payment_cycle: data.paymentCycle }),
        ...(data.contractEndDate && { contract_end_date: data.contractEndDate }),
        ...(data.territoryRadius !== undefined && { territory_radius: data.territoryRadius }),
        updated_at: new Date(),
      },
    });
  }

  /**
   * Suspend franchise
   */
  async suspendFranchise(
    franchiseId: number,
    masterMerchantId: number,
    reason?: string,
  ): Promise<any> {
    const franchise = await this.prisma.franchise.findFirst({
      where: { id: franchiseId, master_merchant_id: masterMerchantId },
    });

    if (!franchise) {
      throw new NotFoundException('Franchise non trouvee');
    }

    // Suspend franchise
    const updated = await this.prisma.franchise.update({
      where: { id: franchiseId },
      data: {
        status: FranchiseStatus.SUSPENDED,
        suspension_reason: reason,
        suspended_at: new Date(),
      },
    });

    // Deactivate franchise merchant
    await this.prisma.merchant.update({
      where: { id: franchise.franchise_merchant_id },
      data: { is_active: 0 },
    });

    // Notify franchise owner
    await this.notificationService.sendEmail(
      franchise.owner_email,
      'Franchise suspendue',
      `Votre franchise a ete suspendue. Raison: ${reason || 'Non specifiee'}`,
    );

    return updated;
  }

  /**
   * Reactivate franchise
   */
  async reactivateFranchise(franchiseId: number, masterMerchantId: number): Promise<any> {
    const franchise = await this.prisma.franchise.findFirst({
      where: { id: franchiseId, master_merchant_id: masterMerchantId },
    });

    if (!franchise) {
      throw new NotFoundException('Franchise non trouvee');
    }

    const updated = await this.prisma.franchise.update({
      where: { id: franchiseId },
      data: {
        status: FranchiseStatus.ACTIVE,
        suspension_reason: null,
        suspended_at: null,
        reactivated_at: new Date(),
      },
    });

    // Reactivate franchise merchant
    await this.prisma.merchant.update({
      where: { id: franchise.franchise_merchant_id },
      data: { is_active: 1 },
    });

    return updated;
  }

  /**
   * Terminate franchise
   */
  async terminateFranchise(
    franchiseId: number,
    masterMerchantId: number,
    reason?: string,
  ): Promise<any> {
    const franchise = await this.prisma.franchise.findFirst({
      where: { id: franchiseId, master_merchant_id: masterMerchantId },
    });

    if (!franchise) {
      throw new NotFoundException('Franchise non trouvee');
    }

    const updated = await this.prisma.franchise.update({
      where: { id: franchiseId },
      data: {
        status: FranchiseStatus.TERMINATED,
        termination_reason: reason,
        terminated_at: new Date(),
      },
    });

    // Deactivate franchise merchant
    await this.prisma.merchant.update({
      where: { id: franchise.franchise_merchant_id },
      data: { is_active: 0 },
    });

    return updated;
  }

  // ============================================================================
  // COMMISSION & PAYMENTS
  // ============================================================================

  /**
   * Calculate commission for a period
   */
  async calculateCommission(
    franchiseId: number,
    startDate: Date,
    endDate: Date,
  ): Promise<{
    totalRevenue: number;
    commissionAmount: number;
    commissionRate: number;
    bookingsCount: number;
  }> {
    const franchise = await this.prisma.franchise.findUnique({
      where: { id: franchiseId },
    });

    if (!franchise) {
      throw new NotFoundException('Franchise non trouvee');
    }

    // Get bookings for the period
    const bookings = await this.prisma.booking.findMany({
      where: {
        merchant_id: franchise.franchise_merchant_id,
        status: 'completed',
        completed_at: {
          gte: startDate,
          lte: endDate,
        },
      },
      select: { total_amount: true },
    });

    const totalRevenue = bookings.reduce(
      (sum, b) => sum + (Number(b.total_amount) || 0),
      0,
    );

    const commissionRate = Number(franchise.commission_rate) || 10;
    const commissionAmount = Math.round((totalRevenue * commissionRate) / 100);

    return {
      totalRevenue,
      commissionAmount,
      commissionRate,
      bookingsCount: bookings.length,
    };
  }

  /**
   * Create commission invoice
   */
  async createCommissionInvoice(
    franchiseId: number,
    masterMerchantId: number,
    periodStart: Date,
    periodEnd: Date,
  ): Promise<any> {
    const franchise = await this.prisma.franchise.findFirst({
      where: { id: franchiseId, master_merchant_id: masterMerchantId },
    });

    if (!franchise) {
      throw new NotFoundException('Franchise non trouvee');
    }

    const commission = await this.calculateCommission(franchiseId, periodStart, periodEnd);

    const invoice = await this.prisma.franchiseInvoice.create({
      data: {
        franchise_id: franchiseId,
        master_merchant_id: masterMerchantId,
        franchise_merchant_id: franchise.franchise_merchant_id,
        invoice_number: this.generateInvoiceNumber(),
        period_start: periodStart,
        period_end: periodEnd,
        total_revenue: commission.totalRevenue,
        commission_rate: commission.commissionRate,
        commission_amount: commission.commissionAmount,
        bookings_count: commission.bookingsCount,
        status: 'pending',
        due_date: new Date(Date.now() + 15 * 24 * 60 * 60 * 1000), // 15 days
        created_at: new Date(),
      },
    });

    // Notify franchise
    await this.notificationService.sendEmail(
      franchise.owner_email,
      'Nouvelle facture de commission',
      `Une facture de ${commission.commissionAmount} XOF est disponible pour la periode du ${periodStart.toLocaleDateString()} au ${periodEnd.toLocaleDateString()}`,
    );

    return invoice;
  }

  /**
   * Get commission invoices
   */
  async getCommissionInvoices(
    franchiseId: number,
    masterMerchantId: number,
    options?: { status?: string; page?: number; limit?: number },
  ): Promise<{ data: any[]; total: number }> {
    const page = options?.page || 1;
    const limit = options?.limit || 20;
    const skip = (page - 1) * limit;

    const where: any = {
      franchise_id: franchiseId,
      master_merchant_id: masterMerchantId,
    };

    if (options?.status) {
      where.status = options.status;
    }

    const [invoices, total] = await Promise.all([
      this.prisma.franchiseInvoice.findMany({
        where,
        skip,
        take: limit,
        orderBy: { created_at: 'desc' },
      }),
      this.prisma.franchiseInvoice.count({ where }),
    ]);

    return { data: invoices, total };
  }

  /**
   * Mark invoice as paid
   */
  async markInvoicePaid(
    invoiceId: number,
    masterMerchantId: number,
    paymentReference?: string,
  ): Promise<any> {
    const invoice = await this.prisma.franchiseInvoice.findFirst({
      where: { id: invoiceId, master_merchant_id: masterMerchantId },
    });

    if (!invoice) {
      throw new NotFoundException('Facture non trouvee');
    }

    return this.prisma.franchiseInvoice.update({
      where: { id: invoiceId },
      data: {
        status: 'paid',
        payment_reference: paymentReference,
        paid_at: new Date(),
      },
    });
  }

  // ============================================================================
  // STATISTICS
  // ============================================================================

  /**
   * Get franchise statistics
   */
  async getFranchiseStats(franchiseMerchantId: number): Promise<any> {
    const now = new Date();
    const startOfMonth = new Date(now.getFullYear(), now.getMonth(), 1);
    const startOfYear = new Date(now.getFullYear(), 0, 1);

    const [
      totalBookings,
      monthlyBookings,
      yearlyBookings,
      totalDrivers,
      activeDrivers,
      totalUsers,
    ] = await Promise.all([
      this.prisma.booking.count({
        where: { merchant_id: franchiseMerchantId, status: 'completed' },
      }),
      this.prisma.booking.count({
        where: {
          merchant_id: franchiseMerchantId,
          status: 'completed',
          created_at: { gte: startOfMonth },
        },
      }),
      this.prisma.booking.count({
        where: {
          merchant_id: franchiseMerchantId,
          status: 'completed',
          created_at: { gte: startOfYear },
        },
      }),
      this.prisma.driver.count({
        where: { merchant_id: franchiseMerchantId },
      }),
      this.prisma.driver.count({
        where: { merchant_id: franchiseMerchantId, is_active: 1 },
      }),
      this.prisma.user.count({
        where: { merchant_id: franchiseMerchantId },
      }),
    ]);

    // Revenue
    const monthlyRevenue = await this.prisma.booking.aggregate({
      where: {
        merchant_id: franchiseMerchantId,
        status: 'completed',
        created_at: { gte: startOfMonth },
      },
      _sum: { total_amount: true },
    });

    const yearlyRevenue = await this.prisma.booking.aggregate({
      where: {
        merchant_id: franchiseMerchantId,
        status: 'completed',
        created_at: { gte: startOfYear },
      },
      _sum: { total_amount: true },
    });

    return {
      bookings: {
        total: totalBookings,
        monthly: monthlyBookings,
        yearly: yearlyBookings,
      },
      drivers: {
        total: totalDrivers,
        active: activeDrivers,
      },
      users: {
        total: totalUsers,
      },
      revenue: {
        monthly: Number(monthlyRevenue._sum.total_amount) || 0,
        yearly: Number(yearlyRevenue._sum.total_amount) || 0,
      },
    };
  }

  /**
   * Get master franchise dashboard
   */
  async getMasterDashboard(masterMerchantId: number): Promise<any> {
    const franchises = await this.prisma.franchise.findMany({
      where: { master_merchant_id: masterMerchantId },
      select: {
        id: true,
        status: true,
        franchise_merchant_id: true,
      },
    });

    const activeFranchises = franchises.filter(f => f.status === FranchiseStatus.ACTIVE);

    // Get aggregated stats
    const startOfMonth = new Date();
    startOfMonth.setDate(1);
    startOfMonth.setHours(0, 0, 0, 0);

    let totalMonthlyRevenue = 0;
    let totalMonthlyBookings = 0;
    let totalPendingCommissions = 0;

    for (const franchise of activeFranchises) {
      const monthlyRevenue = await this.prisma.booking.aggregate({
        where: {
          merchant_id: franchise.franchise_merchant_id,
          status: 'completed',
          created_at: { gte: startOfMonth },
        },
        _sum: { total_amount: true },
      });

      const monthlyBookings = await this.prisma.booking.count({
        where: {
          merchant_id: franchise.franchise_merchant_id,
          status: 'completed',
          created_at: { gte: startOfMonth },
        },
      });

      totalMonthlyRevenue += Number(monthlyRevenue._sum.total_amount) || 0;
      totalMonthlyBookings += monthlyBookings;
    }

    // Get pending invoices
    const pendingInvoices = await this.prisma.franchiseInvoice.aggregate({
      where: {
        master_merchant_id: masterMerchantId,
        status: 'pending',
      },
      _sum: { commission_amount: true },
      _count: true,
    });

    totalPendingCommissions = Number(pendingInvoices._sum.commission_amount) || 0;

    return {
      franchises: {
        total: franchises.length,
        active: activeFranchises.length,
        suspended: franchises.filter(f => f.status === FranchiseStatus.SUSPENDED).length,
        terminated: franchises.filter(f => f.status === FranchiseStatus.TERMINATED).length,
      },
      monthlyStats: {
        revenue: totalMonthlyRevenue,
        bookings: totalMonthlyBookings,
      },
      commissions: {
        pending: totalPendingCommissions,
        pendingInvoices: pendingInvoices._count,
      },
    };
  }

  // ============================================================================
  // FRANCHISE TERRITORIES
  // ============================================================================

  /**
   * Check if location is within franchise territory
   */
  async isWithinFranchiseTerritory(
    franchiseMerchantId: number,
    latitude: number,
    longitude: number,
  ): Promise<boolean> {
    const franchise = await this.prisma.franchise.findFirst({
      where: { franchise_merchant_id: franchiseMerchantId },
      include: { franchiseMerchant: true },
    });

    if (!franchise || !franchise.territory_radius) {
      return true; // No territory restriction
    }

    const merchantLat = Number(franchise.franchiseMerchant.latitude);
    const merchantLng = Number(franchise.franchiseMerchant.longitude);

    if (!merchantLat || !merchantLng) {
      return true;
    }

    const distance = this.calculateDistance(
      merchantLat,
      merchantLng,
      latitude,
      longitude,
    );

    return distance <= franchise.territory_radius;
  }

  /**
   * Find franchise for location
   */
  async findFranchiseForLocation(
    masterMerchantId: number,
    latitude: number,
    longitude: number,
  ): Promise<any | null> {
    const franchises = await this.prisma.franchise.findMany({
      where: {
        master_merchant_id: masterMerchantId,
        status: FranchiseStatus.ACTIVE,
      },
      include: { franchiseMerchant: true },
    });

    for (const franchise of franchises) {
      const merchantLat = Number(franchise.franchiseMerchant.latitude);
      const merchantLng = Number(franchise.franchiseMerchant.longitude);
      const radius = Number(franchise.territory_radius) || 50;

      if (merchantLat && merchantLng) {
        const distance = this.calculateDistance(
          merchantLat,
          merchantLng,
          latitude,
          longitude,
        );

        if (distance <= radius) {
          return franchise;
        }
      }
    }

    return null;
  }

  // ============================================================================
  // HELPERS
  // ============================================================================

  private generateInvoiceNumber(): string {
    const timestamp = Date.now().toString(36).toUpperCase();
    return `FI${timestamp}`;
  }

  private calculateDistance(
    lat1: number,
    lng1: number,
    lat2: number,
    lng2: number,
  ): number {
    const R = 6371;
    const dLat = ((lat2 - lat1) * Math.PI) / 180;
    const dLng = ((lng2 - lng1) * Math.PI) / 180;
    const a =
      Math.sin(dLat / 2) * Math.sin(dLat / 2) +
      Math.cos((lat1 * Math.PI) / 180) *
        Math.cos((lat2 * Math.PI) / 180) *
        Math.sin(dLng / 2) *
        Math.sin(dLng / 2);
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
    return R * c;
  }
}
