import {
  Controller,
  Get,
  Post,
  Put,
  Body,
  Param,
  Query,
  UseGuards,
  ParseIntPipe,
} from '@nestjs/common';
import { ApiTags, ApiOperation, ApiBearerAuth } from '@nestjs/swagger';
import { FranchiseService, FranchiseStatus, PaymentCycle } from './franchise.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantGuard } from '../../common/guards/merchant.guard';
import { MerchantId } from '../../common/decorators/merchant.decorator';

@ApiTags('Franchise')
@Controller('api/franchise')
@UseGuards(JwtAuthGuard, MerchantGuard)
@ApiBearerAuth()
export class FranchiseController {
  constructor(private franchiseService: FranchiseService) {}

  // ============================================================================
  // DASHBOARD
  // ============================================================================

  @Get('dashboard')
  @ApiOperation({ summary: 'Get master franchise dashboard' })
  async getMasterDashboard(@MerchantId() merchantId: number) {
    return this.franchiseService.getMasterDashboard(merchantId);
  }

  // ============================================================================
  // FRANCHISE MANAGEMENT
  // ============================================================================

  @Get()
  @ApiOperation({ summary: 'Get all franchises' })
  async getFranchises(
    @MerchantId() merchantId: number,
    @Query('status') status?: FranchiseStatus,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
  ) {
    return this.franchiseService.getFranchises(merchantId, {
      status,
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 20,
    });
  }

  @Get(':id')
  @ApiOperation({ summary: 'Get franchise details' })
  async getFranchiseById(
    @Param('id', ParseIntPipe) franchiseId: number,
    @MerchantId() merchantId: number,
  ) {
    return this.franchiseService.getFranchiseById(franchiseId, merchantId);
  }

  @Post()
  @ApiOperation({ summary: 'Create franchise' })
  async createFranchise(
    @MerchantId() merchantId: number,
    @Body()
    body: {
      name: string;
      ownerName: string;
      ownerEmail: string;
      ownerPhone: string;
      address: string;
      city: string;
      country: string;
      latitude?: number;
      longitude?: number;
      commissionRate: number;
      paymentCycle: PaymentCycle;
      contractStartDate: string;
      contractEndDate?: string;
      territoryRadius?: number;
    },
  ) {
    return this.franchiseService.createFranchise(merchantId, {
      ...body,
      contractStartDate: new Date(body.contractStartDate),
      contractEndDate: body.contractEndDate ? new Date(body.contractEndDate) : undefined,
    });
  }

  @Put(':id')
  @ApiOperation({ summary: 'Update franchise' })
  async updateFranchise(
    @Param('id', ParseIntPipe) franchiseId: number,
    @MerchantId() merchantId: number,
    @Body()
    body: {
      commissionRate?: number;
      paymentCycle?: PaymentCycle;
      contractEndDate?: string;
      territoryRadius?: number;
    },
  ) {
    return this.franchiseService.updateFranchise(franchiseId, merchantId, {
      ...body,
      contractEndDate: body.contractEndDate ? new Date(body.contractEndDate) : undefined,
    });
  }

  @Post(':id/suspend')
  @ApiOperation({ summary: 'Suspend franchise' })
  async suspendFranchise(
    @Param('id', ParseIntPipe) franchiseId: number,
    @MerchantId() merchantId: number,
    @Body() body?: { reason?: string },
  ) {
    return this.franchiseService.suspendFranchise(franchiseId, merchantId, body?.reason);
  }

  @Post(':id/reactivate')
  @ApiOperation({ summary: 'Reactivate franchise' })
  async reactivateFranchise(
    @Param('id', ParseIntPipe) franchiseId: number,
    @MerchantId() merchantId: number,
  ) {
    return this.franchiseService.reactivateFranchise(franchiseId, merchantId);
  }

  @Post(':id/terminate')
  @ApiOperation({ summary: 'Terminate franchise' })
  async terminateFranchise(
    @Param('id', ParseIntPipe) franchiseId: number,
    @MerchantId() merchantId: number,
    @Body() body?: { reason?: string },
  ) {
    return this.franchiseService.terminateFranchise(franchiseId, merchantId, body?.reason);
  }

  // ============================================================================
  // STATISTICS
  // ============================================================================

  @Get(':id/stats')
  @ApiOperation({ summary: 'Get franchise statistics' })
  async getFranchiseStats(
    @Param('id', ParseIntPipe) franchiseId: number,
    @MerchantId() merchantId: number,
  ) {
    const franchise = await this.franchiseService.getFranchiseById(franchiseId, merchantId);
    return this.franchiseService.getFranchiseStats(franchise.franchise_merchant_id);
  }

  // ============================================================================
  // COMMISSION & INVOICES
  // ============================================================================

  @Get(':id/commission/calculate')
  @ApiOperation({ summary: 'Calculate commission for period' })
  async calculateCommission(
    @Param('id', ParseIntPipe) franchiseId: number,
    @Query('startDate') startDate: string,
    @Query('endDate') endDate: string,
  ) {
    return this.franchiseService.calculateCommission(
      franchiseId,
      new Date(startDate),
      new Date(endDate),
    );
  }

  @Post(':id/invoices')
  @ApiOperation({ summary: 'Create commission invoice' })
  async createInvoice(
    @Param('id', ParseIntPipe) franchiseId: number,
    @MerchantId() merchantId: number,
    @Body() body: { periodStart: string; periodEnd: string },
  ) {
    return this.franchiseService.createCommissionInvoice(
      franchiseId,
      merchantId,
      new Date(body.periodStart),
      new Date(body.periodEnd),
    );
  }

  @Get(':id/invoices')
  @ApiOperation({ summary: 'Get commission invoices' })
  async getInvoices(
    @Param('id', ParseIntPipe) franchiseId: number,
    @MerchantId() merchantId: number,
    @Query('status') status?: string,
    @Query('page') page?: string,
    @Query('limit') limit?: string,
  ) {
    return this.franchiseService.getCommissionInvoices(franchiseId, merchantId, {
      status,
      page: page ? parseInt(page) : 1,
      limit: limit ? parseInt(limit) : 20,
    });
  }

  @Post('invoices/:invoiceId/mark-paid')
  @ApiOperation({ summary: 'Mark invoice as paid' })
  async markInvoicePaid(
    @Param('invoiceId', ParseIntPipe) invoiceId: number,
    @MerchantId() merchantId: number,
    @Body() body?: { paymentReference?: string },
  ) {
    return this.franchiseService.markInvoicePaid(
      invoiceId,
      merchantId,
      body?.paymentReference,
    );
  }

  // ============================================================================
  // TERRITORY
  // ============================================================================

  @Post('find-for-location')
  @ApiOperation({ summary: 'Find franchise for location' })
  async findFranchiseForLocation(
    @MerchantId() merchantId: number,
    @Body() body: { latitude: number; longitude: number },
  ) {
    const franchise = await this.franchiseService.findFranchiseForLocation(
      merchantId,
      body.latitude,
      body.longitude,
    );

    if (!franchise) {
      return { found: false, franchise: null };
    }

    return { found: true, franchise };
  }
}
