import {
  Controller,
  Get,
  Post,
  Put,
  Delete,
  Body,
  Param,
  Query,
  UseGuards,
  ParseIntPipe,
  DefaultValuePipe,
} from '@nestjs/common';
import {
  FleetService,
  FleetStatus,
  CreateFleetDto,
  UpdateFleetDto,
} from './fleet.service';
import { JwtAuthGuard } from '../../common/guards/jwt-auth.guard';
import { MerchantId } from '../../common/decorators/merchant.decorator';

// ============================================================================
// DTOs
// ============================================================================

class AddDriverDto {
  driverId: number;
  role?: 'driver' | 'supervisor' | 'manager';
}

class UpdateDriverRoleDto {
  role: 'driver' | 'supervisor' | 'manager';
}

class AddVehicleDto {
  vehicleId: number;
}

class AssignVehicleDto {
  driverId: number;
}

// ============================================================================
// CONTROLLER
// ============================================================================

@Controller('fleets')
@UseGuards(JwtAuthGuard)
export class FleetController {
  constructor(private readonly fleetService: FleetService) {}

  // ==========================================================================
  // FLEET CRUD
  // ==========================================================================

  /**
   * Lister les flottes
   */
  @Get()
  async listFleets(
    @MerchantId() merchantId: number,
    @Query('page', new DefaultValuePipe(1), ParseIntPipe) page: number,
    @Query('limit', new DefaultValuePipe(20), ParseIntPipe) limit: number,
    @Query('status') status?: FleetStatus,
    @Query('ownerId') ownerId?: string,
    @Query('search') search?: string,
  ) {
    return this.fleetService.listFleets(merchantId, {
      page,
      limit,
      status,
      ownerId: ownerId ? parseInt(ownerId) : undefined,
      search,
    });
  }

  /**
   * Créer une flotte
   */
  @Post()
  async createFleet(
    @MerchantId() merchantId: number,
    @Body() dto: CreateFleetDto,
  ) {
    const fleet = await this.fleetService.createFleet(merchantId, dto);
    return {
      success: true,
      fleet,
    };
  }

  /**
   * Obtenir une flotte
   */
  @Get(':id')
  async getFleet(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const fleet = await this.fleetService.getFleet(id, merchantId);
    if (!fleet) {
      return { error: 'Flotte non trouvée' };
    }
    return fleet;
  }

  /**
   * Mettre à jour une flotte
   */
  @Put(':id')
  async updateFleet(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
    @Body() dto: UpdateFleetDto,
  ) {
    const fleet = await this.fleetService.updateFleet(id, merchantId, dto);
    if (!fleet) {
      return { error: 'Flotte non trouvée' };
    }
    return {
      success: true,
      fleet,
    };
  }

  /**
   * Supprimer une flotte
   */
  @Delete(':id')
  async deleteFleet(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) id: number,
  ) {
    const deleted = await this.fleetService.deleteFleet(id, merchantId);
    return {
      success: deleted,
      message: deleted ? 'Flotte supprimée' : 'Flotte non trouvée',
    };
  }

  // ==========================================================================
  // FLEET DRIVERS
  // ==========================================================================

  /**
   * Lister les chauffeurs d'une flotte
   */
  @Get(':id/drivers')
  async getFleetDrivers(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) fleetId: number,
    @Query('status') status?: 'pending' | 'active' | 'inactive',
  ) {
    return this.fleetService.getFleetDrivers(fleetId, merchantId, status);
  }

  /**
   * Ajouter un chauffeur à la flotte
   */
  @Post(':id/drivers')
  async addDriver(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) fleetId: number,
    @Body() dto: AddDriverDto,
  ) {
    const driver = await this.fleetService.addDriver(
      fleetId,
      dto.driverId,
      merchantId,
      dto.role,
    );
    return {
      success: true,
      fleetDriver: driver,
      message: driver.status === 'pending'
        ? 'Chauffeur ajouté, en attente d\'approbation'
        : 'Chauffeur ajouté à la flotte',
    };
  }

  /**
   * Retirer un chauffeur de la flotte
   */
  @Delete(':id/drivers/:driverId')
  async removeDriver(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) fleetId: number,
    @Param('driverId', ParseIntPipe) driverId: number,
  ) {
    const removed = await this.fleetService.removeDriver(fleetId, driverId, merchantId);
    return {
      success: removed,
      message: removed ? 'Chauffeur retiré de la flotte' : 'Chauffeur non trouvé',
    };
  }

  /**
   * Approuver un chauffeur
   */
  @Post(':id/drivers/:driverId/approve')
  async approveDriver(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) fleetId: number,
    @Param('driverId', ParseIntPipe) driverId: number,
  ) {
    const driver = await this.fleetService.approveDriver(fleetId, driverId, merchantId);
    if (!driver) {
      return { error: 'Chauffeur non trouvé ou déjà approuvé' };
    }
    return {
      success: true,
      fleetDriver: driver,
      message: 'Chauffeur approuvé',
    };
  }

  /**
   * Mettre à jour le rôle d'un chauffeur
   */
  @Put(':id/drivers/:driverId/role')
  async updateDriverRole(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) fleetId: number,
    @Param('driverId', ParseIntPipe) driverId: number,
    @Body() dto: UpdateDriverRoleDto,
  ) {
    const driver = await this.fleetService.updateDriverRole(
      fleetId,
      driverId,
      merchantId,
      dto.role,
    );
    if (!driver) {
      return { error: 'Chauffeur non trouvé' };
    }
    return {
      success: true,
      fleetDriver: driver,
    };
  }

  // ==========================================================================
  // FLEET VEHICLES
  // ==========================================================================

  /**
   * Lister les véhicules d'une flotte
   */
  @Get(':id/vehicles')
  async getFleetVehicles(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) fleetId: number,
    @Query('status') status?: 'available' | 'in_use' | 'maintenance',
  ) {
    return this.fleetService.getFleetVehicles(fleetId, merchantId, status);
  }

  /**
   * Ajouter un véhicule à la flotte
   */
  @Post(':id/vehicles')
  async addVehicle(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) fleetId: number,
    @Body() dto: AddVehicleDto,
  ) {
    const vehicle = await this.fleetService.addVehicle(
      fleetId,
      dto.vehicleId,
      merchantId,
    );
    return {
      success: true,
      fleetVehicle: vehicle,
      message: 'Véhicule ajouté à la flotte',
    };
  }

  /**
   * Retirer un véhicule de la flotte
   */
  @Delete(':id/vehicles/:vehicleId')
  async removeVehicle(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) fleetId: number,
    @Param('vehicleId', ParseIntPipe) vehicleId: number,
  ) {
    const removed = await this.fleetService.removeVehicle(fleetId, vehicleId, merchantId);
    return {
      success: removed,
      message: removed ? 'Véhicule retiré de la flotte' : 'Véhicule non trouvé',
    };
  }

  /**
   * Assigner un véhicule à un chauffeur
   */
  @Post(':id/vehicles/:vehicleId/assign')
  async assignVehicle(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) fleetId: number,
    @Param('vehicleId', ParseIntPipe) vehicleId: number,
    @Body() dto: AssignVehicleDto,
  ) {
    const vehicle = await this.fleetService.assignVehicleToDriver(
      fleetId,
      vehicleId,
      dto.driverId,
      merchantId,
    );
    if (!vehicle) {
      return { error: 'Véhicule non disponible' };
    }
    return {
      success: true,
      fleetVehicle: vehicle,
      message: 'Véhicule assigné au chauffeur',
    };
  }

  /**
   * Libérer un véhicule
   */
  @Post(':id/vehicles/:vehicleId/release')
  async releaseVehicle(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) fleetId: number,
    @Param('vehicleId', ParseIntPipe) vehicleId: number,
  ) {
    const vehicle = await this.fleetService.releaseVehicle(fleetId, vehicleId, merchantId);
    if (!vehicle) {
      return { error: 'Véhicule non trouvé' };
    }
    return {
      success: true,
      fleetVehicle: vehicle,
      message: 'Véhicule libéré',
    };
  }

  // ==========================================================================
  // STATISTICS & REPORTS
  // ==========================================================================

  /**
   * Statistiques de la flotte
   */
  @Get(':id/stats')
  async getFleetStats(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) fleetId: number,
  ) {
    const fleet = await this.fleetService.getFleet(fleetId, merchantId);
    if (!fleet) {
      return { error: 'Flotte non trouvée' };
    }
    return fleet.stats;
  }

  /**
   * Rapport de performance
   */
  @Get(':id/performance')
  async getFleetPerformance(
    @MerchantId() merchantId: number,
    @Param('id', ParseIntPipe) fleetId: number,
    @Query('startDate') startDate?: string,
    @Query('endDate') endDate?: string,
  ) {
    const start = startDate ? new Date(startDate) : new Date(Date.now() - 30 * 24 * 60 * 60 * 1000);
    const end = endDate ? new Date(endDate) : new Date();

    return this.fleetService.getFleetPerformance(fleetId, merchantId, start, end);
  }
}
